#include "baseRobot.h"
#include <argos3/core/utility/configuration/argos_configuration.h>
#include <argos3/core/utility/math/vector2.h>
#include <argos3/core/utility/logging/argos_log.h>
#include <algorithm>

#include <loop_functions/mainLoopFunctions.h>


/* ======================= READ PARAMETERS ================== */

BaseRobot::DiffusionParams::DiffusionParams() : GoStraightAngleRange(CRadians(-1.0f), CRadians(1.0f)) {}

void BaseRobot::DiffusionParams::Init(TConfigurationNode& t_node) {
   try {
      CRange<CDegrees> cGoStraightAngleRangeDegrees(CDegrees(-10.0f), CDegrees(10.0f));
      GetNodeAttribute(t_node, "go_straight_angle_range", cGoStraightAngleRangeDegrees);
      GoStraightAngleRange.Set(ToRadians(cGoStraightAngleRangeDegrees.GetMin()),
                               ToRadians(cGoStraightAngleRangeDegrees.GetMax()));
      GetNodeAttribute(t_node, "delta", Delta);
   }
   catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error initializing controller diffusion parameters.", ex);
   }
}

void BaseRobot::WheelTurningParams::Init(TConfigurationNode& t_node) {
   try {
      CDegrees cAngle;
      GetNodeAttribute(t_node, "hard_turn_angle_threshold", cAngle);
      hardTurnOnAngleThreshold = ToRadians(cAngle);
      GetNodeAttribute(t_node, "soft_turn_angle_threshold", cAngle);
      softTurnOnAngleThreshold = ToRadians(cAngle);
      GetNodeAttribute(t_node, "no_turn_angle_threshold", cAngle);
      noTurnAngleThreshold = ToRadians(cAngle);
      GetNodeAttribute(t_node, "max_speed", maxSpeed);
   }
   catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error initializing controller wheel turning parameters.", ex);
   }
}

void BaseRobot::BaseParameters::Init(TConfigurationNode& t_node) {
   try {
	   uint linuxBased;
	   GetNodeAttribute(t_node, "linuxBased", linuxBased);
	   GetNodeAttribute(t_node, "cartCapacity", cartCapacity);
	   GetNodeAttribute(t_node, "loadTimePerVolumeUnit", loadTimePerVolumeUnit);
	   GetNodeAttribute(t_node, "unloadTimePerVolumeUnit", unloadTimePerVolumeUnit);
	   GetNodeAttribute(t_node, "neighbourhoodSearchRadius", neighbourhoodSearchRadius);
	   GetNodeAttribute(t_node, "neighbourhoodSearchTime", neighbourhoodSearchTime);
	   GetNodeAttribute(t_node, "expectedArenaDiameter", expectedArenaDiameter);
	   GetNodeAttribute(t_node, "useOpportunism", useOpportunism);
	   GetNodeAttribute(t_node, "opportunismThreshold", opportunismThreshold);
	   GetNodeAttribute(t_node, "useRelU", useRelU);


	   //-- convert times in seconds to time in ticks
	   loadTimePerVolumeUnit *= 10;
	   unloadTimePerVolumeUnit *= 10;
	   neighbourhoodSearchTime *= 10;

	   //-- setup sensors - builds of ARGoS on different systems have different setups
	   isLinuxBased = (linuxBased == 1);
	   if (isLinuxBased) {
		  //-- on iridis
		  groundSensorFLId=0;
		  groundSensorBLId=1;
		  groundSensorFRId=3;
		  groundSensorBRId=2;
	   } else {
		  //-- on a mac
		  groundSensorFLId=0;
		  groundSensorBLId=3;
		  groundSensorFRId=1;
		  groundSensorBRId=2;
	   }
   }
   catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error initializing controller wheel turning parameters.", ex);
   }
}



/*================================== INITIALISATION =================== */



BaseRobot::BaseRobot() :
   wheels(NULL),
   leds(NULL),
   collisionAvoidingAngle(0),
   rangeAndBearingActuator(NULL),
   rangeAndBearingSensor(NULL),
   proximitySensor(NULL),
   lightSensor(NULL),
   groundSensor(NULL),
   randNumGen(NULL) {

}


void BaseRobot::Init(TConfigurationNode& t_node) {
   try {
	   //-- init sensors and actuators
	   wheels    = GetActuator<CCI_DifferentialSteeringActuator>("differential_steering");
	   leds      = GetActuator<CCI_LEDsActuator                >("leds"                 );
	   rangeAndBearingActuator      = GetActuator<CCI_RangeAndBearingActuator     >("range_and_bearing"    );
	   rangeAndBearingSensor      = GetSensor  <CCI_RangeAndBearingSensor       >("range_and_bearing"    );
	   proximitySensor = GetSensor  <CCI_FootBotProximitySensor      >("footbot_proximity"    );
	   lightSensor     = GetSensor  <CCI_FootBotLightSensor          >("footbot_light"        );
	   groundSensor    = GetSensor  <CCI_FootBotMotorGroundSensor    >("footbot_motor_ground" );
	   steeringSensor    = GetSensor  <CCI_DifferentialSteeringSensor    >("differential_steering" );

	   //-- parse XML parameters
	   diffustionParams.Init(GetNode(t_node, "diffusion"));
	   wheelParams.Init(GetNode(t_node, "wheel_turning"));
	   baseParameters.Init(GetNode(t_node, "baseParameters"));

   }
   catch(CARGoSException& ex) {
	   THROW_ARGOSEXCEPTION_NESTED("Error initializing the foot-bot foraging controller for robot \"" << GetId() << "\"", ex);
   }

   //-- other stuff
   randNumGen = CRandom::CreateRNG("argos");
   gslRandNumGen = gsl_rng_alloc(gsl_rng_default);
   Reset();
}



void BaseRobot::Reset() {
	randWalkTurnCounter = 0;
	randWalkTurnLeft = false;
	resourceInCartVolume = 0;
	resourceInCartValue = 0;
	timeTillNextScoutAllowed = 0;
	didReportRewardStartReceivingEvent = false;
	didReportRewardStopReceivingEvent = false;
	supressNormalStateBehaviour = false;
	targetTaskInfo = TaskInfo();
	SetState(STATE_RESTING);
}

void BaseRobot::SetState(STATE state_, const bool& isNewState_) {
	if (isNewState_) {
		rangeAndBearingActuator->ClearData();
		didReportTaskReachedEvent = false;
		didReportReportTaskStartedEvent = false;
		didReportScoutEvent = false;
		timeTillNextScoutAllowed = 0;
	}
	state = state_;
	randWalkTurnCounter = 0;

	switch (state) {
		case STATE_RESTING:
			targetTaskInfo = TaskInfo();
			break;
		case STATE_SCOUTING:
			if (isNewState_) {
				targetTaskInfo = TaskInfo();
				timeScouting = 0;
				neighbourhoodSearchTimeLeft = 0;

			}

			break;
		case STATE_GOING_TO_TASK:
			if (isNewState_) {
				timeGoingToTask = 0;
				neighbourhoodSearchTimeLeft = 0;

				didReportRewardStartReceivingEvent = false;
				didReportRewardStopReceivingEvent = false;
			}
			break;

		case STATE_PROCESSING_TASK:
			if (isNewState_) {
				timeTillNextRelUChecked = baseParameters.loadTimePerVolumeUnit;
			}
			break;

		default:
			break;
	}
}


/*======================================= UPDATE LOOP ======================== */

/**
 * Pre-control step, called from the environment, before robots are told to process anything
 */

void BaseRobot::PreControlStep() {
	//--
	ReadCurrentLocation();
	timeCounter++;

	//-- if opportunistic, keep checking if there is a better task around
	supressNormalStateBehaviour = false;
	if (baseParameters.useOpportunism) {
		if (!targetTaskInfo.isNull) {
			if (state == STATE_GOING_TO_TASK || state == STATE_RETURN_TO_BASE || state == STATE_RETURN_TO_BASE_UNSUCCESSFUL || state == STATE_RETURN_TO_BASE_UNSUCCESSFUL_BLIND  ) {
				////////LOGERR << location << "   " << timeTillNextScoutAllowed << std::endl;
				if (location == LOCATION_DEPOSIT && timeTillNextScoutAllowed <= 0) {

					if (nearbyTaskInfo.taskId != targetTaskInfo.taskId) {
						supressNormalStateBehaviour = true;
						if (NavigateTowardsDeposit()) {


							nearbyTaskInfo.taskUtility = EstimateTaskUtility(nearbyTaskInfo.taskType, nearbyTaskInfo.taskValue, nearbyTaskInfo.taskVolume, relBaseLocation.Length(), false);
							//LOGERR << "checking new U " << nearbyTaskInfo.taskUtility << " VS " << targetTaskInfo.taskUtility << std::endl;
							timeTillNextScoutAllowed = DELAY_BETWEEN_SCOUTINGS;

							if (nearbyTaskInfo.taskUtility - targetTaskInfo.taskUtility > baseParameters.opportunismThreshold) {
								//LOGERR << "!!!!! OPPORTUNISM" << std::endl;

								SetNewTarget(nearbyTaskInfo.taskType, nearbyTaskInfo.taskId, nearbyTaskInfo.taskValue, CVector2(), id, nearbyTaskInfo.taskUtility, 1, nearbyTaskInfo.numOfRobotsRequired);
								ProcessTaskReached();

							}
						}
					}

				}
			}
		}
	}
}

/**
 * Main update loop function
 */
void BaseRobot::ControlStep() {
	//-- reset event reports
	//infoEventReports.clear();

	if (!supressNormalStateBehaviour) { // behaviour could be supressed e.g. in the Pre-Step loop
		//-- basic state resolution. Additional states might be resolved by subclasses.
		switch(state) {
			case STATE_RESTING: {
				Rest();
				break;
			}
			case STATE_SCOUTING: {
				Scout();
				break;
			}
			case STATE_PROCESSING_TASK: {
				//ProcessTask(); // called from the main loop function so that parameters can be passed in
				break;
			}
			case STATE_GOING_TO_TASK: {
				GoToTask();
				break;
			}
			case STATE_RETURN_TO_BASE:
			case STATE_RETURN_TO_BASE_UNSUCCESSFUL:
			case STATE_RETURN_TO_BASE_UNSUCCESSFUL_BLIND: {
				ReturnToBase();
				break;
			}

			default:
				break;
		}

		//-- make sure that stays within bounds
		if (relBaseLocation.Length() > baseParameters.expectedArenaDiameter) {
			NavigateToLight();
		}
	}

	if (timeTillNextScoutAllowed > 0) {
		timeTillNextScoutAllowed --;
	}

	if (timeTillNextRelUChecked > 0) {
		timeTillNextRelUChecked --;
	}



	//-- update odometry to target
	if (!targetTaskInfo.isNull) {
		UpdateOdometryToLocation(targetTaskInfo.relLocation);
	}
	//-- update odomerty to base
	UpdateOdometryToLocation(relBaseLocation);

	//-- set front LED color based on state
	switch (state) {
		case STATE_SCOUTING:
			leds->SetAllColors(CColor::MAGENTA);
			break;
		case STATE_PROCESSING_TASK:
			leds->SetAllColors(CColor::CYAN);
			break;
		case STATE_RETURN_TO_BASE:
			leds->SetAllColors(CColor::BLUE);
			break;
		case STATE_GOING_TO_TASK:
			leds->SetAllColors(CColor::GREEN);
			break;
		default:
			leds->SetAllColors(CColor::BLACK);
			break;
	}

	if (supressNormalStateBehaviour) {
		leds->SetAllColors(CColor::RED);
	}

	/*if (this->GetNumId() == 21){
		leds->SetAllColors(CColor::RED);
	} else if (this->GetNumId() == 8){
		leds->SetAllColors(CColor::CYAN);
	} else {
		leds->SetAllColors(CColor::BLACK);
	}*/

}


/**
 * Rest in the resting bay
 */
void BaseRobot::Rest() {
	//-- make sure it's in resting bay
	if (location != LOCATION_RESTING_BAY) {
		NavigateToLight();
	} else {
		SetWheelSpeeds(0,0);
	}

}

//============================================ SCOUTING =======================
/**
 * Scout for Ns nesources
 */
void BaseRobot::Scout() {

	if (location <= LOCATION_UNLOADING_BAY) {
		//-- do levy walk outside of the base
		NavigateFromLight();
	} else {
		DoLevyWalk();
		//-- test if at a deposit
		if (location == LOCATION_DEPOSIT && timeTillNextScoutAllowed <= 0) {
			if (NavigateTowardsDeposit()) {
				if (!didReportScoutEvent) {
					OnTaskScouted();
					didReportScoutEvent = true;
				}
				ProcessTaskReached();
			}
		}
	}

	timeScouting++;

}

/**
 * Triggered once when a robot scouts and approaches a new task
 */
void BaseRobot::OnTaskScouted() {
	didReportScoutEvent = true;
	didReportRewardStartReceivingEvent = false;
	didReportRewardStopReceivingEvent = false;
	NewInfoEventReport(InfoEventReport::TYPE_TASK_SCOUTED,-1,-1,-1,-1,-1);
}

//============================================ GOING TO KNOWN TASK =======================
/**
 * Navigate towards a newly acquired / remembered target task
 */
void BaseRobot::GoToTask() {
	if (location == LOCATION_DEPOSIT) {
		//-- test if it is on the same location based on taskid or if searching, in which case take any deposit found
		if (targetTaskInfo.isNull) {
			//-- scouted a new target
			if (NavigateTowardsDeposit()) {
				if (!didReportScoutEvent) {
					OnTaskScouted();
				}
				ProcessTaskReached();
			}
		} else if (nearbyTaskInfo.taskId == targetTaskInfo.taskId) {
			//-- reached the target as expected
			/*if (this->GetNumId() == 19) {
				//////LOGERR << " going to task " << taskInfo.taskId << std::endl;
			}*/
			if (NavigateTowardsDeposit()) {
				if (!didReportTaskReachedEvent) {
					OnTaskReached();
					didReportTaskReachedEvent = true;
				}
				ProcessTaskReached();
			}
		} else {
			/*if (this->GetNumId() == 19) {
				//////LOGERR << " found different task " << nearbyTaskId << std::endl;
			}*/
			//-- reached a task but it's not the one in memory
			if (Abs(targetTaskInfo.relLocation.Length()) <= GetLoadingProximity()*2 || neighbourhoodSearchTimeLeft > 0) {
				//-- at a task, on believed location but the task doesn't have the same id. Must have used neighbourhood search or must be a dynamic environment.
				//-- abandon the old task
				NewInfoEventReport(InfoEventReport::TYPE_TASK_MISSING, targetTaskInfo.taskId, -1, -1, -1, -1);
				AbandonTask(false);
				//-- register scouting of new
				if (!didReportScoutEvent) {
					OnTaskScouted();
				}
				//-- set the new as the new target
				SetNewTarget(targetTaskInfo.taskType, nearbyTaskInfo.taskId,nearbyTaskInfo.taskValue,CVector2(0,0),id,nearbyTaskInfo.taskUtility,1,0);
				/*if (this->GetNumId() == 19) {
					//////LOGERR << " navigating towards new task " << std::endl;
				}*/
				//if (NavigateTowardsDeposit()) {

				ProcessTaskReached();
				//}
			} else {
				//-- not searching and not near the believed location, navigate towards target
				NavigateTowardsVector(-wheelParams.maxSpeed*targetTaskInfo.relLocation);
				/*if (this->GetNumId() == 19) {
					//////LOGERR << " task still far " << std::endl;
				}*/
			}
		}


	} else {

		//-- test if searching
		if (neighbourhoodSearchTimeLeft > 0) {
			//-- search in neighbourhood
			DoNeighbourhoodSearch();
		} else {
			if (Abs(targetTaskInfo.relLocation.Length()) <= 0.25) {
				//-- should have reached the deposit already based on believed location, start neighbourhood search
				neighbourhoodSearchTimeLeft = baseParameters.neighbourhoodSearchTime;

			} else {
				//-- not at any deposit, navigate towards target deposit
				NavigateTowardsVector(-wheelParams.maxSpeed*targetTaskInfo.relLocation);
			}
		}
	}

	timeGoingToTask++;
	if (timeGoingToTask > 18000) {
		//-- got lost or got stuck
		OnUnableToReachTask();
	}
}

/**
 * Walk randomly around a believed task location for a certain time
 */
void BaseRobot::DoNeighbourhoodSearch() {
	//////////LOGERR << this->GetId() << " searching.. " << std::endl;
	DoRandomWalk(wheelParams.maxSpeed);
	if (Abs(targetTaskInfo.relLocation.Length()) >= baseParameters.neighbourhoodSearchRadius) {
		NavigateTowardsVector(-wheelParams.maxSpeed*targetTaskInfo.relLocation);
	}
	neighbourhoodSearchTimeLeft--;
	if (neighbourhoodSearchTimeLeft <= 0) {
		OnTaskReachedButDoesntExist();
	}
}

/**
 * Triggered when robot reached current task location and task can be executed
 */
void BaseRobot::OnTaskReached() {
	NewInfoEventReport(InfoEventReport::TYPE_TASK_REACHED,-1,-1,-1,-1,-1); // any other info is unknown, will be obtained by the environment
}

/**
 *  Called when robot reached current task location and task can be executed
 */
void BaseRobot::ProcessTaskReached() {

}

/**
 * Abandon a task
 */
void BaseRobot::AbandonTask(bool reportEvent_) {
	if (reportEvent_) {
		NewInfoEventReport(InfoEventReport::TYPE_TASK_ABANDONED, targetTaskInfo.taskId, -1, -1, -1,-1);
	}

	if (targetTaskInfo.taskType == TASK_PERFORM) {
		if (!this->didReportRewardStopReceivingEvent && this->didReportRewardStartReceivingEvent) {
			NewInfoEventReport(InfoEventReport::TYPE_REWARD_FINISHED, resourceInCartTaskId, -1, -1, -1,-1);
			this->didReportRewardStopReceivingEvent = true;
			////////LOGERR << this->GetId() << "========FINISHED REWARD " << taskInfo.taskId << std::endl;
		}
	}

	/*if (this->GetNumId() == 13) {
		//////LOGERR << this->GetId() << "!! ABANDONED t" << taskInfo.taskId << "  val " << taskInfo.taskValue << std::endl;
	}*/
	timeTillNextScoutAllowed = DELAY_BETWEEN_SCOUTINGS;
	targetTaskInfo.Nullify();
}

/**
 *  Triggered when robot reached current task location but nothing is there
 */
void BaseRobot::OnTaskReachedButDoesntExist() {
	NewInfoEventReport(InfoEventReport::TYPE_TASK_MISSING, targetTaskInfo.taskId, -1, -1, -1, -1); // any other info is unknown, will be obtained by the environment
	//-- abandon task but don't report it, previous line reports
	////////LOGERR << this->GetId() << "!! TASK MISSING" << taskInfo.taskId << "  " << taskInfo.taskValue << std::endl;
	AbandonTask(false);

	SetState(STATE_SCOUTING);
}

/**
 *  Triggered when robot was unable to reach task - e.g. when going to task for too long
 */
void BaseRobot::OnUnableToReachTask() {
	SetState(STATE_SCOUTING);
}


//============================================ TASK PROCESSING =======================
/**
 * Called when robot should be performing a task
 */
Real BaseRobot::ProcessTask(TASK_TYPE taskType_, int taskId_,  Real processedAmount_, Real taskValue_, Real taskTotalReward_, uint numOfRobotsRequired_) {

	EstimateTaskUtility(taskType_, taskValue_, taskTotalReward_/taskValue_, relBaseLocation.Length());
	//////LOGERR << this->GetId() << " PROCESSING TASK ID " << taskId_ << " U " << targetTaskInfo.taskUtility << "IS null? " << (targetTaskInfo.isNull) << std::endl;

	//-- report the event
	if (!didReportReportTaskStartedEvent && processedAmount_ > 0) {
		////////LOGERR << this->GetId() << " STARTED NEW " << taskId_ << std::endl;
		SetNewTarget(taskType_, taskId_, taskValue_, CVector2(0,0), id, targetTaskInfo.taskUtility, 1, numOfRobotsRequired_);
		NewInfoEventReport(InfoEventReport::TYPE_TASK_STARTED,taskId_,taskValue_,taskTotalReward_/taskValue_, targetTaskInfo.taskUtility, targetTaskInfo.infoOriginatorId);
		didReportReportTaskStartedEvent = true;
	}

	//---
	bool processingDone = false;
	bool shouldAbandon = false;
	Real taskRewardLeft = taskTotalReward_ - processedAmount_;
	if (taskRewardLeft < MainLoopFunctions::ZERO_WORKSITE_VOLUME) {
		shouldAbandon = true;
		processingDone = true;
	}
	/*if (this->GetNumId() == 12) {
		//////LOGERR << " AMOUNT LEFT " << taskRewardLeft << std::endl;
	}*/

	//-- loading during foraging
	if (processedAmount_ == 0 || (targetTaskInfo.isNull == false && targetTaskInfo.taskId != taskId_) ) {
		//-- nothing to load anymore or somehow task was switched because another robot interfered or task was too close to another
		processingDone = true;
		shouldAbandon = true;
		/*if (this->GetNumId() == 12) {
			//////LOGERR << "should abandon" << std::endl;
		}*/

	} else {
		SetWheelSpeeds(0.0f,0.0f);

		if (taskType_ == TASK_COLLECT) {
			//-- add resource to cart until cart is full
			resourceInCartVolume += processedAmount_;
			resourceInCartValue = taskValue_;

			if (baseParameters.cartCapacity - resourceInCartVolume < MainLoopFunctions::ZERO_WORKSITE_VOLUME) {
				resourceInCartVolume = baseParameters.cartCapacity;
				processingDone = true;
			} else {
				resourceInCartTaskId = taskId_;
			}
		} else {
			//-- task is performed right where the robot is, keep staying at the task until it's complete (until processedAmount_ == 0)
			resourceInCartTaskId = taskId_;

			//-- if using relU, check every second whether it should abandon, the probability of which is the relative task utility
			if (baseParameters.useRelU) {
				if (timeTillNextRelUChecked <= 0) {
					timeTillNextRelUChecked = baseParameters.loadTimePerVolumeUnit;
					Real noThrown = randNumGen->Uniform(CRange<Real>(0,1));
					//LOGERR << this->GetId() << " no thrown " << noThrown << " rel U " << targetTaskInfo.relTaskUtility << std::endl;
					if (noThrown >= targetTaskInfo.relTaskUtility) {
						processingDone = true;
						shouldAbandon = true;
						//LOGERR << this->GetId() << " !!!!! ABANDON BY relU" << std::endl;
					}
				}
			}
		}
	}

	//-- remember the task
	if (taskId_ >= 0 && shouldAbandon == false) {
		SetNewTarget(taskType_, taskId_, taskValue_, CVector2(0,0), id, targetTaskInfo.taskUtility, targetTaskInfo.relTaskUtility, numOfRobotsRequired_);
		/*if (this->GetNumId() == 17) {
			//////LOGERR << this->GetId() << " REMEMBERED " << taskInfo.taskId << "  " << taskInfo.taskValue << " " << taskInfo.taskUtility << std::endl;
		}*/
	}

	//-- finish processing if done
	if (processingDone) {
		if (shouldAbandon) {
			//-- abandon if nothing left
			AbandonTask();
		} else {
			NewInfoEventReport(InfoEventReport::TYPE_TASK_PAUSED, taskId_, taskValue_, taskRewardLeft/taskValue_, targetTaskInfo.taskUtility,-1);
		}

		if (taskType_ == TASK_COLLECT || (taskType_ == TASK_NONE && resourceInCartVolume > 0)) {
			//-- collected material needs to be returned to the base
			SetState(STATE_RETURN_TO_BASE);
		} else {
			//-- collected nothing, only performed the task, scout for new task immediatelly
			SetState(STATE_SCOUTING);
		}

		OnTaskProcessingDone(taskType_, taskId_, processedAmount_, taskValue_, taskRewardLeft, numOfRobotsRequired_);
	}

	return processedAmount_;
}

/**
 * Triggered once when the robot has finished or paused (duing collection, to return to the base) a task
 */
void BaseRobot::OnTaskProcessingDone(TASK_TYPE taskType_, int taskId_,  Real processedAmount_, Real taskValue_, Real taskRewardLeft_, uint numOfRobotsRequired_) {
	//-- wait a bit with scouting so that the same task doesn't get re-scouted
	timeTillNextScoutAllowed = DELAY_BETWEEN_SCOUTINGS;

	//-- update info about task utility
	EstimateTaskUtility(taskType_, taskValue_, taskRewardLeft_/taskValue_, relBaseLocation.Length());
}

void BaseRobot::OnTaskRewardReceived(TASK_TYPE taskType_, int taskId_, double reward_, double taskValue_, double taskVolume_) {
	//////////LOGERR << this->GetId() << " TASK ID " << taskId_ << " REWARD " << reward_ << std::endl;
	if (!this->didReportRewardStartReceivingEvent && reward_ > 0) {
		NewInfoEventReport(InfoEventReport::TYPE_REWARD_STARTED, resourceInCartTaskId, taskValue_, taskVolume_, EstimateTaskUtility(taskType_,taskValue_,taskVolume_,relBaseLocation.Length(), false),-1);
		this->didReportRewardStartReceivingEvent = true;
		//////////LOGERR << this->GetId() << "========= STARTED REWARD " << taskInfo.taskId << std::endl;
	}
}




/**
 * Return to nest with resource in cart
 */
void BaseRobot::ReturnToBase() {
	//-- test if should unload or keep going
	if (location == LOCATION_UNLOADING_BAY) {
		//- in unloading bay, unload and proceed
		SetState(STATE_UNLOADING_RESOURCE);
	} else {
		//-- still going to the base
		NavigateToLight();

		//-- test if found a deposit - only when foraging was unsuccessful
		if (location == LOCATION_DEPOSIT && state == STATE_RETURN_TO_BASE_UNSUCCESSFUL  && timeTillNextScoutAllowed <= 0) {
			////LOGERR << this->GetId() << "Found depo by returning home" << std::endl;
			if (NavigateTowardsDeposit()) {
				if (!didReportScoutEvent) {
					OnTaskScouted();
					////LOGERR << this->GetId() << "! SCOUTED EVENT" << std::endl;
				}
				ProcessTaskReached();
			}
		}
	}
}



/**
 * Unload a certain amount of resource to unloading bay
 */
float BaseRobot::PerformUnloadResource() {
	//-- keep within the unloading bay boundaries
	if (location < LOCATION_UNLOADING_BAY) {
		NavigateFromLight();
	} else if (location > LOCATION_UNLOADING_BAY) {
		NavigateToLight();
	} else {
		DoRandomWalk(wheelParams.maxSpeed);
	}

	//-- unload until cart is empty
	Real toUnload = 0;
	if (resourceInCartVolume >= MainLoopFunctions::ZERO_WORKSITE_VOLUME) {
		if (location == LOCATION_UNLOADING_BAY) {
			toUnload = 1.0/baseParameters.unloadTimePerVolumeUnit;
			resourceInCartVolume -= toUnload;
		}
	} else {
		OnUnloadResourceDone();
	}

	return toUnload*resourceInCartValue;
}

/**
 * Triggered once when all resource has been unloaded
 */
void BaseRobot::OnUnloadResourceDone() {
	if (!this->didReportRewardStopReceivingEvent && this->didReportRewardStartReceivingEvent) {
		NewInfoEventReport(InfoEventReport::TYPE_REWARD_FINISHED, resourceInCartTaskId, -1, -1, -1,-1);
		this->didReportRewardStopReceivingEvent = true;
		//////////LOGERR << this->GetId() << "========FINISHED REWARD " << taskInfo.taskId << std::endl;
	}

	//-- if using relU, check every second whether it should abandon, the probability of which is the relative task utility
	if (baseParameters.useRelU) {

		Real noThrown = randNumGen->Uniform(CRange<Real>(0,1));
		//LOGERR << this->GetId() << " num thrown " << noThrown << " rel U " << targetTaskInfo.relTaskUtility << std::endl;
		if (noThrown >= targetTaskInfo.relTaskUtility) {
			AbandonTask();
			//LOGERR << this->GetId() << " !!!!! ABANDON BY relU" << std::endl;

		}

	}
}


/* ================================ PERCEPTIONS =============================== */

/**
 * Remember info about a nearby task. Called from the main loop functions.
 */
void BaseRobot::SetNearbyTaskInfo(TASK_TYPE taskType_, int taskId_, Real taskValue_, Real taskVolume_, uint numOfRobotsRequired_) {
	nearbyTaskInfo = TaskInfo(taskType_, taskId_, taskValue_, taskVolume_, numOfRobotsRequired_);
	////LOGERR << this->GetId() << " nearby task id " << taskId_
}

/**
 * Set a certain task info as new target of work.
 */
void BaseRobot::SetNewTarget(TASK_TYPE taskType_, int taskId_, Real taskValue_, CVector2 relLocation_, uint infoOriginatorId_, Real taskUtility_, Real relTaskUtility_, uint numOfRobotsRequired_) {

	//////LOGERR << this->GetId() << " set target to " << taskId_ << " U=" << taskUtility_ << std::endl;

	//-- abandonment previous task
	if (!targetTaskInfo.isNull && targetTaskInfo.taskId != taskId_) {
		AbandonTask();
		didReportTaskReachedEvent = false;
		didReportScoutEvent = false;
		didReportReportTaskStartedEvent = false;
		didReportRewardStartReceivingEvent = false;

		OnTaskScouted();
	}

	//-- set to new task
	targetTaskInfo = TaskInfo(taskType_, taskId_, taskValue_, CVector2(0,0), id, taskUtility_, relTaskUtility_, numOfRobotsRequired_);
}

/**
 * Estimate task utility depending on its volume, value, type and, if robot needs to go back to the base, distance from the base.
 * Also, if should remember it, estimate the relative utility
 */
Real BaseRobot::EstimateTaskUtility(TASK_TYPE taskType_, Real taskValue_, Real taskVolume_, Real distanceFromBase_) {
	return EstimateTaskUtility(taskType_, taskValue_, taskVolume_, distanceFromBase_, true);
}

Real BaseRobot::EstimateTaskUtility(TASK_TYPE taskType_, Real taskValue_, Real taskVolume_, Real distanceFromBase_, bool remeberEstimation_) {

	Real taskUtility = DoTaskUtilityCalculation(taskType_, taskValue_, taskVolume_, distanceFromBase_);

	if (remeberEstimation_) {
		Real relUtility;
		if (taskUtility <= 0) {
			relUtility = 0.0;
		} else	if (targetTaskInfo.isNull || targetTaskInfo.taskUtility == 0) { //completely new
			relUtility = 1.0;
		} else {
			if (baseParameters.useRelU) {
				relUtility = Min(1.0,(taskUtility/targetTaskInfo.taskUtility)*targetTaskInfo.relTaskUtility);
			} else {
				relUtility = 1.0;
			}
		}

		//////LOGERR << " VAL = " << taskValue_ << "VOL = " << taskVolume_ << " D from base" << distanceFromBase_  << " U:" << taskUtility << " relUold:" << targetTaskInfo.relTaskUtility << " relUnow:" << relUtility << std::endl;
		//////LOGERR << " U:" << taskUtility << " relUold:" << targetTaskInfo.relTaskUtility << " relUnow:" << relUtility << std::endl;

		targetTaskInfo.taskUtility = taskUtility;
		targetTaskInfo.relTaskUtility = relUtility;

	}

	return taskUtility;
}

/**
 * Called by the EstimateTaskUtility function
 */
Real BaseRobot::DoTaskUtilityCalculation(TASK_TYPE taskType_, Real taskValue_, Real taskVolume_, Real distanceFromBase_) {
	if (taskType_ == TASK_COLLECT) {
		return(taskValue_*taskVolume_ / distanceFromBase_);
	} else {
		return taskValue_*taskVolume_;
	}
}



/* ================================ MOVEMENT =============================== */


/**
 * Navigate down the gradient towards deposit (where deposit is black).
 * Returns true of at the deposit, otherwise false.
 */
bool BaseRobot::NavigateTowardsDeposit() {
	//-- find out if at the deposit already
	const CCI_FootBotProximitySensor::TReadings& proximitySensorData = proximitySensor->GetReadings();
	const CCI_FootBotMotorGroundSensor::TReadings& groudSensorData = groundSensor->GetReadings();
	if (groudSensorData[baseParameters.groundSensorFRId].Value < 0.55 || groudSensorData[baseParameters.groundSensorFLId].Value < 0.55) {
		for(size_t i = 0; i < proximitySensorData.size(); ++i) {
			if (proximitySensorData[i].Value > 0 && proximitySensorData[i].Value <= GetLoadingProximity()) {
				return true;
			}
		}
	}

	//-- go down the gradient towards deposit
	if (groudSensorData[baseParameters.groundSensorFRId].Value > groudSensorData[baseParameters.groundSensorFLId].Value) {
		//-- left side of robot senses lower value, turn towards it
		SetWheelSpeeds(wheelParams.maxSpeed*0.5, wheelParams.maxSpeed);
	} else {
		//-- right side of robot senses higher value, turn towards it
		SetWheelSpeeds(wheelParams.maxSpeed, wheelParams.maxSpeed*0.5);

	}
	return false;
}
/**
 * Navigate towards the light source
 */
void BaseRobot::NavigateToLight() {
	NavigateTowardsVector(wheelParams.maxSpeed * 0.75f * CalculateVectorToLight());
	//-- test if back is facing the light, which means that the robot has to turn by max angle
	const CCI_FootBotLightSensor::TReadings& tLightReads = lightSensor->GetReadings();
	if (tLightReads[11].Value > 0 && tLightReads[12].Value > 0) {
		SetWheelSpeeds(wheelParams.maxSpeed, 0);
	}
}

/**
 * Navigate away from light
 */
void BaseRobot::NavigateFromLight() {
	NavigateTowardsVector(-wheelParams.maxSpeed * 0.75f * CalculateVectorToLight());
	//-- test if front is facing the light , which means that the robot has to turn by max angle
	const CCI_FootBotLightSensor::TReadings& tLightReads = lightSensor->GetReadings();
	if (tLightReads[0].Value > 0 && tLightReads[23].Value > 0) {
		SetWheelSpeeds(wheelParams.maxSpeed, 0);
	}
}

/**
 * Perform random walk of certain speed
 */
void BaseRobot::DoRandomWalk(const float baseSpeed_, const float maxDistanceFromTargetDeposit_) {

	CRadians angle = CRadians(0);
	//-- turn only each N ticks
	if (randWalkTurnCounter == 0) {
		//-- turn with small probability
		if (randNumGen->Uniform(CRange<Real>(0,1)) < 0.1) {
			randWalkTurnCounter = 10;
			if (randNumGen->Uniform(CRange<Real>(0,1)) < 0.5) {
				randWalkTurnLeft = true;
			} else {
				randWalkTurnLeft = false;
			}
		}
	}
	if (randWalkTurnCounter > 0) {
		//-- turn
		if (randWalkTurnLeft) {
			angle = CRadians(randNumGen->Uniform(CRange<Real>(0.5,1)));
		} else {
			angle = CRadians(-randNumGen->Uniform(CRange<Real>(0.5,1)));

		}
		randWalkTurnCounter--;
	}
	bool bCollision;
	SetWheelSpeedsFromVector(wheelParams.maxSpeed * DiffusionVector(bCollision) + wheelParams.maxSpeed* CVector2(1.0f,angle));

	//-- return back to deposit origin if too far away and should wark around a deposit
	if (maxDistanceFromTargetDeposit_ > 0 && targetTaskInfo.relLocation.Length() > maxDistanceFromTargetDeposit_) {
		if (location == LOCATION_DEPOSIT) {
			NavigateTowardsDeposit();
		} else {
			NavigateTowardsVector(-wheelParams.maxSpeed*targetTaskInfo.relLocation);
		}
	}
}

/**
 * Do Levy-random walk
 */
void BaseRobot::DoLevyWalk() {
	randWalkTurnCounter--;
	if (randWalkTurnCounter <= 0) {
		randWalkTurnCounter = randNumGen->Uniform(CRange<int>(0,5));
		CRadians angle = CRadians(gsl_ran_levy(gslRandNumGen, 1.9, 2.0));
		if (angle.GetAbsoluteValue() < 0.25) {
			angle = CRadians(0);
		}
		/*if (this->GetNumId() == 0 || this->GetNumId() == 1) {
			LOGERR << "Robot " << this->GetNumId() << " levy angle " << angle << std::endl;
		}*/
		NavigateTowardsVector(wheelParams.maxSpeed * 0.75f * CVector2(1.0f,angle));
	}
}

/**
 * Find out whether in one of the sections of base or outside
 */
void BaseRobot::ReadCurrentLocation() {

	location = LOCATION_OUT;

	//-- Read values from the ground sensor. Test value from sensors at the back (2,3), whre 0 = black, 1=white  */
	const CCI_FootBotMotorGroundSensor::TReadings& groudSensorData = groundSensor->GetReadings();
	////////////LOGERR << "LOC? L " << groudSensorData[bodyParams.groundSensorFLId].Value << "   " << groudSensorData[bodyParams.groundSensorBLId].Value << "R " << groudSensorData[bodyParams.groundSensorFRId].Value << "  " << groudSensorData[bodyParams.groundSensorBRId].Value << std::endl;
	if ((groudSensorData[baseParameters.groundSensorFRId].Value > 0.37f && groudSensorData[baseParameters.groundSensorFRId].Value < 0.9f) ||
		(groudSensorData[baseParameters.groundSensorFLId].Value > 0.37f && groudSensorData[baseParameters.groundSensorFLId].Value < 0.9f)) {
		location = LOCATION_DEPOSIT;
	} else if (groudSensorData[baseParameters.groundSensorBRId].Value > 0.27f && groudSensorData[baseParameters.groundSensorBRId].Value < 0.33f &&
			groudSensorData[baseParameters.groundSensorBLId].Value > 0.27f && groudSensorData[baseParameters.groundSensorBLId].Value < 0.33f) {
		location = LOCATION_DANCE_FLOOR;
	} else if (groudSensorData[baseParameters.groundSensorBRId].Value > 0.17f && groudSensorData[baseParameters.groundSensorBRId].Value < 0.23f &&
			groudSensorData[baseParameters.groundSensorBLId].Value > 0.17f && groudSensorData[baseParameters.groundSensorBLId].Value < 0.23f) {
		location = LOCATION_RESTING_BAY;
	} else if (groudSensorData[baseParameters.groundSensorBRId].Value > 0.07f && groudSensorData[baseParameters.groundSensorBRId].Value < 0.13f &&
			groudSensorData[baseParameters.groundSensorBLId].Value > 0.07f && groudSensorData[baseParameters.groundSensorBLId].Value < 0.13f) {
		location = LOCATION_UNLOADING_BAY;
		relBaseLocation = CVector2(0,0);
	}
}

/**
 * Update relative location of target in memory
 */
void BaseRobot::UpdateOdometryToLocation(CVector2& location_) {
	//-- update relative location of the deposit
	Real distR, distL, interWheel;

	const CCI_DifferentialSteeringSensor::SReading& steeringData = steeringSensor->GetReading();
	interWheel = steeringData.WheelAxisLength * 0.01;
	distR = steeringData.CoveredDistanceRightWheel * 0.01;
	distL = steeringData.CoveredDistanceLeftWheel * 0.01;

	Real stepDist = ((distR + distL)/2);
	CRadians stepAngle = CRadians((distR-distL)/(interWheel));

	location_ += CVector2(stepDist, stepAngle);
	location_.Rotate(-stepAngle);
}

/* ================================ OTHER FUNCTIONS =================== */

void BaseRobot::NewInfoEventReport(InfoEventReport::TYPE type_, int taskId_, Real taskValue_, Real taskVolume_, Real taskUtility_, Real recruiterId_) {
	InfoEventReport report;
	report.type = type_;
	report.taskId = taskId_;
	report.taskValue = taskValue_;
	report.taskVolume = taskVolume_;
	report.taskUtility = taskUtility_;
	report.recruiterId = recruiterId_;

	infoEventReports.push_back(report);
}


int* BaseRobot::GenerateRandomPacketOrder(const CCI_RangeAndBearingSensor::TReadings& packets_) {
	int* indeces = new int[packets_.size()];
	for(size_t i = 0; i < packets_.size(); ++i) {
		int randInt;
		bool unique = true;
		int counter = 0;
		do {
			randInt = randNumGen->Uniform(CRange<int>(0,packets_.size()));
			for (int j=0; j<i; j++) {
				if (indeces[j] == randInt) {
					unique = false;
					break;
				} else {
					unique = true;
				}
			}
		} while (!unique);

		indeces[i] = randInt;
	}
	return indeces;
}



/* ================================ VECTOR CALCULATIONS AND NAVIATION ==================== */

CVector2 BaseRobot::CalculateVectorToLight() {
   /* Get readings from light sensor */
   const CCI_FootBotLightSensor::TReadings& tLightReads = lightSensor->GetReadings();
   /* Sum them together */
   CVector2 cAccumulator;
   for(size_t i = 0; i < tLightReads.size(); ++i) {
	   cAccumulator += CVector2(tLightReads[i].Value, tLightReads[i].Angle);
   }
   //-- if the light was perceived, return the vector, otherwise return 0
   ////////////LOGERR << "ANGLE " << cAccumulator.Angle() << std::endl;
   if(cAccumulator.Length() > 0.0f) {
	   return CVector2(1.0f, cAccumulator.Angle());
   } else {
	   return CVector2();
   }
}

/*
 * Calculates the diffusion vector. If there is a close obstacle,
 * it points away from it; it there is none, it points forwards.
 * The b_collision parameter is used to return true or false whether
 * a collision avoidance just happened or not. It is necessary for the
 * collision rule.
 */
CVector2 BaseRobot::DiffusionVector(bool b_collision) {
   /* Get readings from proximity sensor */
   const CCI_FootBotProximitySensor::TReadings& tProxReads = proximitySensor->GetReadings();
   /* Sum them together */
   CVector2 cDiffusionVector;
   for(size_t i = 0; i < tProxReads.size(); ++i) {
      cDiffusionVector += CVector2(tProxReads[i].Value, tProxReads[i].Angle);
   }
   ////////////LOGERR << cDiffusionVector.Length() << cDiffusionVector.Angle() << std::endl;
   /* If the angle of the vector is small enough and the closest obstacle
      is far enough, ignore the vector and go straight, otherwise return
      it */
   if(diffustionParams.GoStraightAngleRange.WithinMinBoundIncludedMaxBoundIncluded(cDiffusionVector.Angle()) &&
      cDiffusionVector.Length() < diffustionParams.Delta ) {
      b_collision = false;
      return CVector2::X;
   }
   else {
      b_collision = true;
      cDiffusionVector.Normalize();
      return -cDiffusionVector;
   }

}

/**
 * Returns true if an object was detected by proximity sensors
 */
bool BaseRobot::IsSomethingNearby() {
	const CCI_FootBotProximitySensor::TReadings& tProxReads = proximitySensor->GetReadings();
	for(size_t i = 0; i < tProxReads.size(); ++i) {
		////////////LOGERR << tProxReads[i].Value << " :" << tProxReads[i].Angle << std::endl;
		if (tProxReads[i].Value > 0) {
			return true;
		}
	}
	return false;
}

/**
 * Navigates towards a vector, avoiding objects at the same time.
 * Returns true if collision detected.
 */
bool BaseRobot::NavigateTowardsVector(const CVector2& vector_) {
	bool collision;
	CVector2 diffusionVec = DiffusionVector(collision);

	if (!collision) {
		//-- no collision, head towards the target
		SetWheelSpeedsFromVector(wheelParams.maxSpeed*diffusionVec + vector_);
		collisionAvoidingAngle = CRadians(0);
		//LOGERR << this->GetId() << "Following to vector" << vector_ << std::endl;
	} else {
		//-- has direction of this collision's turn been specified?
		if (collisionAvoidingAngle.GetValue() == 0) {
			//-- no, specify a new one
			if (randNumGen->Uniform(CRange<Real>(0,1)) < 2) {
				collisionAvoidingAngle = CRadians(1);
			} else {
				collisionAvoidingAngle = -CRadians(1);
			}
		}
		//-- turn away
		SetWheelSpeedsFromVector(wheelParams.maxSpeed*diffusionVec);
	}
	return collision;
}
/*
 * Gets a direction vector as input and transforms it into wheel
 * actuation.
 */
void BaseRobot::SetWheelSpeedsFromVector(const CVector2& c_heading) {
   /* Get the heading angle */
   CRadians cHeadingAngle = c_heading.Angle().SignedNormalize();
   ////////////LOGERR << cHeadingAngle << std::endl;
   /* Get the length of the heading vector */
   Real fHeadingLength = c_heading.Length();
   /* Clamp the speed so that it's not greater than MaxSpeed */
   Real fBaseAngularWheelSpeed = Min<Real>(fHeadingLength, wheelParams.maxSpeed);

   ////////////LOGERR << Abs(cHeadingAngle) << std::endl;
   /* Turning state switching conditions */
   if(Abs(cHeadingAngle) <= wheelParams.noTurnAngleThreshold) {
      /* No Turn, heading angle very small */
      wheelParams.turningMechanism = WheelTurningParams::NO_TURN;
      //LOGERR << " > too low " << Abs(cHeadingAngle) << std::endl;
   }
   else if(Abs(cHeadingAngle) > wheelParams.hardTurnOnAngleThreshold) {
      /* Hard Turn, heading angle very large */
      wheelParams.turningMechanism = WheelTurningParams::HARD_TURN;
      //LOGERR << " > hard turn " << Abs(cHeadingAngle) << std::endl;
   }
   else  { //if(wheelTurningParams.TurningMechanism == WheelTurningParams::NO_TURN && Abs(cHeadingAngle) > wheelTurningParams.SoftTurnOnAngleThreshold
      /* Soft Turn, heading angle in between the two cases */
      wheelParams.turningMechanism = WheelTurningParams::SOFT_TURN;
      //LOGERR << " > soft turn " << Abs(cHeadingAngle) << std::endl;
   }

   /* Wheel speeds based on current turning state */
   Real fSpeed1, fSpeed2;
   switch(wheelParams.turningMechanism) {
      case WheelTurningParams::NO_TURN: {
         /* Just go straight */
         fSpeed1 = fBaseAngularWheelSpeed;
         fSpeed2 = fBaseAngularWheelSpeed;
         break;
      }

      case WheelTurningParams::SOFT_TURN: {
         /* Both wheels go straight, but one is faster than the other */
         Real fSpeedFactor = (wheelParams.hardTurnOnAngleThreshold - Abs(cHeadingAngle)) / wheelParams.hardTurnOnAngleThreshold;
         fSpeed1 = fBaseAngularWheelSpeed - fBaseAngularWheelSpeed * (1.0 - fSpeedFactor);
         fSpeed2 = fBaseAngularWheelSpeed + fBaseAngularWheelSpeed * (1.0 - fSpeedFactor);
         break;
      }

      case WheelTurningParams::HARD_TURN: {
         /* Opposite wheel speeds */
         fSpeed1 = -wheelParams.maxSpeed;
         fSpeed2 =  wheelParams.maxSpeed;
         break;
      }
   }

   /* Apply the calculated speeds to the appropriate wheels */
   Real fLeftWheelSpeed, fRightWheelSpeed;
   if(cHeadingAngle > CRadians::ZERO) {
      /* Turn Left */
	  //LOGERR << "Turn left" << "|" << fSpeed1 << "|" << fSpeed2 << std::endl;
      fLeftWheelSpeed  = fSpeed1;
      fRightWheelSpeed = fSpeed2;
   }
   else {
      /* Turn Right */
	 // LOGERR << "Turn right" << "|" << fSpeed2 << "|" << fSpeed1 << std::endl;
      fLeftWheelSpeed  = fSpeed2;
      fRightWheelSpeed = fSpeed1;
   }
   /* Finally, set the wheel speeds */

   SetWheelSpeeds(fLeftWheelSpeed, fRightWheelSpeed);
}

/**
 * Sets wheel speed and remembers the decision
 */
void BaseRobot::SetWheelSpeeds(Real left_, Real right_) {
	wheels->SetLinearVelocity(left_, right_);
}


