#include "beeRecruiter.h"

#include <argos3/core/utility/configuration/argos_configuration.h>
#include <argos3/core/utility/math/vector2.h>
#include <argos3/core/utility/logging/argos_log.h>
#include <algorithm>

/**
 * BeeRecruiter
 * 	SCOUT		----->	(rT=RT) 	-----> 	LOAD
 * 						(else) 		-----> 	RECRUIT IN BASE
 * 						(time up)	----->	OBSERVE
 *
 * 	LOAD 		----->	(t. paused) ----->	RECRUIT IN BASE
 * 						(t. done)	----->	OBSERVE
 *
 * 	RECRUIT		----->	(time up)	-----> (rT >= RT)	-----> WAIT NEAR TASK
 * 											(else)		----->	OBSERVE
 *
 * 	WAIT N. T.	----->	(rT=RT)		----->	LOAD
 * 						(time up)	----->	OBSERVE
 *
 * 	OBSERVE		----->	(recruited)	----->	WAIT NEAR TASK
 * 						(with p(S))	----->	SCOUT
 *
 */

/**
 * Initialisation of set / explored parameters
 */
void BeeRecruiter::Parameters::Init(TConfigurationNode& t_node) {
   try {
	   GetNodeAttribute(t_node, "signalDistance", signalDistance);
	   GetNodeAttribute(t_node, "maxWaggleDanceTime", maxWaggleDanceTime);
	   GetNodeAttribute(t_node, "scoutingProb", scoutingProb);
	   GetNodeAttribute(t_node, "maxScoutingTime", maxScoutingTime);
	   GetNodeAttribute(t_node, "maxTeamSizeUnchangedTime", maxTeamSizeUnchangedTime);

	   //-- convert times in seconds to time in ticks
	   maxWaggleDanceTime *= 10;
	   maxScoutingTime *= 10;
	   maxTeamSizeUnchangedTime *= 10;

   } catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error initializing parameters.", ex);
   }

}


//======================================= INITIALISATIONS
BeeRecruiter::BeeRecruiter() {

}

void BeeRecruiter::Init(TConfigurationNode& t_node) {
	BaseRobot::Init(t_node);
	try {
		parameters.Init(GetNode(t_node, "parameters"));
	}
	catch(CARGoSException& ex) {
		THROW_ARGOSEXCEPTION_NESTED("Error initializing the foot-bot foraging controller for robot \"" << GetId() << "\"", ex);
	}

	/*if (this->GetId() == "robot0") {
		parameters.scoutingProb = 1;
	}*/

}

void BeeRecruiter::Reset() {
	BaseRobot::Reset();
	advertisedTaskInfo = TaskInfo();
	SetState(STATE_SCOUTING);
}


void BeeRecruiter::SetState(STATE state_, const bool& isNewState_) {
	BaseRobot::SetState(state_, isNewState_);
	switch(state_) {
		case STATE_BEE_RECRUITER_WAGGLE_DANCING:
			if (isNewState_) {
				encounteredObserverIds.clear();
				timeWaggleDancing = 0;
			}
			break;
		case STATE_BEE_READY_TO_DO_TASK:
			if (isNewState_) {
				timeWaggleDancing = 0;
			}
			break;
		case STATE_SCOUTING:
			if (isNewState_) {

			}
			break;
		case STATE_BEE_OBSERVING:
			if (isNewState_) {
				isTaskNewlyScouted = false;
			}
			break;
		case STATE_GOING_TO_TASK:
			if (isNewState_) {
				lastNumOfRobotsAtTask = 0;
				timeSinceLastRobotArrivedToTask = 0;
			}
			break;
		default:
			break;
	}

	ResetInfoExchange();

}

void BeeRecruiter::ResetInfoExchange() {
	ResetInfoExchange(false);
}

void BeeRecruiter::ResetInfoExchange(bool listeningOnly_) {
	if (!listeningOnly_) {
		taskInfoAngleToTargetSent = false;
		taskInfoDistanceToTargetSent = false;
		taskInfoToIdSent = false;
		taskInfoTaskIdSent = false;
		taskInfoRelUtilitySent = false;
		taskInfoUtilitySent = false;
		taskInfoNumOfRobotsRequiredSent = false;
	}

	taskInfoAngleToTargetReceived = false;
	taskInfoDistanceToTargetReceived = false;
	taskInfoToIdReceived = false;
	taskInfoTaskIdReceived = false;
	taskInfoRelUtilityReceived = false;
	taskInfoUtilityReceived = false;
	taskInfoNumOfRobotsRequiredReceived = false;
	taskInfoReceivingFromId = -1;

}

//======================================= UPDATE LOOP

/**
 * Main update loop function
 */
void BeeRecruiter::ControlStep() {
	BaseRobot::ControlStep();
	switch(state) {
		case STATE_BEE_RECRUITER_WAGGLE_DANCING:
			PerformWaggleDance();
			leds->SetAllColors(CColor::WHITE);
			break;
		case STATE_BEE_OBSERVING:
			Observe();
			leds->SetAllColors(CColor::YELLOW);
			break;
		case STATE_BEE_READY_TO_DO_TASK:
			WaitForRecruitsInBase();
			leds->SetAllColors(CColor::RED);
			break;
		default:
			break;
	}
}


void BeeRecruiter::Scout() {
	BaseRobot::Scout();
	//-- set the data to none to prevent recruitees from getting stuck sometimes
	rangeAndBearingActuator->SetData(0, SIGNAL_NONE);

	//-- return to base if unsuccessful
	if (timeScouting > parameters.maxScoutingTime) {
		SetState(STATE_RETURN_TO_BASE_UNSUCCESSFUL);
	}
}

/**
 * Triggered once when a robot scouts and approaches a new task
 */
void BeeRecruiter::OnTaskScouted() {
	BaseRobot::OnTaskScouted();
	isTaskNewlyScouted = true;
	didFindEnoughRobotsAtTask = false;
}

//============================================ GOING TO KNOWN TASK =======================
/**
 * Navigate towards a newly acquired / remembered target task
 */
void BeeRecruiter::GoToTask() {
	BaseRobot::GoToTask();
	if (location <= LOCATION_DANCE_FLOOR || location == LOCATION_DEPOSIT) {
		//-- send signals that the robot is at task and find others that are there
		rangeAndBearingActuator->ClearData();
		rangeAndBearingActuator->SetData(4, SIGNAL_READY_TO_DO_TASK);
		rangeAndBearingActuator->SetData(5, targetTaskInfo.taskId);
		rangeAndBearingActuator->SetData(1,id);
		//////////LOGERR << this->GetId() << " YES sign " << std::endl;
	} else {
		//-- set the data to none to prevent recruiters from getting stuck sometimes after they recruit
		rangeAndBearingActuator->SetData(0, SIGNAL_NONE);
		rangeAndBearingActuator->SetData(4, SIGNAL_NONE);
		//////////LOGERR << this->GetId() << " no sign " << std::endl;
	}

}
/**
 *  Triggered when robot reached current task location and task can be executed
 */
void BeeRecruiter::ProcessTaskReached() {
	BaseRobot::ProcessTaskReached();
	SetState(STATE_PROCESSING_TASK);
}

//============================================ TASK PROCESSING =======================
/**
 * Called when robot should be performing a task
 */
Real BeeRecruiter::ProcessTask(TASK_TYPE taskType_, int taskId_,  Real processedAmount_, Real taskValue_, Real taskTotalReward_, uint numOfRobotsRequired_) {

	//-- send signals that the robot is at task and find others that are there
	rangeAndBearingActuator->ClearData();
	rangeAndBearingActuator->SetData(4, SIGNAL_READY_TO_DO_TASK);
	rangeAndBearingActuator->SetData(5, targetTaskInfo.taskId);
	rangeAndBearingActuator->SetData(1,id);

	FindOtherTaskBroadcasters();

	//////////LOGERR << this->GetId() << "  others at task = " << otherWorkersAtTask.size() << " taskId" << taskId_ << " required " << numOfRobotsRequired_ << " t" << timeSinceLastRobotArrivedToTask << std::endl;

	//-- check if enough other robots at task to perform it
	if ((otherWorkersAtTask.size() >= numOfRobotsRequired_-1 || taskId_ < 0) && (!(isTaskNewlyScouted && taskType_ == TASK_PERFORM))) { //|| taskId_ < 0 : so that it can properly abandon task in the BaseRobot::ProcessTask method
		if (location != LOCATION_DEPOSIT) {

			//------- task has already been depleted

			//-- the robot didn't reach the middle of the gradient fast enough and others have already done the work
			if (processedAmount_ > 0) {
				NewInfoEventReport(InfoEventReport::TYPE_TASK_STARTED,taskId_,taskValue_,taskTotalReward_/taskValue_,EstimateTaskUtility(taskType_, taskValue_, taskTotalReward_/taskValue_, relBaseLocation.Length(), false), targetTaskInfo.infoOriginatorId);
			}
			AbandonTask();
			SetState(STATE_RETURN_TO_BASE_UNSUCCESSFUL);
			/*if (this->GetNumId() == 17) {
				//////LOGERR << this->GetId() << " ABANDON, task " << taskInfo.taskId << "( from env: " << taskId_ << ") has been depleted" << std::endl;
			}*/
		} else {

			//-------- task is valid and should be done
			/*if (this->GetNumId() == 12) {
				//////LOGERR << " at task " << std::endl;
			}*/
			//-- re-travel to the task's middle to be able to perform it
			didFindEnoughRobotsAtTask = true;
			if (NavigateTowardsDeposit()) {
				didReportScoutEvent = true;
				return BaseRobot::ProcessTask(taskType_, taskId_, processedAmount_, taskValue_, taskTotalReward_, numOfRobotsRequired_);
			}
		}
	} else {
		if (isTaskNewlyScouted) {

			//------ need to go to base and recruit

			//-- new scout just arrived at a task and there are not enough robot around to do it - go to base and recruit
			EstimateTaskUtility(taskType_, taskValue_, taskTotalReward_/taskValue_, relBaseLocation.Length());
			SetNewTarget(taskType_, taskId_, taskValue_, CVector2(0,0), id, targetTaskInfo.taskUtility, 1, numOfRobotsRequired_);
			isTaskNewlyScouted = false;
			CalculateWaggleDaceTime();

			////LOGERR << this->GetId() << " New scouted, go recruit t" << taskId_ << std::endl;

			SetState(STATE_RETURN_TO_BASE);

		} else {

			//------ need to wait for other recruits to arrive TODO: check this since the SetNewTarget change

			DoRandomWalk(wheelParams.maxSpeed/8,0.1);

			//-- remember the task if it's new
			if (targetTaskInfo.taskId != taskId_ || targetTaskInfo.isNull) {

				////LOGERR << this->GetId() << " depleting new " << taskId_ << std::endl;

				SetNewTarget(taskType_, taskId_, taskValue_, CVector2(0,0), id, targetTaskInfo.taskUtility, 1, numOfRobotsRequired_);
			}
			/*if (this->GetNumId() == 12) {
				//////LOGERR << this->GetId() << " Wait for recruits t" << taskId_ << " others " << otherWorkersAtTask.size() << " t" << timeSinceLastRobotArrivedToTask << std::endl;
			}*/
			//-- if any new robots have been recruited, reset the timeSinceLastRecruitment time
			if (lastNumOfRobotsAtTask < otherWorkersAtTask.size()) {
				//////////LOGERR << this->GetId() << " ! RESET now " << otherWorkersAtTask.size() << "  last " << lastNumOfRobotsAtTask << std::endl;
				timeSinceLastRobotArrivedToTask = 0;
				lastNumOfRobotsAtTask = otherWorkersAtTask.size();
			}


			//-- check if no one has been recruited for a long time
			timeSinceLastRobotArrivedToTask ++;
			if (timeSinceLastRobotArrivedToTask > parameters.maxTeamSizeUnchangedTime) {
				/*if (this->GetNumId() == 17) {
					//////LOGERR << "HERE2 "<< std::endl;
				}*/
				AbandonTask();
				SetState(STATE_RETURN_TO_BASE_UNSUCCESSFUL_BLIND);
			}
		}
	}

	return 0;
}

/**
 * Triggered once when the robot has finished or paused (during collection, to return to the base) a task
 */
void BeeRecruiter::OnTaskProcessingDone(TASK_TYPE taskType_, int taskId_,  Real processedAmount_, Real taskValue_, Real taskRewardLeft_, uint numOfRobotsRequired_) {
	BaseRobot::OnTaskProcessingDone(taskType_, taskId_, processedAmount_, taskValue_, taskRewardLeft_, numOfRobotsRequired_);

	CalculateWaggleDaceTime();
	//-- base robot will go scouting when a task is done, but beeRecruiter should return to the base to become an observer
	if (state == STATE_SCOUTING) {
		SetState(STATE_RETURN_TO_BASE_UNSUCCESSFUL);
		timeTillNextScoutAllowed = DELAY_BETWEEN_SCOUTINGS;
		//////LOGERR << this->GetId() << "Nothing else to load, returning to base " << std::endl;
	}

	//////////LOGERR << this->GetId() << " TASK U" << taskInfo.taskUtility << " rel " << taskInfo.relTaskUtility << " Wt=" << waggleDanceTime << std::endl;
}

/**
 *  Triggered when robot reached current task location but nothing is there
 */
void BeeRecruiter::OnTaskReachedButDoesntExist() {
	BaseRobot::OnTaskReachedButDoesntExist();
	SetState(STATE_RETURN_TO_BASE_UNSUCCESSFUL);
}

/**
 * Triggered once when all resource has been unloaded
 */
void BeeRecruiter::OnUnloadResourceDone() {
	BaseRobot::OnUnloadResourceDone();
	if (targetTaskInfo.isNull) {
		//////////LOGERR << this->GetId() << "!! DONE UNLOADING, no task in memory " << std::endl;
		SetState(STATE_BEE_OBSERVING);
	} else {
		////////LOGERR << this->GetId() << "!! DONE UNLOADING, go to recruit for task " << std::endl;
		SetState(STATE_BEE_RECRUITER_WAGGLE_DANCING);
	}

}


/**
 * Return to nest with resource in cart
 */
void BeeRecruiter::ReturnToBase() {
	BaseRobot::ReturnToBase();
	if (location <= LOCATION_DANCE_FLOOR || (location == LOCATION_DEPOSIT && this->resourceInCartVolume > 0) ) {
		//-- send signals that the robot is at task and find others that are there
		rangeAndBearingActuator->ClearData();
		rangeAndBearingActuator->SetData(4, SIGNAL_READY_TO_DO_TASK);
		rangeAndBearingActuator->SetData(5, targetTaskInfo.taskId);
		rangeAndBearingActuator->SetData(1,id);
	} else {
		//-- set the data to none
		rangeAndBearingActuator->SetData(0, SIGNAL_NONE);
		rangeAndBearingActuator->SetData(4, SIGNAL_NONE);
	}
}

/**
 * Listen to waggle dances on the dance floor
 */
void BeeRecruiter::Observe() {
	//-- make sure it is on dance floor
	if (location > LOCATION_DANCE_FLOOR) {
		NavigateToLight();
	} else {
		//-- slow random walk across the dance floor
		DoRandomWalk(wheelParams.maxSpeed);
	}

	ReceiveTaskInfo(false, true);

	//-- scout with scouting probability
	if (randNumGen->Uniform(CRange<Real>(0,1)) < parameters.scoutingProb) {
		SetState(STATE_SCOUTING);
	}

}


//====================================== COMMUNICATION

/**
 * Stay near the task deposit and broadcast it to others
 */
void BeeRecruiter::PerformWaggleDance() {

	FindOtherTaskBroadcasters();
	if (otherWorkersAtTask.size() >= targetTaskInfo.numOfRobotsRequired - 1) {
		didFindEnoughRobotsAtTask = true;
	}

	//-- make sure it's on the dance floor
	if (location > LOCATION_DANCE_FLOOR) {
		NavigateToLight();
	} else {
		//-- slow random walk across the dance floor
		DoRandomWalk(wheelParams.maxSpeed);
		timeWaggleDancing++;
		//-- check if should end waggle dance
		if (timeWaggleDancing >= waggleDanceTime) {
			if (targetTaskInfo.numOfRobotsRequired == 1 || didFindEnoughRobotsAtTask) {
				SetState(STATE_GOING_TO_TASK);
			} else {
				//////////LOGERR << this->GetId() << "!! done waggle dancing, not enough recruits";
				AbandonTask();
				SetState(STATE_BEE_OBSERVING);
			}
		}
	}

	//-- send signals
	rangeAndBearingActuator->ClearData();
	rangeAndBearingActuator->SetData(4, SIGNAL_TASK_INFO_AVAILABLE);
	rangeAndBearingActuator->SetData(5, targetTaskInfo.taskId);
	rangeAndBearingActuator->SetData(1,id);


	//-- generate random order in which packets are read
	const CCI_RangeAndBearingSensor::TReadings& packets = rangeAndBearingSensor->GetReadings();
	int* indeces = GenerateRandomPacketOrder(packets);

	//-- read signals from observers
	CRadians angleToObserver = CRadians(0);
	int toId = -1;
	isCommunicating = false;
	for(int p=0; p < packets.size(); p++) {
		int packetId = indeces[p];

		//-- find out if someone is receiving from the robot's id - if yes, remember to communicate with that robot
		if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_RECEIVING && packets[packetId].Data[1] == id && packets[packetId].Range < parameters.signalDistance) {
			toId = packets[packetId].Data[2];
			bool alreadyCommunicatedWithIt = false;
			//-- beginning of new communication stream, check if already did this with this robot
			for(UInt32 j = 0; j < encounteredObserverIds.size(); j++) {
				if (encounteredObserverIds[j] == toId) {
					alreadyCommunicatedWithIt = true;
					break;
				}
			}

			if (!alreadyCommunicatedWithIt) {
				////////LOGERR << this->GetId() << " communicating to " << toId << " " << packets[packetId].Data[0] << std::endl;
				isCommunicating = true;
				SetWheelSpeeds(0,0);
				angleToObserver = packets[packetId].HorizontalBearing;
			} else {
				////////LOGERR << this->GetId() << " already communicated with " << toId << " " << packets[packetId].Data[0] << std::endl;
				toId = 0;
				isCommunicating = false;
			}
		}

	}
	delete[] indeces;

	//-- broadcast the task info
	if (isCommunicating) {
		SendTaskInfoToRobot(angleToObserver, toId);
	} else {
		if (baseParameters.useOpportunism) {
			////////LOGERR << this->GetId() << isCommunicating << std::endl;
			ReceiveTaskInfo(true, false);
		}

	}
}

void BeeRecruiter::WaitForRecruitsInBase() {

	//-- send signals
	/*rangeAndBearingActuator->ClearData();
	rangeAndBearingActuator->SetData(4, SIGNAL_READY_TO_DO_TASK);
	rangeAndBearingActuator->SetData(5, taskInfo.taskId);
	rangeAndBearingActuator->SetData(1,id);

	FindOtherTaskBroadcasters();

	if (otherWorkersAtTask.size() >= taskInfo.numOfRobotsRequired - 1) {
		SetState(STATE_GOING_TO_TASK);
	}

	//-- make sure it's on the dance floor
	if (location > LOCATION_DANCE_FLOOR) {
		NavigateToLight();
	} else {
		//-- slow random walk across the dance floor
		DoRandomWalk(wheelParams.maxSpeed);
		timeWaggleDancing++;
		//-- check if shouls end waiting for others and just become an observer again
		if (timeWaggleDancing >= waggleDanceTime) {
			AbandonTask();
			SetState(STATE_BEE_OBSERVING);
		}
	}*/



}

/**
 * Send parts of the waggle dance one by one. In each packet, its bits are:
 * 0 = signal type
 * 1 = id of the robot
 * 2-9 = encoded float value
 */
void BeeRecruiter::SendTaskInfoToRobot(CRadians angleToObserver_, int toId_) {
	if (isCommunicating) {
		////////////LOGERR << this->GetId() << " RECRUITING " << toId_ << std::endl;

		//-- send data
		Real valueToSend;
		if (!taskInfoToIdSent) {
			valueToSend =  toId_;
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_TO_ID);
			taskInfoToIdSent = true;
		} else if (!taskInfoAngleToTargetSent) {
			//-- send angle to target, adjusted by angle to the receiver
			valueToSend = (angleToObserver_ - (CRadians::PI_OVER_TWO + targetTaskInfo.relLocation.Angle())).UnsignedNormalize().GetValue();
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_ANGLE_TO_TARGET);
			taskInfoAngleToTargetSent = true;
		} else if (!taskInfoDistanceToTargetSent) {
			//-- send distance to the target
			valueToSend = targetTaskInfo.relLocation.Length();
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_DISTANCE_TO_TARGET);
			taskInfoDistanceToTargetSent = true;
		} else if (!taskInfoUtilitySent) {
			//-- send distance to the target
			valueToSend = targetTaskInfo.taskUtility;
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_UTILITY);
			taskInfoUtilitySent = true;
		} else if (!taskInfoRelUtilitySent) {
			//-- send distance to the target
			valueToSend = targetTaskInfo.relTaskUtility;
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_REL_UTILITY);
			taskInfoRelUtilitySent = true;
		} else if (!taskInfoNumOfRobotsRequiredSent) {
			//-- send distance to the target
			valueToSend = targetTaskInfo.numOfRobotsRequired;
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_NUM_OF_ROBOTS_REQUIRED);
			taskInfoNumOfRobotsRequiredSent = true;
		} else if (!taskInfoTaskIdSent) {
			//-- send distance to the target
			valueToSend = targetTaskInfo.taskId;
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_TASK_ID);
			taskInfoTaskIdSent = true;
		}
		if (this->GetNumId() == 21) {
			//////LOGERR << this->GetId() << "Sending task id " << targetTaskInfo.taskId << " sig type " << SIGNAL_TASK_INFO_TASK_ID << std::endl;
		}

		//-- add the encoded float into the packet data
		UInt8* encodedFloat = (UInt8*)(&valueToSend);
		for (int i=0; i<8; i++) {
			rangeAndBearingActuator->SetData(i+2,encodedFloat[i]);
		}

		//-- check if all information has been sent, in which case memory of what was sent is reset
		if (taskInfoToIdSent && taskInfoAngleToTargetSent && taskInfoDistanceToTargetSent && taskInfoTaskIdSent && taskInfoUtilitySent && taskInfoRelUtilitySent && taskInfoNumOfRobotsRequiredSent) {
			taskInfoToIdSent = false;
			taskInfoAngleToTargetSent = false;
			taskInfoDistanceToTargetSent = false;
			taskInfoTaskIdSent = false;
			taskInfoUtilitySent = false;
			taskInfoRelUtilitySent = false;
			taskInfoNumOfRobotsRequiredSent = false;
			////////LOGERR << this->GetId() << " remembering recruit " << toId_ << std::endl;
			encounteredObserverIds.push_back(toId_);
		}


	}
}

/*
 * Find and count how many others are broadcasting about the current task.
 * On the dance floor, this will be all the other recruiters that are recruiting for a task.
 * Near a task, this will be all the robots that are waiting at the task for enough robots to come or the robots that just scouted the task.
 */
void BeeRecruiter::FindOtherTaskBroadcasters() {
	otherWorkersAtTask.clear();
	const CCI_RangeAndBearingSensor::TReadings& packets = rangeAndBearingSensor->GetReadings();
	int* indeces = GenerateRandomPacketOrder(packets);
	for(int p=0; p<packets.size(); p++) {
		int packetId = indeces[p];
		//if ((packets[packetId].Data[0] >= SIGNAL_TASK_INFO_TO_ID && packets[packetId].Data[0] <= SIGNAL_TASK_INFO_TASK_ID) ||
		if ((packets[packetId].Data[4] == SIGNAL_TASK_INFO_AVAILABLE || packets[packetId].Data[4] == SIGNAL_READY_TO_DO_TASK) && packets[packetId].Data[5] == targetTaskInfo.taskId ) {
			//////////LOGERR << this->GetId() << " found another at task " << packets[i].Data[1] << std::endl;
			bool alreadyNoted = false;
			for(UInt32 j = 0; j < otherWorkersAtTask.size(); j++) {
				if (otherWorkersAtTask[j] == packets[packetId].Data[1]) {
					alreadyNoted = true;
					break;
				}
			}
			if (!alreadyNoted) {
				//////////LOGERR << this->GetId() << " encountered " << packets[packetId].Data[1] << " sig types" << packets[packetId].Data[0] << " " << packets[packetId].Data[4] << std::endl;
				otherWorkersAtTask.push_back(packets[packetId].Data[1]);
			}
		}
	}
	//////////LOGERR << this->GetId() << " encountered " << otherWorkersAtTask.size() << " robots " << std::endl;
}

void BeeRecruiter::ReceiveTaskInfo(bool adoptWorksiteOnlyIfBetter_, bool goToWorksiteWhenReceived_) {

	bool sendingAcknowledgementSignalToRecruiter = false;

	//-- generate random order in which packets are read
	const CCI_RangeAndBearingSensor::TReadings& packets = rangeAndBearingSensor->GetReadings();
	int* indeces = GenerateRandomPacketOrder(packets);

	bool alreadyRecruited = false;
	for(int p=0; p<packets.size(); p++) {
		int packetId = indeces[p];

		//-- find out if this is any part of the recruitment signal and if signal is close enough
		//////////LOGERR << this->GetId() << " got packed from " << packets[i].Data[1] << " type " << packets[i].Data[0] <<  "d=" << parameters.signalDistance << std::endl;
		if (((packets[packetId].Data[0] >= SIGNAL_TASK_INFO_TO_ID && packets[packetId].Data[0] <= SIGNAL_TASK_INFO_TASK_ID) || packets[packetId].Data[4] == SIGNAL_TASK_INFO_AVAILABLE) && packets[packetId].Range < parameters.signalDistance) { //

			//-- if in the middle of receiving from another robot, ignore this
			if ((taskInfoAngleToTargetReceived || taskInfoDistanceToTargetReceived || taskInfoToIdReceived || taskInfoTaskIdReceived || taskInfoUtilityReceived || taskInfoRelUtilityReceived || taskInfoNumOfRobotsRequiredReceived) && taskInfoReceivingFromId != packets[packetId].Data[1]) {
				//-- someone else's message, ignore
				//////////LOGERR << this->GetId() << "!!! Ignore recruitmend from " << packets[i].Data[1] << std::endl;
			} else {
				//-- set who currently receiving from
				//isCommunicating = true;
				sendingAcknowledgementSignalToRecruiter = true;
				taskInfoReceivingFromId = packets[packetId].Data[1];
				/*if (this->GetNumId() == 21) {
					//////LOGERR << this->GetId() << "Receiving rec from " << taskInfoReceivingFromId  << " signal type " << packets[packetId].Data[0] << std::endl;
				}*/
				CRadians angleToSender = packets[packetId].HorizontalBearing;

				//-- stop moving
				//SetWheelSpeeds(0,0);

				//-- decode the float
				UInt8* encodedFloat;
				encodedFloat = new UInt8[8];
				for(size_t d = 2; d < 10; ++d) {
					encodedFloat[d-2] = packets[packetId].Data[d];
				}
				Real receivedFloat = *(Real*)encodedFloat;
				delete encodedFloat;

				//-- remember it
				if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_TO_ID) {
					SetWheelSpeeds(0,0);
					//-- reset everything if this message is for a different robot
					if (int(receivedFloat) != this->GetNumId()) {
						if (!alreadyRecruited) { // don't change anything if the robot was already recruited
							rangeAndBearingActuator->ClearData();
							if (this->GetNumId() == 21) {
								//////LOGERR << this->GetId() << "received by mistake package of robot " << receivedFloat << std::endl;
							}
							ResetInfoExchange();
						}
					} else {
						taskInfoToIdReceived = true;
						/*if (this->GetNumId() == 21) {
							//////LOGERR << " received info id" << std::endl;
						}*/
					}
				}
				else if (taskInfoToIdReceived) {
					SetWheelSpeeds(0,0);
					if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_ANGLE_TO_TARGET) {
						//-- use the x axis of the vector to store angle for now
						advertisedTaskInfo.relLocation.SetX(-receivedFloat);
						taskInfoAngleToTargetReceived = true;
					} else if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_DISTANCE_TO_TARGET) {
						//-- use y axis of the vector to store distance for now
						advertisedTaskInfo.relLocation.SetY(receivedFloat);
						taskInfoDistanceToTargetReceived = true;
					} else if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_UTILITY) {
						advertisedTaskInfo.taskUtility = receivedFloat;
						taskInfoUtilityReceived = true;
					} else if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_REL_UTILITY) {
						advertisedTaskInfo.relTaskUtility = receivedFloat;
						taskInfoRelUtilityReceived = true;
					} else if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_NUM_OF_ROBOTS_REQUIRED) {
						advertisedTaskInfo.numOfRobotsRequired = receivedFloat;
						taskInfoNumOfRobotsRequiredReceived = true;
					} else if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_TASK_ID) {
						advertisedTaskInfo.taskId = int(receivedFloat);
						taskInfoTaskIdReceived = true;
					}
				}

				/*if (this->GetNumId() == 21) {
					//////LOGERR << taskInfoToIdReceived << taskInfoAngleToTargetReceived << taskInfoDistanceToTargetReceived << taskInfoTaskIdReceived << taskInfoUtilityReceived << taskInfoRelUtilityReceived << taskInfoRelUtilityReceived << std::endl;
				}*/

				//-- check if received everything about the advertised deposit
				if (taskInfoAngleToTargetReceived && taskInfoDistanceToTargetReceived && taskInfoToIdReceived && taskInfoTaskIdReceived && taskInfoUtilityReceived && taskInfoRelUtilityReceived && taskInfoNumOfRobotsRequiredReceived) {
					if (this->GetNumId() == 21) {
						//////LOGERR << this->GetId() << " all info received " << std::endl;
					}
					bool adoptWorksite = true;
					Real debugPrevU = -1;
					if (adoptWorksiteOnlyIfBetter_) {
						//-- compare the advertised and current worksite
						if (!targetTaskInfo.isNull) {
							//if (this->GetNumId() == 21 || this->GetNumId() == 8) {
							debugPrevU = targetTaskInfo.taskUtility;
							//LOGERR << this->GetId() << " remembered U " << targetTaskInfo.taskUtility << " advertised " << advertisedTaskInfo.taskUtility << "(by robot " << taskInfoReceivingFromId << ")" <<  std::endl;
							if (advertisedTaskInfo.taskUtility - targetTaskInfo.taskUtility <= baseParameters.opportunismThreshold) {
								//-- the advertised is not sufficiently better than the current worksite
								adoptWorksite = false;
							} else {
								//LOGERR << this->GetId() << " !!!! OPPORTUNISM " << std::endl;
							}
						} else {
							////////LOGERR << this->GetId() << " robot doesn't have a worksite in memory " << std::endl;
						}
					}
					if (adoptWorksite) {
						CRadians gamma = CRadians(advertisedTaskInfo.relLocation.GetX()) + packets[packetId].HorizontalBearing - CRadians::PI_OVER_TWO;
						CVector2 translatedPos = -CVector2(packets[packetId].Range/100 * cos(packets[packetId].HorizontalBearing.GetValue()) + advertisedTaskInfo.relLocation.GetY() * cos(gamma.GetValue()),
													packets[packetId].Range/100 * sin(packets[packetId].HorizontalBearing.GetValue()) + advertisedTaskInfo.relLocation.GetY() * sin(gamma.GetValue()));

						//-- make the deposit current target and go foraging for it
						alreadyRecruited = true;
						targetTaskInfo.isNull = false;
						targetTaskInfo.relLocation = CVector2(translatedPos.GetX(), translatedPos.GetY());
						targetTaskInfo.taskUtility = advertisedTaskInfo.taskUtility;
						targetTaskInfo.relTaskUtility = advertisedTaskInfo.relTaskUtility;
						targetTaskInfo.numOfRobotsRequired = advertisedTaskInfo.numOfRobotsRequired;
						targetTaskInfo.taskId = advertisedTaskInfo.taskId;
						targetTaskInfo.infoOriginatorId = taskInfoReceivingFromId;

						NewInfoEventReport(InfoEventReport::TYPE_ROBOT_RECRUITED,targetTaskInfo.taskId,-1,-1,-1,targetTaskInfo.infoOriginatorId);

						didFindEnoughRobotsAtTask = false;
						//LOGERR << this->GetId() << " recruited to task " << targetTaskInfo.taskId << " U " << targetTaskInfo.taskUtility << std::endl;

						ResetInfoExchange();
						if (goToWorksiteWhenReceived_) {
							SetState(STATE_GOING_TO_TASK);
							////////LOGERR << this->GetId() << " GOING TO TASK " << taskInfo.taskId << std::endl;
						}
						//NOTE: robots will just wait at each other near the task, this will make dance floor less congested.
						/*if (taskInfo.numOfRobotsRequired == 1) {
							////////LOGERR << this->GetId() << "----- recruited for U" << taskInfo.taskUtility << "  " << taskInfo.relTaskUtility << std::endl;
							SetState(STATE_GOING_TO_TASK);
						} else {
							CalculateWaggleDaceTime();
							////////LOGERR << this->GetId() << "----- recruited for U" << taskInfo.taskUtility << "  " << taskInfo.relTaskUtility << " Wt=" << waggleDanceTime << std::endl;
							SetState(STATE_BEE_READY_TO_DO_TASK);
						}*/
					} else {
						ResetInfoExchange(true);
					}
				}
			}
		}
	}
	delete[] indeces;

	if (sendingAcknowledgementSignalToRecruiter) {
		//-- send a signal to the robot specifying someone is receiving
		rangeAndBearingActuator->ClearData();
		rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_RECEIVING);
		rangeAndBearingActuator->SetData(1,taskInfoReceivingFromId);
		rangeAndBearingActuator->SetData(2,this->GetNumId());
	} else {
		/*if (this->GetNumId() == 21) {
			//////LOGERR << " receiving from id " <<  taskInfoReceivingFromId << std::endl;
		}*/
		if (taskInfoReceivingFromId >= 0) { //only reset when abandoning listening, we don't want to clear range and bearing data on every loop because other signals might be being sent

			/*taskInfoToIdReceived = false;
			taskInfoAngleToTargetReceived = false;
			taskInfoDistanceToTargetReceived = false;
			taskInfoTaskIdReceived = false;
			taskInfoUtilityReceived = false;
			taskInfoRelUtilityReceived = false;
			taskInfoNumOfRobotsRequiredReceived = false;
			rangeAndBearingActuator->SetData(0, SIGNAL_NONE);
			if (this->GetNumId() == 21) {
				//////LOGERR << " resetting listening" << std::endl;
			}*/
		}
	}
}


/* ================================ PERCEPTIONS, PROBABILITIES AND TIMES =============================== */
/**
 * Calculate task utility depending on its volume, value, type and distance from the base (robot always needs to go back to base when this is used)
 */

Real BeeRecruiter::DoTaskUtilityCalculation(TASK_TYPE taskType_, Real taskValue_, Real taskVolume_, Real distanceFromBase_) {
	return(taskValue_*taskVolume_ / distanceFromBase_);
}


void BeeRecruiter::CalculateWaggleDaceTime() {
	waggleDanceTime = targetTaskInfo.relTaskUtility*parameters.maxWaggleDanceTime;
	////////LOGERR << this->GetId() << " rel U  " << taskInfo.relTaskUtility << " T waggle dance " << waggleDanceTime << std::endl;
}



/****************************************/
/****************************************/

/*
 * This statement notifies ARGoS of the existence of the controller.
 * It binds the class passed as first argument to the string passed as
 * second argument.
 * The string is then usable in the XML configuration file to refer to
 * this controller.
 * When ARGoS reads that string in the XML file, it knows which controller
 * class to instantiate.
 * See also the XML configuration files for an example of how this is used.
 */
REGISTER_CONTROLLER(BeeRecruiter, "beeRecruiterController")


