#include "localBroadcaster.h"

#include <argos3/core/utility/configuration/argos_configuration.h>
#include <argos3/core/utility/math/vector2.h>
#include <argos3/core/utility/logging/argos_log.h>
#include <algorithm>

/**
 * LocalBroadcaster
 * 	SCOUT		----->	(rT=RT) -----> 		LOAD
 * 						(else) ------> 		WAIR NEAR TASK
 *
 * 	LOAD 		----->	(task done) ----->	SCOUT
 *
 * 	WAIT N.T.	----->	(rT=RT)		----->	LOAD
 * 						(time up)	----->	SCOUT
 */
/**
 * Initialisation of set / explored parameters
 */
void LocalBroadcaster::Parameters::Init(TConfigurationNode& t_node) {
   try {
	   GetNodeAttribute(t_node, "signalDistance", signalDistance);
	   GetNodeAttribute(t_node, "maxTeamSizeUnchangedTime", maxTeamSizeUnchangedTime);

	   //-- convert times in seconds to time in ticks
	   maxTeamSizeUnchangedTime *= 10;

   } catch(CARGoSException& ex) {
      THROW_ARGOSEXCEPTION_NESTED("Error initializing parameters.", ex);
   }

}


//======================================= INITIALISATIONS
LocalBroadcaster::LocalBroadcaster() {

}

void LocalBroadcaster::Init(TConfigurationNode& t_node) {
	BaseRobot::Init(t_node);
	try {
		parameters.Init(GetNode(t_node, "parameters"));
	}
	catch(CARGoSException& ex) {
		THROW_ARGOSEXCEPTION_NESTED("Error initializing the foot-bot foraging controller for robot \"" << GetId() << "\"", ex);
	}

}

void LocalBroadcaster::Reset() {
	BaseRobot::Reset();
	SetState(STATE_SCOUTING);
	advertisedTaskInfo = TaskInfo();
}


void LocalBroadcaster::SetState(STATE state_, const bool& isNewState_) {
	BaseRobot::SetState(state_, isNewState_);

	if (isNewState_ && (state_ == STATE_GOING_TO_TASK || state_ == STATE_SCOUTING)) {
		encounteredObserverIds.clear();
		//LOGERR << this->GetId() << " clearing encountered observers memory " << std::endl;
	}


	timeTillRerecruitmentAllowed = 0; //each 200 ticks caters for 1m distance travelled
	forbiddenTaskId = -1;

	ResetInfoExchange();
}

void LocalBroadcaster::ResetInfoExchange() {
	ResetInfoExchange(false);
}

void LocalBroadcaster::ResetInfoExchange(bool listeningOnly_) {
	if (!listeningOnly_) {
		taskInfoAngleToTargetSent = false;
		taskInfoDistanceToTargetSent = false;
		taskInfoToIdSent = false;
		taskInfoTaskIdSent = false;
		taskInfoRelUtilitySent = false;
		taskInfoUtilitySent = false;
		taskInfoNumOfRobotsRequiredSent = false;
	}

	taskInfoAngleToTargetReceived = false;
	taskInfoDistanceToTargetReceived = false;
	taskInfoToIdReceived = false;
	taskInfoTaskIdReceived = false;
	taskInfoRelUtilityReceived = false;
	taskInfoUtilityReceived = false;
	taskInfoNumOfRobotsRequiredReceived = false;
	taskInfoReceivingFromId = -1;
}

//======================================= UPDATE LOOP

/**
 * Main update loop function
 */
void LocalBroadcaster::ControlStep() {
	BaseRobot::ControlStep();

	if (timeTillRerecruitmentAllowed > 0) {
		timeTillRerecruitmentAllowed --;
		if (timeTillRerecruitmentAllowed == 0) {
			forbiddenTaskId = -1;
		}
	}
}


void LocalBroadcaster::Scout() {
	BaseRobot::Scout();
	/*if (this->GetNumId() == 19) {
		//LOGERR << " scouting " << std::endl;
	}*/
	//-- receive task info signals
	if (timeTillNextScoutAllowed <= 0) {
		ReceiveTaskInfo(false);
	} else {
		rangeAndBearingActuator->ClearData();
		rangeAndBearingActuator->SetData(0, SIGNAL_NONE);
		rangeAndBearingActuator->SetData(4, SIGNAL_NONE);
	}
}

/**
 * Triggered once when a robot scouts and approaches a new task
 */
void LocalBroadcaster::OnTaskScouted() {
	BaseRobot::OnTaskScouted();
	timeSinceLastRecruitment = 0;
	lastNumOfRecruitedRobots = 0;
}

//============================================ GOING TO KNOWN TASK =======================
/**
 * Navigate towards a newly acquired / remembered target task
 */
void LocalBroadcaster::GoToTask() {
	BaseRobot::GoToTask();
	timeSinceLastRecruitment = 0;
	lastNumOfRecruitedRobots = 0;
	if (location == LOCATION_DEPOSIT && nearbyTaskInfo.taskId == targetTaskInfo.taskId && targetTaskInfo.isNull == false) {
		BroadcastTask();
	}

	if (baseParameters.useOpportunism && timeTillNextScoutAllowed <= 0) {
		ReceiveTaskInfo(true);
	}
}

void LocalBroadcaster::DoNeighbourhoodSearch() {
	BaseRobot::DoNeighbourhoodSearch();
	if (timeTillNextScoutAllowed <= 0) {
		ReceiveTaskInfo(false);
	}
	/*if (this->GetNumId() == 19) {
		//LOGERR << " doing neigh search " << std::endl;
	}*/
}

/**
 *  Triggered when robot reached current task location and task can be executed
 */
void LocalBroadcaster::ProcessTaskReached() {
	BaseRobot::ProcessTaskReached();
	SetState(STATE_PROCESSING_TASK);
}

//============================================ TASK PROCESSING =======================
/**
 * Called when robot should be performing a task
 */
Real LocalBroadcaster::ProcessTask(TASK_TYPE taskType_, int taskId_,  Real processedAmount_, Real taskValue_, Real taskTotalReward_, uint numOfRobotsRequired_) {
	FindOtherTaskBroadcasters();

	//-- if any new robots have been recruited, reset the timeSinceLastRecruitment time
	if (lastNumOfRecruitedRobots < encounteredIdsAtTask.size()) {
		timeSinceLastRecruitment = 0;
		lastNumOfRecruitedRobots = encounteredIdsAtTask.size();
		////LOGERR << this->GetId() << " new recruit, reset timer " << std::endl;
	}

	////LOGERR << this->GetId() << "  others at task = " << encounteredIdsAtTask.size() << " req " << numOfRobotsRequired_ << " time since last rec? " << timeSinceLastRecruitment << " taskId" << taskId_ << std::endl;

	if (encounteredIdsAtTask.size() >= numOfRobotsRequired_-1 || taskId_ < 0 ) { //  || taskId_ < 0 : so that it can properly abandon task in the BaseRobot::ProcessTask method

		if (location != LOCATION_DEPOSIT) {
			//-- the robot didn't reach the middle of the gradient fast enough and others have already done the work
			if (processedAmount_ > 0) {
				NewInfoEventReport(InfoEventReport::TYPE_TASK_STARTED,taskId_,taskValue_,taskTotalReward_/taskValue_,EstimateTaskUtility(taskType_, taskValue_, taskTotalReward_/taskValue_, relBaseLocation.Length(), false), targetTaskInfo.infoOriginatorId);
			}
			AbandonTask();
			SetState(STATE_SCOUTING);
		} else {
			//-- remember the task and broadcast it to others if it's valid
			if (taskId_ >= 0) {
				SetNewTarget(taskType_, taskId_, taskValue_, CVector2(0,0), id, targetTaskInfo.taskUtility, targetTaskInfo.relTaskUtility, numOfRobotsRequired_);
				BroadcastTask();
			}
			//-- at the same time, re-travel to the task's middle to be able to perform it
			if (NavigateTowardsDeposit()) {
				return BaseRobot::ProcessTask(taskType_, taskId_, processedAmount_, taskValue_, taskTotalReward_, numOfRobotsRequired_);
			}
		}
	} else {

		//-- check if no one has been recruited for a long time
		timeSinceLastRecruitment ++;
		if (timeSinceLastRecruitment > parameters.maxTeamSizeUnchangedTime) {
			////LOGERR << this->GetId() << " ABANDON DUE TO LONG WAIT " << std::endl;
			AbandonTask();
			SetState(STATE_SCOUTING);
			timeTillNextScoutAllowed = 1200;
			return 0;
		}

		//-- broadcast message and try to recruit others
		////LOGERR << this->GetId() << "  waiting for others at task = " << encounteredIdsAtTask.size() << " req " << numOfRobotsRequired_ << " time since last rec? " << timeSinceLastRecruitment << " taskId" << taskId_ << std::endl;

		DoRandomWalk(wheelParams.maxSpeed/8,0.5);

		//-- since the robot is moving around the task, need to check if it reached another task by mistake and abandon the old.
		//	!!! best make sure this doesn't happen in the environment, because there will be inconsistencies with recruitment and task ids.
		//	this is taken care of in mainLoopFunctions when deposits are placed. If TYPE_TASK_SWITCHED event still occurs, there is a bug..
		if (targetTaskInfo.isNull == false && taskId_ != targetTaskInfo.taskId) {
			NewInfoEventReport(InfoEventReport::TYPE_TASK_SWITCHED,taskId_,taskValue_,taskTotalReward_,EstimateTaskUtility(taskType_, taskValue_, taskTotalReward_, relBaseLocation.Length(), false),targetTaskInfo.infoOriginatorId);
		}

		//-- remember the task and broadcast it to others if it's valid and new
		if (targetTaskInfo.taskId != taskId_ || targetTaskInfo.isNull) {
			SetNewTarget(taskType_, taskId_, taskValue_, CVector2(0,0), id, targetTaskInfo.taskUtility, 1, numOfRobotsRequired_);
		}
		BroadcastTask();


	}
	return 0;
}

/**
 * Triggered once when the robot has finished or paused (during collection, to return to the base) a task
 */
void LocalBroadcaster::OnTaskProcessingDone(TASK_TYPE taskType_, int taskId_,  Real processedAmount_, Real taskValue_, Real taskRewardLeft_, uint numOfRobotsRequired_) {
	BaseRobot::OnTaskProcessingDone(taskType_, taskId_, processedAmount_, taskValue_, taskRewardLeft_, numOfRobotsRequired_);

	//-- base robot will go scouting when a task is done. Local broadcaster should make sure it can't be immediately recruited to the same task
	if (state == STATE_SCOUTING) {
		timeTillRerecruitmentAllowed = 1000; //each 200 ticks caters for 1m distance travelled
		forbiddenTaskId = taskId_;
	}

}

/**
 * Triggered once when all resource has been unloaded
 */
void LocalBroadcaster::OnUnloadResourceDone() {

	BaseRobot::OnUnloadResourceDone();
	if (targetTaskInfo.isNull) {
		////LOGERR << this->GetId() << "!! DONE UNLOADING, no task in memory " << std::endl;
		SetState(STATE_SCOUTING);
	} else {
		////LOGERR << this->GetId() << "!! DONE UNLOADING, return to task " << std::endl;
		SetState(STATE_GOING_TO_TASK);
	}

}

/**
 * Abandon a task
 */
void LocalBroadcaster::AbandonTask(bool reportEvent_) {
	BaseRobot::AbandonTask();
}

/**
 * Return to nest with resource in cart
 */
void LocalBroadcaster::ReturnToBase() {
	BaseRobot::ReturnToBase();
	if (location == LOCATION_DEPOSIT) {
		BroadcastTask();
	}
}


//====================================== COMMUNICATION


/**
 * Stay near the task deposit and broadcast it to others
 */
void LocalBroadcaster::BroadcastTask() {

	//-- send signals
	rangeAndBearingActuator->ClearData();
	rangeAndBearingActuator->SetData(4, SIGNAL_TASK_INFO_AVAILABLE);
	rangeAndBearingActuator->SetData(5, targetTaskInfo.taskId);
	rangeAndBearingActuator->SetData(1,id);

	////LOGERR << this->GetId() << " BROADCASTING t" << taskInfo.taskId << std::endl;

	//-- generate random order in which packets are read
	const CCI_RangeAndBearingSensor::TReadings& packets = rangeAndBearingSensor->GetReadings();
	int* indeces = GenerateRandomPacketOrder(packets);


	//-- read signals from observers
	CRadians angleToObserver = CRadians(0);
	int toId = -1;
	isCommunicating = false;
	for(int p=0; p < packets.size(); p++) {
		int packetId = indeces[p];

		//-- find out if someone is receiving from the robot's id - if yes, remember to communicate with that robot
		if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_RECEIVING && packets[packetId].Data[1] == id && packets[packetId].Range < parameters.signalDistance) {
			toId = packets[packetId].Data[2];
			bool alreadyCommunicatedWithIt = false;
			//-- beginning of new communication stream, check if already did this with this robot
			for(UInt32 j = 0; j < encounteredObserverIds.size(); j++) {
				if (encounteredObserverIds[j] == toId) {
					alreadyCommunicatedWithIt = true;
					break;
				}
			}

			if (!alreadyCommunicatedWithIt) {
				////////LOGERR << this->GetId() << " communicating to " << toId << " " << packets[packetId].Data[0] << std::endl;
				isCommunicating = true;
				SetWheelSpeeds(0,0);
				angleToObserver = packets[packetId].HorizontalBearing;
			} else {
				////////LOGERR << this->GetId() << " already communicated with " << toId << " " << packets[packetId].Data[0] << std::endl;
				toId = 0;
				isCommunicating = false;
			}
		}

	}
	delete[] indeces;

	//-- broadcast the task info
	if (isCommunicating) {
		SendTaskInfoToRobot(angleToObserver, toId);
	} else {
		if (baseParameters.useOpportunism && timeTillNextScoutAllowed <= 0) {
			////////LOGERR << this->GetId() << isCommunicating << std::endl;
			ReceiveTaskInfo(true);
		}

	}
}

/**
 * Send parts of the waggle dance one by one. In each packet, its bits are:
 * 0 = signal type
 * 1 = id of the robot
 * 2-9 = encoded float value
 */
void LocalBroadcaster::SendTaskInfoToRobot(CRadians angleToObserver_, int toId_) {
	if (isCommunicating) {
		//////LOGERR << this->GetId() << " RECRUITING " << toId_ << std::endl;

		//-- send data
		Real valueToSend;
		/*if (!taskInfoToIdSent) {
			valueToSend =  toId_;
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_TO_ID);
			taskInfoToIdSent = true;
		} else if (!taskInfoAngleToTargetSent) {
			//-- send angle to target, adjusted by angle to the receiver
			valueToSend = (angleToObserver_ - (CRadians::PI_OVER_TWO + targetTaskInfo.relLocation.Angle())).UnsignedNormalize().GetValue();
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_ANGLE_TO_TARGET);
			taskInfoAngleToTargetSent = true;
		} else if (!taskInfoDistanceToTargetSent) {
			//-- send distance to the target
			valueToSend = targetTaskInfo.relLocation.Length();
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_DISTANCE_TO_TARGET);
			taskInfoDistanceToTargetSent = true;
		} else if (!taskInfoTaskIdSent) {
			//-- send distance to the target
			valueToSend = targetTaskInfo.taskId;
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_TASK_ID);
			taskInfoTaskIdSent = true;
			////LOGERR << this->GetId() << "Sending task id " << taskInfo.taskId << " sig type " << SIGNAL_TASK_INFO_TASK_ID << std::endl;
		}*/

		if (!taskInfoToIdSent) {
			valueToSend =  toId_;
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_TO_ID);
			taskInfoToIdSent = true;
		} else if (!taskInfoAngleToTargetSent) {
			//-- send angle to target, adjusted by angle to the receiver
			valueToSend = (angleToObserver_ - (CRadians::PI_OVER_TWO + targetTaskInfo.relLocation.Angle())).UnsignedNormalize().GetValue();
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_ANGLE_TO_TARGET);
			taskInfoAngleToTargetSent = true;
		} else if (!taskInfoDistanceToTargetSent) {
			//-- send distance to the target
			valueToSend = targetTaskInfo.relLocation.Length();
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_DISTANCE_TO_TARGET);
			taskInfoDistanceToTargetSent = true;
		} else if (!taskInfoUtilitySent) {
			//-- send distance to the target
			valueToSend = targetTaskInfo.taskUtility;
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_UTILITY);
			taskInfoUtilitySent = true;
		} else if (!taskInfoRelUtilitySent) {
			//-- send distance to the target
			valueToSend = targetTaskInfo.relTaskUtility;
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_REL_UTILITY);
			taskInfoRelUtilitySent = true;
		} else if (!taskInfoNumOfRobotsRequiredSent) {
			//-- send distance to the target
			valueToSend = targetTaskInfo.numOfRobotsRequired;
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_NUM_OF_ROBOTS_REQUIRED);
			taskInfoNumOfRobotsRequiredSent = true;
		} else if (!taskInfoTaskIdSent) {
			//-- send distance to the target
			valueToSend = targetTaskInfo.taskId;
			rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_TASK_ID);
			taskInfoTaskIdSent = true;
		}

		//-- add the encoded float into the packet data
		UInt8* encodedFloat = (UInt8*)(&valueToSend);
		for (int i=0; i<8; i++) {
			rangeAndBearingActuator->SetData(i+2,encodedFloat[i]);
		}

		//-- check if all information has been sent, in which case memory of what was sent is reset
		if (taskInfoToIdSent && taskInfoAngleToTargetSent && taskInfoDistanceToTargetSent && taskInfoTaskIdSent && taskInfoUtilitySent && taskInfoRelUtilitySent && taskInfoNumOfRobotsRequiredSent) {
			taskInfoToIdSent = false;
			taskInfoAngleToTargetSent = false;
			taskInfoDistanceToTargetSent = false;
			taskInfoTaskIdSent = false;
			taskInfoUtilitySent = false;
			taskInfoRelUtilitySent = false;
			taskInfoNumOfRobotsRequiredSent = false;
			//LOGERR << this->GetId() << " remembering recruit " << toId_ << std::endl;
			encounteredObserverIds.push_back(toId_);
		}


	}
}

/*
 * Find and count how many others are broadcasting about the current task
 */
void LocalBroadcaster::FindOtherTaskBroadcasters() {
	encounteredIdsAtTask.clear();
	//-- generate random order in which packets are read
	const CCI_RangeAndBearingSensor::TReadings& packets = rangeAndBearingSensor->GetReadings();
	int* indeces = GenerateRandomPacketOrder(packets);

	for(int p=0; p < packets.size(); p++) {
		int packetId = indeces[p];
		//if ((packets[packetId].Data[0] >= SIGNAL_TASK_INFO_TO_ID && packets[packetId].Data[0] <= SIGNAL_TASK_INFO_TASK_ID) ||
		if (packets[packetId].Data[4] == SIGNAL_TASK_INFO_AVAILABLE && packets[packetId].Data[5] == targetTaskInfo.taskId ) {
			////LOGERR << this->GetId() << " found another at task " << packets[i].Data[1] << std::endl;
			bool alreadyNoted = false;
			for(UInt32 j = 0; j < encounteredIdsAtTask.size(); j++) {
				if (encounteredIdsAtTask[j] == packets[packetId].Data[1]) {
					alreadyNoted = true;
					break;
				}
			}
			if (!alreadyNoted) {
				////LOGERR << this->GetId() << " encountered " << packets[i].Data[1] << std::endl;
				encounteredIdsAtTask.push_back(packets[packetId].Data[1]);
			}
		}
	}
	delete[] indeces;
}

void LocalBroadcaster::ReceiveTaskInfo(bool adoptWorksiteOnlyIfBetter_) {
	bool sendingAcknowledgementSignalToRecruiter = false;

	//-- generate random order in which packets are read
	const CCI_RangeAndBearingSensor::TReadings& packets = rangeAndBearingSensor->GetReadings();
	int* indeces = GenerateRandomPacketOrder(packets);

	bool alreadyRecruited = false;
	for(int p=0; p<packets.size(); p++) {
		int packetId = indeces[p];

		//-- find out if this is any part of the recruitment signal and if signal is close enough
		/*if (this->GetNumId() == 19) {
			//LOGERR << this->GetId() << " got packed from " << packets[p].Data[1] << " type " << packets[p].Data[0] <<  "d=" << parameters.signalDistance << std::endl;
		}*/
		if (((packets[packetId].Data[0] >= SIGNAL_TASK_INFO_TO_ID && packets[packetId].Data[0] <= SIGNAL_TASK_INFO_TASK_ID) || packets[packetId].Data[4] == SIGNAL_TASK_INFO_AVAILABLE) && packets[packetId].Range < parameters.signalDistance) { //
			//-- if in the middle of receiving from another robot, ignore this
			if ((taskInfoAngleToTargetReceived || taskInfoDistanceToTargetReceived || taskInfoToIdReceived || taskInfoTaskIdReceived) && taskInfoReceivingFromId != packets[packetId].Data[1]) {
				//-- someone else's message, ignore
				////LOGERR << this->GetId() << "!!! Ignore recruitmend from " << packets[i].Data[1] << std::endl;
			} else {
				//-- set who currently receiving from

				sendingAcknowledgementSignalToRecruiter = true;
				taskInfoReceivingFromId = packets[packetId].Data[1];
				////LOGERR << this->GetId() << "Receiving rec from " << taskInfoReceivingFromId << " signal type " << packets[i].Data[0] << std::endl;
				CRadians angleToSender = packets[packetId].HorizontalBearing;

				//-- decode the float
				UInt8* encodedFloat;
				encodedFloat = new UInt8[8];
				for(size_t d = 2; d < 10; ++d) {
					encodedFloat[d-2] = packets[packetId].Data[d];
				}
				Real receivedFloat = *(Real*)encodedFloat;
				delete encodedFloat;

				//-- remember it
				if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_TO_ID) {
					SetWheelSpeeds(0,0);
					//-- reset everything if this message is for a different robot
					if (int(receivedFloat) != this->GetNumId()) {
						if (!alreadyRecruited) { // don't change anything if the robot was already recruited
							rangeAndBearingActuator->ClearData();
							ResetInfoExchange();
						}
					} else {
						taskInfoToIdReceived = true;
						if (this->GetNumId() == 0) {
							//LOGERR << this->GetId() << " received info id from " << taskInfoReceivingFromId << std::endl;
						}
					}
				} else if (taskInfoToIdReceived) {
					SetWheelSpeeds(0,0);
					if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_ANGLE_TO_TARGET) {
						//-- use the x axis of the vector to store angle for now
						advertisedTaskInfo.relLocation.SetX(-receivedFloat);
						taskInfoAngleToTargetReceived = true;
					} else if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_DISTANCE_TO_TARGET) {
						//-- use y axis of the vector to store distance for now
						advertisedTaskInfo.relLocation.SetY(receivedFloat);
						taskInfoDistanceToTargetReceived = true;
					} else if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_UTILITY) {
						advertisedTaskInfo.taskUtility = receivedFloat;
						taskInfoUtilityReceived = true;
					} else if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_REL_UTILITY) {
						advertisedTaskInfo.relTaskUtility = receivedFloat;
						taskInfoRelUtilityReceived = true;
					} else if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_NUM_OF_ROBOTS_REQUIRED) {
						advertisedTaskInfo.numOfRobotsRequired = receivedFloat;
						taskInfoNumOfRobotsRequiredReceived = true;
					} else if (packets[packetId].Data[0] == SIGNAL_TASK_INFO_TASK_ID) {
						//-- if was recruited during neighbourhood search, report a task missing event
						if (neighbourhoodSearchTimeLeft > 0) {
							NewInfoEventReport(InfoEventReport::TYPE_TASK_MISSING, targetTaskInfo.taskId, -1, -1, -1, -1);
						}

						advertisedTaskInfo.taskId = int(receivedFloat);
						taskInfoTaskIdReceived = true;
					}
				}


				//-- check if received everything about the advertised deposit
				if (taskInfoAngleToTargetReceived && taskInfoDistanceToTargetReceived && taskInfoToIdReceived && taskInfoTaskIdReceived && taskInfoUtilityReceived && taskInfoRelUtilityReceived && taskInfoNumOfRobotsRequiredReceived) {
					bool adoptWorksite = true;
					Real debugPrevU = -1;
					if (adoptWorksiteOnlyIfBetter_) {
						//-- compare the advertised and current worksite
						if (!targetTaskInfo.isNull) {
							//if (this->GetNumId() == 21 || this->GetNumId() == 8) {
							debugPrevU = targetTaskInfo.taskUtility;
							//LOGERR << this->GetId() << " rememebered U " << targetTaskInfo.taskUtility << " advertised " << advertisedTaskInfo.taskUtility <<  std::endl;
							if (advertisedTaskInfo.taskUtility - targetTaskInfo.taskUtility <= baseParameters.opportunismThreshold) {
								//-- the advertised is not sufficiently better than the current worksite
								adoptWorksite = false;
							} else {
								//LOGERR << this->GetId() << " !!!! OPPORTUNISM " << std::endl;
							}
						} else {
							////////LOGERR << this->GetId() << " robot doesn't have a worksite in memory " << std::endl;
						}
					}
					if (adoptWorksite) {
						CRadians gamma = CRadians(advertisedTaskInfo.relLocation.GetX()) + packets[packetId].HorizontalBearing - CRadians::PI_OVER_TWO;
						CVector2 translatedPos = -CVector2(packets[packetId].Range/100 * cos(packets[packetId].HorizontalBearing.GetValue()) + advertisedTaskInfo.relLocation.GetY() * cos(gamma.GetValue()),
													packets[packetId].Range/100 * sin(packets[packetId].HorizontalBearing.GetValue()) + advertisedTaskInfo.relLocation.GetY() * sin(gamma.GetValue()));

						//-- make the deposit current target and go foraging for it
						/*if (this->GetNumId() == 19) {
							//LOGERR << this->GetId() << "----- recruited to " << taskInfo.taskId << std::endl;
						}*/

						timeTillNextScoutAllowed = DELAY_BETWEEN_SCOUTINGS;

						if (forbiddenTaskId != advertisedTaskInfo.taskId || baseParameters.useRelU == false) {
							alreadyRecruited = true;
							targetTaskInfo.isNull = false;
							targetTaskInfo.relLocation = CVector2(translatedPos.GetX(), translatedPos.GetY());
							targetTaskInfo.taskUtility = advertisedTaskInfo.taskUtility;
							targetTaskInfo.relTaskUtility = advertisedTaskInfo.relTaskUtility;
							targetTaskInfo.numOfRobotsRequired = advertisedTaskInfo.numOfRobotsRequired;
							targetTaskInfo.taskId = advertisedTaskInfo.taskId;
							targetTaskInfo.infoOriginatorId = taskInfoReceivingFromId;

							NewInfoEventReport(InfoEventReport::TYPE_ROBOT_RECRUITED,targetTaskInfo.taskId,-1,-1,-1,targetTaskInfo.infoOriginatorId);

							ResetInfoExchange();

							SetState(STATE_GOING_TO_TASK);

							//LOGERR << this->GetId() << " recruited to task " << targetTaskInfo.taskId << " U " << targetTaskInfo.taskUtility << std::endl;
						}
					} else {
						ResetInfoExchange(true);
					}
				}
			}
		}
	}
	delete[] indeces;

	if (sendingAcknowledgementSignalToRecruiter) {
		//-- send a signal to the robot specifying someone is receiving
		rangeAndBearingActuator->ClearData();
		rangeAndBearingActuator->SetData(0, SIGNAL_TASK_INFO_RECEIVING);
		rangeAndBearingActuator->SetData(1,taskInfoReceivingFromId);
		rangeAndBearingActuator->SetData(2,this->GetNumId());
	} /*else if (taskInfoReceivingFromId >= 0) { //only reset when abandoning listening, we don't want to clear range and bearing data on every loop because other signals might be being sent
		taskInfoToIdReceived = false;
		taskInfoAngleToTargetReceived = false;
		taskInfoDistanceToTargetReceived = false;
		taskInfoTaskIdReceived = false;
		rangeAndBearingActuator->SetData(0, SIGNAL_NONE);
	}*/
}





/****************************************/
/****************************************/

/*
 * This statement notifies ARGoS of the existence of the controller.
 * It binds the class passed as first argument to the string passed as
 * second argument.
 * The string is then usable in the XML configuration file to refer to
 * this controller.
 * When ARGoS reads that string in the XML file, it knows which controller
 * class to instantiate.
 * See also the XML configuration files for an example of how this is used.
 */
REGISTER_CONTROLLER(LocalBroadcaster, "localBroadcasterController")


