%% IMAGE BASED INERTIAL IMPACT (IBII) TEST - Data Processing
% Authors: Lloyd Fletcher, Jared Van-Blitterswyk
% PhotoDyn Group, University of Southampton
% Date Created: 12/12/2017
% Date Edited: 13/6/2018
%
% Takes input images processes them with the grid method to obtain
% displacement fields. Further kinematic fields are derived from the
% displacement fields (acceleration, strain). Kinematic fields are then
% used to identify material properties including: stiffness and strength
%
% This code uses the grid processing tool box that can be found at:
% www.thegridMethodOpts.net, developed by Grediace

clc
clear all
close all

fprintf('--------------------------------------------------------------\n')
fprintf('IMAGE-BASED INERTIAL IMPACT (IBII): Data Processing v2.3\n')
fprintf('--------------------------------------------------------------\n')

%% INITIALISE - Add path for processing functions
% Add the path for the grid method code and other useful functions:
funcPath = [pwd,'\Functions\'];

% If the default path is not found we should find it
if exist(funcPath,'file') ~= 7
    hWarn = warndlg('Folder for global processing functions not found.','Function folder not found');
    waitfor(hWarn);
    funcPath = uigetdir(pwd,'Locate Processing Function Folder');
end
addpath(funcPath);
addpath([funcPath,'GridMethodToolbox\']);
addpath([funcPath,'Export_Fig\'])

%% LOAD RAW DATA FILES - Raw .tiff images
fprintf('Loading reference image from the selected test data folder.\n')
hardCodePath = false;
if ~hardCodePath
    [imageFile,imagePath] = uigetfile({'*.*','All Files'},'Select the first image in the sequence');
else   
    imageFile = 'Camera_15_21_38_001.tiff';
    imagePath = 'E:\Matlab_WorkingDirectory\1_ImpactTest_ProcessingToolbox_2018\Z_CFIP90_CodeTestData\';    
end

%% INITIALISE - Load Processing Parameter Data Structures
initPath = imagePath;
initFile = 'processingParameters.mat';
if exist([initPath,initFile],'file') ~= 2
    hWarn = warndlg('Processing parameter file does not exist.','Processing parameters not found');
    waitfor(hWarn);
    [initFile,initPath,~] = uigetfile('*.mat','Locate processing parameter file');
end
% Load the processing parameters from file
load([initPath,initFile])

%% PRE-PROCESSING - Grid Method Processing of Raw Images
% Process the raw tiff images using the grid method code developed by
% Grediac et al.
fprintf('\n--------------------------------------------------------------\n')
fprintf('GRID METHOD PROCESSING\n')
fprintf('--------------------------------------------------------------\n')

%--------------------------------------------------------------------------
% GRID IMAGE PROCESSING

% Check if the images have already been processed otherwise process the data
gridDataSavePath = imagePath;
gridDataFile = 'GridMethod_ProcessedData.mat';

fprintf('Checking for existing processed data file.\n')
processGridImages = true;
if exist([gridDataSavePath,gridDataFile],'file') == 2
    if gridMethodOpts.autoLoadProccessedDataFile
        processGridImages = false;
    else   
        choice = questdlg('Processed grid data file found, process again?', ...
        'Process Grid Images', 'Yes','No','No');
        switch choice
            case 'Yes'
                processGridImages = true;
            case 'No'
                processGridImages = false;
        end
    end
end

if processGridImages
    fprintf('Processing images using the grid method toolbox.\n')
    % Process the image squence with the grid method toolbox
    [grid,pos,disp] = func_gridMethodImageProcessing(imagePath,imageFile,...
        grid,gridMethodOpts,imageNoise);    
    
    % Save the data to file so a reprocess is not necessary
    fprintf('Saving processed grid data to: %s.\n',gridDataFile)
    save([gridDataSavePath,gridDataFile],'grid','pos','disp') 

    fprintf('Grid Method Processing Complete.\n')
else
    fprintf('Loading processed grid data from: %s.\n',gridDataFile)
        load([gridDataSavePath,gridDataFile]);
end

%--------------------------------------------------------------------------
% Update Geometry and Number of Frames Based on Displacement Matrix Size
fprintf('Updating geometry variables based on the size of displacement field matrices.\n')

% Check that the nominal specimen dimensions and the dimensions of the
% selected grid window are the same, if they are not update them
if isfield(grid,'asymmPitch')
if grid.asymmPitch
    checkLength = grid.mPerPxX*size(disp.x,2);
    if checkLength ~= specimen.length
        specimen.length = checkLength;
        grid.length = checkLength;
    end
    checkHeight = grid.mPerPxY*size(disp.x,1);
    if checkHeight ~= specimen.height
        specimen.height = checkHeight;
        grid.height = checkHeight;
    end
else
    checkLength = grid.mPerPx*size(disp.x,2);
    if checkLength ~= specimen.length
        specimen.length = checkLength;
        grid.length = checkLength;
    end
    checkHeight = grid.mPerPx*size(disp.x,1);
    if checkHeight ~= specimen.height
        specimen.height = checkHeight;
        grid.height = checkHeight;
    end
end
else
    checkLength = grid.mPerPx*size(disp.x,2);
    if checkLength ~= specimen.length
        specimen.length = checkLength;
        grid.length = checkLength;
    end
    checkHeight = grid.mPerPx*size(disp.x,1);
    if checkHeight ~= specimen.height
        specimen.height = checkHeight;
        grid.height = checkHeight;
    end
end

% Create the time vector based on the number of frames in the disp struct
time.numFrames = size(disp.x,3);
time.vec = 0:time.step:(size(disp.x,3)-1)*time.step;

% Currently the rotations are unused so remove them to save RAM
disp = rmfield(disp,'rot');

%--------------------------------------------------------------------------
% Update the geometry vars in the slice data structure for orthoAngle model
if strcmp('orthotropicAngle',globalOpts.matModel)
    slice.length = abs(specimen.height/sind(slice.angle));
    slice.lengthX = abs(slice.length*cosd(slice.angle));
    slice.xMax = specimen.length - slice.lengthX;
    slice.xMaxInd = sum(pos.x < slice.xMax);
end

%% POST-PROCESSING - Smoothing and Kinematic Field Derivation
% Smooth the displacement data and then calculate acceleration and strains
% Extrapolate the data to account for the missing pitch on the edges
fprintf('\n--------------------------------------------------------------\n')
fprintf('POST PROCESSING: Smoothing and Kinematic Field Calculation\n')
fprintf('--------------------------------------------------------------\n')

%--------------------------------------------------------------------------
% Load the Reference Image and Determine Where the Free Edge is
% TODO: setup a way to hardcode the free edge location
fprintf('Obtaining and setting the free edge location.\n')
[freeEdge,specimen,disp] = func_getFreeEdge(globalOpts.hardCodeFreeEdge,...
    imagePath,imageFile,specimen,disp);

if ~globalOpts.useOldEdgeMethod % Use new fitted linear extrap procesurd
    %--------------------------------------------------------------------------
    % Smooth and Calculate Strain
    fprintf('Calculating strain from the displacement fields.\n')
    [disp,strain,strainRate] = func_smoothCalcStrain(globalOpts,pos,time,...
        grid,disp,smoothingOpts,extrapOpts);

    %--------------------------------------------------------------------------
    % Smooth and Calculate Acceleration
    fprintf('Calculating acceleration from the displacement fields.\n')
    [disp,~,accel] = func_smoothCalcAccel(pos,time,grid,disp,smoothingOpts,...
        extrapOpts,diffOpts);

else % Use the old two point linear extrap method
    %--------------------------------------------------------------------------
    % Smooth and Calculate Strain/Acceleration
    fprintf('Calculating acceleration and strain from the displacement fields.\n')
    [~,~,accel,strain,strainRate] = func_smoothCalcAccelStrain(...
        pos,time,material,grid,disp,smoothingOpts,extrapOpts,true);
end

%--------------------------------------------------------------------------
% Rotate Kinematic Fields to Material Co-ords
% NOTE: only required for orthotropicAngle material model
if strcmp('orthotropicAngle',globalOpts.matModel)
    fprintf('Rotating kinematic fields into the material co-ords, angle = %0.2f.\n',material.rotAngle)
    
    fprintf('Rotating the acceleration fields.\n')
    [accel.mat11,accel.mat22] = func_rotateVector2D(accel.x,accel.y,material.rotAngle);
    
    fprintf('Rotating strain fields to the material co-ord system.\n')
    [strain.mat11,strain.mat22,strain.mat12] = func_rotateStrain2D(...
        strain.x,strain.y,strain.s,material.rotAngle);
    
    fprintf('Rotating strain rate fields to the material co-ord system.\n')
    [strainRate.mat11,strainRate.mat22,strainRate.mat12] = func_rotateStrain2D(...
        strainRate.x,strainRate.y,strainRate.s,material.rotAngle);
    
    fprintf('Interpolating average strain over each angled slice.\n')
    strain = func_calcRotatedFieldAvgAlongSlice(slice,material,pos,strain);
    
    fprintf('Interpolating average strain rate over each angled slice.\n')
    strainRate = func_calcRotatedFieldAvgAlongSlice(slice,material,pos,strainRate);
end

%% POST-PROCESSING - Stiffness Identification
% Use the smoothed data and the stress gauge approach/VFM to identify the
% in-plane stiffness components. 
fprintf('\n--------------------------------------------------------------\n')
fprintf('POST PROCESSING: Stiffness Identification\n')
fprintf('--------------------------------------------------------------\n')

%--------------------------------------------------------------------------
% STRESS GAUGE: CALCULATION
fprintf('Calculating average stress with the stress-gauge equation.\n')
% Standard Stress Gauge equation calculation
[stress.xAvg,~] = func_stressGaugeProcess(material,time,pos,accel);

% First moment of the stress-gauge
fprintf('Calculating the average stress with the linear stress-gauge equation.\n')
stress = func_linearStressGaugeProcess(specimen,material,pos,accel,stress);
    
if strcmp('orthotropicAngle',globalOpts.matModel)
    fprintf('Calculating average stress with the angled stress-gauge at an angle of: %.2f.\n',material.rotAngle)
    % Calculate the angled stress-gauge over each angled slice
    [stress.slice22Avg,stress.slice12Avg] = func_stressGaugeProcess_Ang(slice,material,specimen,time,pos,accel);
end

%--------------------------------------------------------------------------
% VFM: MANUAL
% TODO: crop the fields for the VFM taking away half a smoothing kernal on
% the impact edge
fprintf('Identifying stiffnesses with manual virtual fields.\n')
fprintf('Selected material model is: %s.\n',globalOpts.matModel) 

if strcmp('isotropic',globalOpts.matModel)
    % Create the virtual fields
    VFs = func_VFDynInitManIsoLinElas(VFOpts,pos,accel,strain);
    % Use the virtual fields for stiffness identification
    identStiffVFMan = func_VFDynManIsoLinElas(VFOpts,pos,time,material,...
        VFs,strain,accel);

    % Calculate the median to work out the 'average' identified value
    identStiffVFMan.QxxAvgOverT = nanmedian(identStiffVFMan.QxxVsT(VFOpts.avgQVsTRange));
    identStiffVFMan.QxyAvgOverT = nanmedian(identStiffVFMan.QxyVsT(VFOpts.avgQVsTRange));
    identStiffVFMan.ExxAvgOverT = nanmedian(identStiffVFMan.ExxVsT(VFOpts.avgQVsTRange));
    identStiffVFMan.NuxyAvgOverT = nanmedian(identStiffVFMan.NuxyVsT(VFOpts.avgQVsTRange));

elseif strcmp('orthotropicReduced',globalOpts.matModel)
    % Define manual virtual fields
    VFOpts.ux_star = @(x) x - specimen.length;
    VFOpts.epsx_star = @(x) 1;
    % Process kinematic data with manual virtual fields
    identStiffVFMan.ExxVsT = func_VFDynManReducedLinElas(VFOpts,pos,material,accel,strain);
    
    % Calculate an average stiffness over the specified time range
    identStiffVFMan.ExxAvgOverT = nanmedian(identStiffVFMan.ExxVsT(VFOpts.avgQVsTRange));    
    
    % For this case Qxx approx Exx
    identStiffVFMan.QxxVsT = identStiffVFMan.ExxVsT;
    identStiffVFMan.QxxAvgOverT = identStiffVFMan.ExxAvgOverT;
elseif strcmp('orthotropic',globalOpts.matModel)
    fprintf('WARNING: orthtropic manual virtual fields have not been implemented.\n')
    identStiffVFMan = nan;
    
elseif strcmp('orthotropicAngle',globalOpts.matModel)
    fprintf('WARNING: angled orthtropic manual virtual fields have not been implemented.\n')
    identStiffVFMan = nan;
else
    fprintf('WARNING: specifed material model not recognised.\n')
    identStiffVFMan = nan;
end 

%--------------------------------------------------------------------------
% VFM: PIECE-WISE SPECIAL OPTIMISED
if globalOpts.processOptVF
    fprintf('Identifying stiffnesses with piece-wise optimised virtual fields.\n') 
    fprintf('Selected material model is: %s.\n',globalOpts.matModel) 
    
    if strcmp('isotropic',globalOpts.matModel)
        % Use isotropic virtual fields to get Qxx and Qxy
        fprintf('Identifying Qxx and Qxy with optimised virtual fields.\n')
        [identStiffVFOpt,VFOptDiag] = func_VFDynPWSpecOptIsoLinElas(VFOpts,...
            pos,specimen,material,accel,strain);

        % Calculate the median to work out the 'average' identified value
        identStiffVFOpt.QxxAvgOverT = nanmedian(identStiffVFOpt.QxxVsT(VFOpts.avgQVsTRange));
        identStiffVFOpt.QxyAvgOverT = nanmedian(identStiffVFOpt.QxyVsT(VFOpts.avgQVsTRange));
        identStiffVFOpt.ExxAvgOverT = nanmedian(identStiffVFOpt.ExxVsT(VFOpts.avgQVsTRange));
        identStiffVFOpt.NuxyAvgOverT = nanmedian(identStiffVFOpt.NuxyVsT(VFOpts.avgQVsTRange));

    elseif strcmp('orthotropicReduced',globalOpts.matModel)
        % Use reduced optimised virtual fields to obtain Qxx
        fprintf('Identifying Qxx with reduced optimised virtual fields.\n')
        identStiffVFOpt.ExxVsT = func_VFDynPWSpecOptReducedLinElas(VFOpts,...
            pos, strain, accel, specimen, material);

        % Calculate an average stiffness over the specified time range
        identStiffVFOpt.ExxAvgOverT = nanmedian(identStiffVFOpt.ExxVsT(VFOpts.avgQVsTRange));
        
        % For this case Qxx is approx Exx
        identStiffVFOpt.QxxVsT = identStiffVFOpt.ExxVsT;
        identStiffVFOpt.QxxAvgOverT = identStiffVFOpt.ExxAvgOverT;
        
    elseif strcmp('orthotropic',globalOpts.matModel)
        fprintf('Identifying all orthotropic stiffness components with optimised virtual fields.\n')
        fprintf('NOTE: assumes angle of orthotropy is aligned with global co-ords.\n')
        identStiffVFOpt = func_VFDynPWSpecOptOrthoLinElas(VFOpts,...
            pos,specimen,material,accel,strain);
        
    elseif strcmp('orthotropicAngle',globalOpts.matModel)
        fprintf('Identifying all orthotropic stiffness components with optimised virtual fields.\n')
        fprintf('Specified angle of orthotropy is: %0.2f degrees.\n',material.rotAngle)
        identStiffVFOpt = func_VFDynPWSpecOptOrthoLinElas_Ang(VFOpts,...
            pos,specimen,material,accel,strain,material.rotAngle);
        
    else
        fprintf('WARNING: specifed material model not recognised.\n')
        identStiffVFOpt = nan;
    end
else
    fprintf('Optimised virtual fields processing disabled.\n')
    identStiffVFOpt = nan;
end

%--------------------------------------------------------------------------
% STRESS GAUGE: find stiffness(es) by fitting the stress-strain curves
fprintf('Identifying stiffness(es) by fitting the stress-strain curves.\n')
    
if strcmp('orthotropicReduced',globalOpts.matModel)
    fprintf('Linearly fitting the stress and strain (e_xx) components to obtain Qxx.\n')
    % Fit the stress strain curves to obtain Qxx/Exx
    % identStiffSG = func_identQxxFromStressStrainCurve(stressGaugeOpts,stress,strain);
    [identStiffSG.QxxVsL,identStiffSG.QxxLinFitCoeffs]...
        = func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stress.xAvg,strain.xAvg);
    
    % Set the range over which the average stiffness is identified
    stressGaugeOpts.avgQVsLRange = round(stressGaugeOpts.avgQVsLRangePc(1)*length(pos.x))...
        :round(stressGaugeOpts.avgQVsLRangePc(2)*length(pos.x));
    % Calculate Identified Averages
    identStiffSG.QxxAvgOverL = nanmedian(identStiffSG.QxxVsL(stressGaugeOpts.avgQVsLRange));
    identStiffSG.ExxVsL = identStiffSG.QxxVsL;
    identStiffSG.ExxAvgOverL = nanmedian(identStiffSG.ExxVsL(stressGaugeOpts.avgQVsLRange));
    
elseif strcmp('orthotropicAngle',globalOpts.matModel)
    % Set the range over which the average stiffness is identified
    stressGaugeOpts.avgQVsLRange = round(stressGaugeOpts.avgQVsLRangePc(1)*slice.xMaxInd)...
        :round(stressGaugeOpts.avgQVsLRangePc(2)*slice.xMaxInd);
    
    fprintf('Linearly fitting the 22 stress and strain components to obtain Q22.\n')
    [identStiffSG.Q22VsL,identStiffSG.Q22LinFitCoeffs]...
        = func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stress.slice22Avg,strain.slice22Avg);
    identStiffSG.Q22AvgOverL = nanmean(identStiffSG.Q22VsL(stressGaugeOpts.avgQVsLRange));
    
    fprintf('Linearly fitting the 12 stress and strain components to obtain Q66.\n')
    [identStiffSG.Q66VsL,identStiffSG.Q66LinFitCoeffs] = ...
        func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stress.slice12Avg,strain.slice12Avg);
    identStiffSG.Q66AvgOverL = nanmean(identStiffSG.Q66VsL(stressGaugeOpts.avgQVsLRange));
else
    fprintf('Linearly fitting the stress and strain (e_xx + nu_xy*e_yy) components to obtain Qxx.\n')
    % Calculate the axial strain including the poisson effect
    if strcmp('VFOpt',stressGaugeOpts.strainCalcNuxy)
        identStiffSG.strainCalcNuxy = identStiffVFOpt.NuxyAvgOverT;
    elseif strcmp('VFMan',stressGaugeOpts.strainCalcNuxy)
        identStiffSG.strainCalcNuxy = identStiffVFMan.NuxyAvgOverT;
    else % Assume the QS value
        identStiffSG.strainCalcNuxy = material.nuxy;
    end
    strain.xnyAvg = strain.xAvg + identStiffSG.strainCalcNuxy*strain.yAvg;
    
    % Fit the stress strain curves to obtain Qxx
    [identStiffSG.QxxVsL,identStiffSG.QxxLinFitCoeffs] = ...
        func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stress.xAvg,strain.xnyAvg);
    
    % Set the range over which the average stiffness is identified
    stressGaugeOpts.avgQVsLRange = round(stressGaugeOpts.avgQVsLRangePc(1)*length(pos.x))...
        :round(stressGaugeOpts.avgQVsLRangePc(2)*length(pos.x));
    % Calculate Identified Averages
    identStiffSG.QxxAvgOverL = nanmedian(identStiffSG.QxxVsL(stressGaugeOpts.avgQVsLRange));
    identStiffSG.ExxVsL = identStiffSG.QxxVsL.*(1-identStiffSG.strainCalcNuxy^2);
    identStiffSG.ExxAvgOverL = nanmedian(identStiffSG.ExxVsL(stressGaugeOpts.avgQVsLRange));
end

%--------------------------------------------------------------------------
% Display the identified stiffnesses
fprintf('Identified stiffness values:\n')
IdentifiedStiffness_SG = identStiffSG
IdentifiedStiffness_VFMan =identStiffVFMan
IdentifiedStiffness_VFOpt =identStiffVFOpt

%% POST-PROCESSING - Diagnostic Figures and Image Sequences
fprintf('\n--------------------------------------------------------------\n')
fprintf('POST PROCESSING: Diagnostic Image Sequences\n')
fprintf('--------------------------------------------------------------\n')
plotProps = func_initPlotPropsStruct(plotParams.formatType);

%--------------------------------------------------------------------------
% Create Video Sequences of Disp, Strain, Accel and Strain Rate
% NOTE: these images are required for the crack location identification
if strcmp(globalOpts.plotImageSeqs,'prompt') || strcmp(globalOpts.plotImageSeqs,'yes')
    fprintf('Plotting image sequence of disp, accel, strain and strain rate.\n')
    fprintf('NOTE: image sequence used to identify crack location for strength identification.\n')
    func_plotAllFullFieldVideos(globalOpts,plotParams,imagePath,pos,time,disp,strain,strainRate,accel);
else
    fprintf('Image sequence plotting disabled.\n')
end

%--------------------------------------------------------------------------
% Plot Diagnostic Figures of Identified Stiffnesses
savePath = [imagePath,'ProcessedData_DiagnosticFigs\'];
if strcmp(globalOpts.plotImageSeqs,'prompt')
    % Check if the save path exits, if it does ask the user if it
    plotDiagFigs = func_checkIfPathExistsDlg(savePath,...
        'Folder for diagnostic figures exists, plot again?','Plot Diagnostics?');
elseif strcmp(globalOpts.plotImageSeqs,'yes')
    plotDiagFigs = true;
    if exist(savePath,'file') ~= 7
        mkdir(savePath);
    end
else
    plotDiagFigs = false; 
end

if plotDiagFigs
    fprintf('Ploting diagnostic figures.\n')
    
    % Plot field averages over time on a single plot
    fprintf('\tPlotting field averages over time.\n')
    plotParams.Rows = 2;
    plotParams.Cols = 2;
    plotParams.tRange = 1:time.numFrames;
    hf = func_plotFieldAvgVsTime(plotParams,time,pos,disp,accel,strain,strainRate);
    saveFile = [savePath,'\','FieldAveragesVsTime'];
    print(hf,saveFile,plotProps.format,plotProps.saveRes)
    saveas(hf,saveFile,'fig')
    
    %----------------------------------------------------------------------
    % STRESS-GAUGE: Plot stress-strain curves at several locations
    fprintf('\tPlotting stress-strain curves.\n')
    plotParams.Rows = 2;
    plotParams.Cols = 4;
    plotParams.tRange = time.cutFrames+2:time.numFrames-time.cutFrames-2;
    plotParams.locXPcVec = [0.2,0.3,0.4,0.5,0.6,0.7,0.8,0.9];
    plotParams.locXInd = round(length(pos.x)*plotParams.locXPcVec);
    
    if strcmp('orthotropicReduced',globalOpts.matModel)
        labelStrs.y = 'Stress, $\overline{\sigma_{xx}}$ (MPa)';
        labelStrs.x = 'Strain, $\overline{\epsilon_{xx}}$ (mm/m)';
        hf = func_plotStressStrainCurvesGeneric(labelStrs,plotParams,pos,...
            stress.xAvg,strain.xAvg);
        saveFile = [savePath,'\','SG_SSCurves_E22'];
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')
        
    elseif strcmp('orthotropicAngle',globalOpts.matModel)
        plotParams.locXInd = round(slice.xMaxInd*plotParams.locXPcVec);
        labelStrs.y = 'Stress, $\overline{\sigma_{22}}$ (MPa)';
        labelStrs.x = 'Strain, $\overline{\epsilon_{22}}$ (mm/m)';
        hf = func_plotStressStrainCurvesGeneric(labelStrs,plotParams,pos,...
            stress.slice22Avg,strain.slice22Avg);
        saveFile = [savePath,'\','SG_SSCurves_Q22'];
        %export_fig(saveFile,'-eps','-pdf','-c[Inf,Inf,Inf,Inf]',hf)
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')
        
        labelStrs.y = 'Stress, $\overline{\sigma_{12}}$ (MPa)';
        labelStrs.x = 'Strain, $\overline{\epsilon_{12}}$ (mm/m)';
        hf = func_plotStressStrainCurvesGeneric(labelStrs,plotParams,pos,...
            stress.slice12Avg,strain.slice12Avg);
        saveFile = [savePath,'\','SG_SSCurves_Q66'];
        %export_fig(saveFile,'-eps','-pdf','-c[Inf,Inf,Inf,Inf]',hf)
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')
    else
        labelStrs.y = 'Stress, $\overline{\sigma_{xx}}$ (MPa)';
        labelStrs.x = 'Strain, $\overline{\epsilon_{xx}+\nu \epsilon_{yy}}$ (mm/m)';
        hf = func_plotStressStrainCurvesGeneric(labelStrs,plotParams,pos,...
            stress.xAvg,strain.xnyAvg);
        saveFile = [savePath,'\','SG_SSCurves'];
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')
    end
    
    %----------------------------------------------------------------------
    % STRESS-GAUGE: Plot identified Stiffness vs Length
    fprintf('\tPlotting identified stiffness over the length of the specimen.\n')
    %stiffPlotOpts.formatType = plotParams.formatType;
    stiffPlotOpts.formatType = plotParams.formatType;
    stiffPlotOpts.plotTarg = true;
    stiffPlotOpts.targPc = 0.1;
    stiffPlotOpts.rangePc = 10*stiffPlotOpts.targPc;
    stiffPlotOpts.specifyLRange = true;
    %stiffPlotOpts.indLRange = 2*extrapOpts.strainPx:length(pos.x)-2*extrapOpts.strainPx;
    stiffPlotOpts.indLRange = 1:length(pos.x);
    stiffPlotOpts.unitConv = 10^-9; % normally stiffness is in GPa

    if strcmp('orthotropicReduced',globalOpts.matModel)
        stiffPlotOpts.targVal = material.Exx*10^-9;
        labelStrs.y = '$E_{22}$ (GPa)';
        labelStrs.x = 'Position, $x$ (mm)';
        hf = func_plotStiffnessVsLength(labelStrs,stiffPlotOpts,pos,...
            identStiffSG.ExxVsL,identStiffSG.ExxAvgOverL);
        saveFile = [savePath,'\','SG_StiffVsL_E22'];
        %export_fig(saveFile,'-eps','-pdf','-c[Inf,Inf,Inf,Inf]',hf)
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')

    elseif strcmp('orthotropicAngle',globalOpts.matModel)
        stiffPlotOpts.specifyLRange = false;
        
        stiffPlotOpts.targVal = material.Eyy*10^-9;
        stiffPlotOpts.plotTarg = false;
        labelStrs.y = '$E_{22}$ (GPa)';
        labelStrs.x = 'Position, $x$ (mm)';
        hf = func_plotStiffnessVsLength(labelStrs,stiffPlotOpts,pos,...
            identStiffSG.Q22VsL,identStiffSG.Q22AvgOverL);
        saveFile = [savePath,'\','SG_StiffVsL_E22'];
        %export_fig(saveFile,'-eps','-pdf','-c[Inf,Inf,Inf,Inf]',hf)
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')
        
        stiffPlotOpts.targVal = material.Gxy*10^-9;
        stiffPlotOpts.plotTarg = false;
        labelStrs.y = '$G_{12}$ (GPa)';
        labelStrs.x = 'Position, $x$ (mm)';
        hf = func_plotStiffnessVsLength(labelStrs,stiffPlotOpts,pos,...
            identStiffSG.Q66VsL,identStiffSG.Q66AvgOverL);
        saveFile = [savePath,'\','SG_StiffVsL_G12'];
        %export_fig(saveFile,'-eps','-pdf','-c[Inf,Inf,Inf,Inf]',hf)
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')
    else
        stiffPlotOpts.targVal = material.Qxx*10^-9;
        labelStrs.y = 'Stiffness, $Q_{xx}$ (GPa)';
        labelStrs.x = 'Position, $x$ (mm)';
        hf = func_plotStiffnessVsLength(labelStrs,stiffPlotOpts,pos,...
            identStiffSG.QxxVsL,identStiffSG.QxxAvgOverL);
        saveFile = [savePath,'\','SG_StiffVsL_Qxx'];
        %export_fig(saveFile,'-eps','-pdf','-c[Inf,Inf,Inf,Inf]',hf)
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')
        
        stiffPlotOpts.targVal = material.Exx*10^-9;
        labelStrs.y = '$E$ (GPa)';
        labelStrs.x = 'Position, $x$ (mm)';
        hf = func_plotStiffnessVsLength(labelStrs,stiffPlotOpts,pos,...
            identStiffSG.ExxVsL,identStiffSG.ExxAvgOverL);
        saveFile = [savePath,'\','SG_StiffVsL_Exx'];
        %export_fig(saveFile,'-eps','-pdf','-c[Inf,Inf,Inf,Inf]',hf)
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')
    end

    %----------------------------------------------------------------------
    % MANUAL VF: Stiffness diagnostic vs time
    labelStrs.x = 'Time $t$, ($\mu s$)';
    stiffPlotOpts.formatType = 'article';
    stiffPlotOpts.targPc = 0.1;
    stiffPlotOpts.rangePc = 10*stiffPlotOpts.targPc;
    stiffPlotOpts.tRange = 5:time.numFrames;
    stiffPlotOpts.specifyAxisLims = false;
    stiffPlotOpts.axisLims = [0,time.vec(end)*10^6,0,0.5]; % if the axis limit is being specified it is for nu_xy
    stiffPlotOpts.unitConv = 10^-9; % normally stiffness is in GPa
        
    if strcmp('isotropic',globalOpts.matModel)  
        % Plot identified Qxx vs time
        labelStrs.y = 'Stiffness $Q_{xx}$, (GPa)';
        stiffPlotOpts.targVal = material.Qxx*10^-9;
        hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
            identStiffVFMan.QxxVsT,identStiffVFMan.QxxAvgOverT);
        saveFile = [savePath,'\','VFMan_StiffVsT_Qxx'];
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')

        % Plot identified Qxx vs time
        labelStrs.y = 'Stiffness $Q_{xy}$, (GPa)';
        stiffPlotOpts.targVal = material.Qxy*10^-9;
        hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
            identStiffVFMan.QxyVsT,identStiffVFMan.QxyAvgOverT);
        saveFile = [savePath,'\','VFMan_StiffVsT_Qxy'];
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')

         % Plot identified Exx vs time
        labelStrs.y = '$E$, (GPa)';
        stiffPlotOpts.targVal = material.Exx*10^-9;
        hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
            identStiffVFMan.ExxVsT,identStiffVFMan.ExxAvgOverT);
        saveFile = [savePath,'\','VFMan_StiffVsT_E'];
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')

        % Plot identified Nuxy vs length
        labelStrs.y = '$\nu_{xy}$, (GPa)';
        stiffPlotOpts.unitConv = 1;
        stiffPlotOpts.specifyAxisLims = true;
        stiffPlotOpts.targVal = material.nuxy;
        hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
            identStiffVFMan.NuxyVsT,identStiffVFMan.NuxyAvgOverT);
        saveFile = [savePath,'\','VFMan_StiffVsT_Nu'];
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')
        
        stiffPlotOpts.unitConv = 10^-9;
        stiffPlotOpts.specifyAxisLims = false;
        
    elseif strcmp('orthotropicReduced',globalOpts.matModel)
        % Plot identified Qxx vs time
        labelStrs.y = 'Stiffness $Q_{xx}$, (GPa)';
        stiffPlotOpts.targVal = material.Qxx*10^-9;
        hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
            identStiffVFMan.QxxVsT,identStiffVFMan.QxxAvgOverT);
        saveFile = [savePath,'\','VFMan_StiffVsT_Qxx'];
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')

        % Plot identified Exx vs time
        labelStrs.y = 'Stiffness $E_{xx}$, (GPa)';
        stiffPlotOpts.targVal = material.Exx*10^-9;
        hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
            identStiffVFMan.ExxVsT,identStiffVFMan.ExxAvgOverT);
        saveFile = [savePath,'\','VFMan_StiffVsT_Exx'];
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')
        
    elseif strcmp('orthotropic',globalOpts.matModel) || strcmp('orthotropicAngle',globalOpts.matModel) 
        fprintf('\tWARNING: Manual orthotropic VFs are not implemented.\n')
    end
    
    %----------------------------------------------------------------------
    % OPTIMISED VF: Stiffness diagnostic vs time
    if globalOpts.processOptVF
        fprintf('\tPlotting identified stiffness components over time with optimised virtual fields.\n')
        labelStrs.x = 'Time $t$, ($\mu s$)';
        stiffPlotOpts.formatType = 'article';
        stiffPlotOpts.targPc = 0.1;
        stiffPlotOpts.rangePc = 10*stiffPlotOpts.targPc;
        stiffPlotOpts.tRange = 5:time.numFrames;
        stiffPlotOpts.specifyAxisLims = false;
        stiffPlotOpts.axisLims = [0,time.vec(end)*10^6,0,0.5];
        stiffPlotOpts.unitConv = 10^-9; % normally stiffness is in GPa
        
        if strcmp('isotropic',globalOpts.matModel)
            % Plot identified Qxx vs time
            labelStrs.y = 'Stiffness $Q_{xx}$, (GPa)';
            stiffPlotOpts.targVal = material.Qxx*10^-9;
            hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
                identStiffVFOpt.QxxVsT,identStiffVFOpt.QxxAvgOverT);
            saveFile = [savePath,'\','VFOpt_StiffVsT_Qxx'];
            print(hf,saveFile,plotProps.format,plotProps.saveRes)
            saveas(hf,saveFile,'fig')
            
            % Plot identified Qxx vs time
            labelStrs.y = 'Stiffness $Q_{xy}$, (GPa)';
            stiffPlotOpts.targVal = material.Qxy*10^-9;
            hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
                identStiffVFOpt.QxyVsT,identStiffVFOpt.QxyAvgOverT);
            saveFile = [savePath,'\','VFOpt_StiffVsT_Qxy'];
            print(hf,saveFile,plotProps.format,plotProps.saveRes)
            saveas(hf,saveFile,'fig')
            
             % Plot identified Exx vs time
            labelStrs.y = '$E$, (GPa)';
            stiffPlotOpts.targVal = material.Exx*10^-9;
            hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
                identStiffVFOpt.ExxVsT,identStiffVFOpt.ExxAvgOverT);
            saveFile = [savePath,'\','VFOpt_StiffVsT_E'];
            print(hf,saveFile,plotProps.format,plotProps.saveRes)
            saveas(hf,saveFile,'fig')

            % Plot identified Nuxy vs length
            labelStrs.y = '$\nu_{xy}$, (GPa)';
            stiffPlotOpts.unitConv = 1;
            stiffPlotOpts.specifyAxisLims = true;
            stiffPlotOpts.targVal = material.nuxy;
            hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
                identStiffVFOpt.NuxyVsT,identStiffVFOpt.NuxyAvgOverT);
            saveFile = [savePath,'\','VFOpt_StiffVsT_Nu'];
            print(hf,saveFile,plotProps.format,plotProps.saveRes)
            saveas(hf,saveFile,'fig')
            
            stiffPlotOpts.unitConv = 10^-9;
            stiffPlotOpts.specifyAxisLims = false;
            
        elseif strcmp('orthotropicReduced',globalOpts.matModel)
            % Plot identified Qxx vs time
            labelStrs.y = 'Stiffness $Q_{xx}$, (GPa)';
            stiffPlotOpts.targVal = material.Qxx*10^-9;
            hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
                identStiffVFOpt.QxxVsT,identStiffVFOpt.QxxAvgOverT);
            saveFile = [savePath,'\','VFOpt_StiffVsT_Qxx'];
            print(hf,saveFile,plotProps.format,plotProps.saveRes)
            saveas(hf,saveFile,'fig')
            
            % Plot identified Exx vs time
            labelStrs.y = 'Stiffness $E_{xx}$, (GPa)';
            stiffPlotOpts.targVal = material.Exx*10^-9;
            hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
                identStiffVFOpt.ExxVsT,median(identStiffVFOpt.ExxAvgOverT));
            saveFile = [savePath,'\','VFOpt_StiffVsT_Exx'];
            print(hf,saveFile,plotProps.format,plotProps.saveRes)
            saveas(hf,saveFile,'fig')
            
        elseif strcmp('orthotropic',globalOpts.matModel) || strcmp('orthotropicAngle',globalOpts.matModel)
            % Plot identified Q11 vs Time
            labelStrs.y = 'Stiffness $Q_{11}$, (GPa)';
            stiffPlotOpts.targVal = material.Q11*10^-9;
            hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
                identStiffVFOpt.Q11VsT,median(identStiffVFOpt.Q11VsT));
            saveFile = [savePath,'\','VFOpt_StiffVsT_Q11'];
            print(hf,saveFile,plotProps.format,plotProps.saveRes)
            saveas(hf,saveFile,'fig')
            
            % Plot identified Q22 vs Time
            labelStrs.y = 'Stiffness $Q_{22}$, (GPa)';
            stiffPlotOpts.targVal = material.Q22*10^-9;
            hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
                identStiffVFOpt.Q22VsT,median(identStiffVFOpt.Q22VsT));
            saveFile = [savePath,'\','VFOpt_StiffVsT_Q22'];
            print(hf,saveFile,'-dpng','-r0')
            saveas(hf,saveFile,'fig')
            
            % Plot identified Q12 vs Time
            labelStrs.y = 'Stiffness $Q_{12}$, (GPa)';
            stiffPlotOpts.targVal = material.Q12*10^-9;
            hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
                identStiffVFOpt.Q12VsT,median(identStiffVFOpt.Q12VsT));
            saveFile = [savePath,'\','VFOpt_StiffVsT_Q12'];
            print(hf,saveFile,plotProps.format,plotProps.saveRes)
            saveas(hf,saveFile,'fig')
            
            % Plot identified Q66 vs Time
            labelStrs.y = 'Stiffness $Q_{66}$, (GPa)';
            stiffPlotOpts.targVal = material.Q66*10^-9;
            hf = func_plotStiffnessVsTime(labelStrs,stiffPlotOpts,time,...
                identStiffVFOpt.Q66VsT,median(identStiffVFOpt.Q66VsT));
            saveFile = [savePath,'\','VFOpt_StiffVsT_Q66'];
            print(hf,saveFile,plotProps.format,plotProps.saveRes)
            saveas(hf,saveFile,'fig')  
        end
    end
end

%% POST-PROCESSING - Energy Balance
if globalOpts.calcEBal
    fprintf('\n--------------------------------------------------------------\n')
    fprintf('POST PROCESSING: Energy Balance\n')
    fprintf('--------------------------------------------------------------\n')
    fprintf('Initialising energy balance options structure.\n')
    % Create the element structure for the energy balance
    element.areaxy = pos.xStep*pos.yStep;
    element.volume = pos.xStep*pos.yStep*specimen.thickness;
    element.mass = material.rho*element.volume;

    % Set the energy balance options - control volume specification
    EBalOpts.xBound = [EBalOpts.cutPxY+1,size(disp.x,2)-EBalOpts.cutPxX];
    EBalOpts.xSubset = EBalOpts.xBound(1):EBalOpts.xBound(2);
    EBalOpts.yBound = [EBalOpts.cutPxY+1,size(disp.x,1)-EBalOpts.cutPxY];
    EBalOpts.ySubset = EBalOpts.yBound(1):EBalOpts.yBound(2);
    EBalOpts.impEdgeX = size(disp.x,2);
    EBalOpts.impEdgeY = 1:size(disp.x,1);

    fprintf('Performing energy balance calculation.\n')
    % Calculate the energy components from the kinematic fields
    energy = func_calcEnergyBalance2D_strainOnly(EBalOpts,pos,time,specimen,...
        material,element,disp,vel,accel,strain);

    % Plot the energy balance figure
    fprintf('Plotting energy balance diagnostic figure.\n')
    eBalPath = [imagePath,'EnergyBalance\'];
    if exist(eBalPath,'file') ~= 7
        mkdir(eBalPath);
    end
    plotParams.specifyAxisLims = true;
    plotParams.xLim = [0,time.vec(end)*10^6];
    plotParams.yLim = [-10,40];
    hf = func_plotEnergyBalance(plotParams,EBalOpts,energy,time);
    print(hf,[eBalPath,'\','0_EnergyBalance'],'-dpng','-r0')
    saveas(hf,[eBalPath,'\','0_EnergyBalance'],'fig')
end

%% POST-PROCESSING - Strength Identification
if globalOpts.identifyStrength
    fprintf('\n--------------------------------------------------------------\n')
    fprintf('POST PROCESSING: Strength Identification\n')
    fprintf('--------------------------------------------------------------\n')
    
    %--------------------------------------------------------------------------
    % Obtain Fracture Location
    % Check if the fracture file exists, if it does ask to load it
    loadFractFile = false;
    fractFile = 'fracture.mat';
    fractFilePath = imagePath;
    if exist([fractFilePath,fractFile],'file') == 2
        if strOpts.autoLoadFractureFile
            loadFractFile = true;
            load([fractFilePath,fractFile])
        else
            % Load the file and present the parameters:
            load([fractFilePath,fractFile])
            dlgStr = {'Fracture file found with:',...
                ['locX: ',sprintf('%i',fracture.locX)],...
                ['locY: ',sprintf('%i',fracture.locY)],...
                'Load file?'};

            choice = questdlg(dlgStr,'Load fracture file?', 'Yes','No','No');
            switch choice
                case 'Yes'
                    loadFractFile = true;
                case 'No'
                    loadFractFile = false;
                    clear fracture
            end
        end
    end

    % If the fracture location doesn't already exist, let the user input it
    if ~loadFractFile
        % Check if the user wants to manually input this data or not
        inputFractLoc = false;
        choice = questdlg('Manually input fracture location and time?', ...
        'Fracture Location Input', 'Manual','Graphic Pick','Manual');
        switch choice
            case 'Manual'
                inputFractLoc = true;
            case 'Graphic Pick'
                inputFractLoc = false;
        end

        if inputFractLoc
            fprintf('Obtaining fracture location (x,y) from user input by manual input.\n')
            fracture = func_manualInputFractureLoc(pos,time);
        else
            fprintf('Obtaining fracture location (x,y) from user input by picking.\n')
            fracture = func_findFractureLoc(imagePath,pos,time,disp,accel,strain,strainRate);
        end
        % Save the fracture location to file
        save([fractFilePath,fractFile],'fracture')
    end

    %--------------------------------------------------------------------------
    % Iniatialise strength processing options
    strOpts.virtualGauge = func_createVirtualGauge(strOpts.virtualGauge,fracture,strain);
    
    % Depending on material model set the stiffness matrix for stres
    % calculation from strain
    if strcmp('isotropic',globalOpts.matModel)
        if strcmp('SG',strOpts.stiffnessMethod)
            strOpts.stiffnessQxx = identStiffSG.QxxAvgOverL;
            strOpts.stiffnessQxy = material.Qxy;
        elseif strcmp('VF',strOpts.stiffnessMethod)
            strOpts.stiffnessQxx = identStiffVFOpt.QxxAvgOverL;
            strOpts.stiffnessQxy = identStiffVFOpt.QxyAvgOverL; 
        else
            strOpts.stiffnessQxx = nanmean([identStiffSG.QxxAvgOverL,identStiffVFOpt.QxxAvgOverT]);
            strOpts.stiffnessQxy = identStiffVFOpt.QxyAvgOverT;
        end
        strOpts.stiffnessQMat = [strOpts.stiffnessQxx,strOpts.stiffnessQxy,0;...
                                     strOpts.stiffnessQxy,strOpts.stiffnessQxx,0;...
                                     0,0,(strOpts.stiffnessQxx-strOpts.stiffnessQxy)/2]; 
    elseif strcmp('orthotropicReduced',globalOpts.matModel)
        if strcmp('SG',strOpts.stiffnessMethod)
            strOpts.stiffnessQxx = identStiffSG.QxxAvgOverL;
        elseif strcmp('VF',strOpts.stiffnessMethod)
            strOpts.stiffnessQxx = identStiffVFOpt.QxxAvgOverL;
        else
            strOpts.stiffnessQxx = nanmean([identStiffSG.QxxAvgOverL,identStiffVFOpt.QxxAvgOverT]);
        end
        strOpts.stiffnessQMat = [strOpts.stiffnessQxx,0,0;...
                                 0,0,0;...
                                 0,0,0]; 
    elseif strcmp('orthotropic',globalOpts.matModel)
        fprintf('WARNING: orthotropic strength identification not implemented.\n')
    elseif strcmp('orthotropicAngle',globalOpts.matModel)
        fprintf('WARNING: angled orthotropic strength identification not implemented.\n')
    else
        fprintf('WARNING: specifed material model not recognised.\n')
    end
    
    %--------------------------------------------------------------------------
    % Calculate the stress from the strains using stiffness matrix
    fprintf('Calculating stress from strain using the identified stiffness matrix.\n')
    [stress.QRecon.x,~,~] = func_calcStressFromStrain2D(strain,strOpts.stiffnessQMat);
    
    fprintf('Calculating average stress and strain in the virtual gauge area at the fracture location.\n')
    [stress,strain] = func_calcAvgStressStrainInGaugeArea(globalOpts,strOpts,stress,strain);
    
    % Plot the difference between the acceleration and strain stress measures
    fprintf('Plotting the difference between the stress gauge and stress from strain vs time.\n');
    hf = func_plotStressDiffVsTime(plotParams,savePath,time,stress,fracture);
    saveFile = [savePath,'\','StrIdent_StressDiff']; 
    print(hf,saveFile,plotProps.format,plotProps.saveRes)
    saveas(hf,saveFile,'fig')

    %--------------------------------------------------------------------------
    % Manually or Automatically Identify the Strength
    if strcmp(strOpts.method,'Manual')
        % MANUAL Strength Identification
        fprintf('ANALYSE STRESS DIFFERENCE TO DETERMINE FRACTURE POINT.\n');
        fprintf('Press any key to continue.\n');
        pause;

        % Get the Fracture Frame/Time
        fprintf('Obtaining fracture frame from user input by manual input.\n')
        fracture.strengthFrame = func_manualInputFractureFrame(pos,time);

        % Identify strength
        fprintf('Identifying strength from stress gauge and strains using the virtual gauge area.\n')
        [identStrength.stressGauge,identStrength.linStressGauge,identStrength.fromStrain] =...
            func_identifyStrength(time,strOpts.virtualGauge,fracture,stress,strain,strainRate);
    else
        % AUTOMATED Strength Identification
        [identStrength.stressGauge,identStrength.linStressGauge,fracture] = ...
                func_identifyStrengthAutomated(strOpts.virtualGauge,fracture,stress);
        identStrength.fromStrain = nan;
    end

    %--------------------------------------------------------------------------
    % DIAGNOSTIC IMAGE SEQ 1: Width averaged stress measures and fields
    fprintf('Plotting diagnostic image sequence 1: width average stress and fields.\n')
    plotParams.cutPxY = 0;

    strImageSeqSavePath = [imagePath,'StrIdent_ImageSeq_WAvg\'];
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        plotImageSeq = func_checkIfPathExistsDlg(strImageSeqSavePath,...
            'Width averaged strength identification image sequence folder found, plot again?','Plot Image Sequence?');
    elseif strcmp(globalOpts.plotImageSeqs,'yes')
        plotImageSeq = true;
        if exist(strImageSeqSavePath,'file') ~= 7
            mkdir(strImageSeqSavePath);
        end
    else
        plotImageSeq = false; 
    end

    % Video of width average stress, raw strain, strain, accel, strain rate
    if plotImageSeq
        func_plotIdentStrVid_WAvg(plotParams,strImageSeqSavePath,pos,time,...
            disp,accel,strain,strainRate,stress,fracture);
    end

    %--------------------------------------------------------------------------
    % DIAGNOSTIC IMAGE SEQ 2: Stress fields and local stress measures/curve
    fprintf('Plotting diagnostic image sequence 2: stress fields and stress measures.\n')

    strImageSeqSavePath = [imagePath,'StrIdent_ImageSeq_Field\'];
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        plotImageSeq = func_checkIfPathExistsDlg(strImageSeqSavePath,...
            'Stress field strength identification image sequence folder found, plot again?','Plot Image Sequence?');
    elseif strcmp(globalOpts.plotImageSeqs,'yes')
        plotImageSeq = true;
        if exist(strImageSeqSavePath,'file') ~= 7
            mkdir(strImageSeqSavePath);
        end
    else
        plotImageSeq = false; 
    end

    % Video of stress fields from linear gauge and strain, local 
    % stress-strain curve and stress difference 
    if plotImageSeq
        if strcmp('isotropic',globalOpts.matModel)
            ssCurveLabels.xStr = '$\overline{\epsilon_{xx} + \nu \epsilon_{yy}}^{A}$ [$mm.m^{-1}$]';
            ssCurveLabels.yStr = '$\overline{\sigma_{xx}}^{A}$ [$MPa$]';
            func_plotIdentStrVid_Field(plotParams,strImageSeqSavePath,pos,time,...
                stress,stress.virtGaugeAvg.x,strain.virtGaugeAvg.xny,ssCurveLabels,fracture,strOpts,false);

        elseif strcmp('orthotropicReduced',globalOpts.matModel)
            ssCurveLabels.xStr = '$\overline{\epsilon_{xx}}^{A}$ [$mm.m^{-1}$]';
            ssCurveLabels.yStr = '$\overline{\sigma_{xx}}^{A}$ [$MPa$]';
            func_plotIdentStrVid_Field(plotParams,strImageSeqSavePath,pos,time,...
                stress,stress.virtGaugeAvg.x,strain.virtGaugeAvg.x,ssCurveLabels,fracture,strOpts,false);

        elseif strcmp('orthotropic',globalOpts.matModel)
            fprintf('WARNING: orthotropic strength identification not implemented.\n')

        elseif strcmp('orthotropicAngle',globalOpts.matModel)
            fprintf('WARNING: angled orthotropic strength identification not implemented.\n')

        else
            fprintf('WARNING: specifed material model not recognised.\n')
        end 
    end
    
    %--------------------------------------------------------------------------
    % Plot the strain rate at the fracture location up to the fracture frame
    fprintf('Calculating maximum strain rate at the fracture location.\n')
    fprintf('Plotting the strain rate at the fracture location.\n')
    strainRate.strStrainRate = func_plotStrainRateAtFracLoc(plotParams,savePath,...
        time,strOpts.virtualGauge,fracture,strainRate);
    
    % Update the maximum strain rate variables
    if smoothingOpts.spatialSmooth == true
        edgePx = grid.pxPerPeriod+round(smoothingOpts.spatialKernal(1)/2)+1;
    else
        edgePx = grid.pxPerPeriod+1;
    end
    xRange = edgePx:size(strainRate.xAvg,1) - edgePx;
    tRange = 1:fracture.strengthFrame;
    % Ignore the edges where the data is corrupted by smoothing effects
    strainRate.xAvgMinPreFract = min(min(strainRate.xAvg(xRange,tRange)));
    strainRate.xAvgMaxPreFract = max(max(strainRate.xAvg(xRange,tRange)));
    identStrainRate.strStrainRate = strainRate.strStrainRate;
    
    %--------------------------------------------------------------------------
    % FINAL DIAG FIG: Plot the stress fields maps and local stress-strain curves
    fprintf('Plotting diagnostic images at fracture time: stress fields and stress measures.\n')
     if strcmp('isotropic',globalOpts.matModel)
        ssCurveLabels.xStr = '$\overline{\epsilon_{xx} + \nu \epsilon_{yy}}^{A}$ [$mm.m^{-1}$]';
        ssCurveLabels.yStr = '$\overline{\sigma_{xx}}^{A}$ [$MPa$]';
        hf = func_plotIdentStrVid_Field(plotParams,savePath,pos,time,...
            stress,stress.virtGaugeAvg.x,strain.virtGaugeAvg.xny,ssCurveLabels,fracture,strOpts,true);
        saveFile = [savePath,'\','StrIdent_Diagnostics'];
        %export_fig(saveFile,'-eps','-pdf','-c[Inf,Inf,Inf,Inf]',hf)
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')

    elseif strcmp('orthotropicReduced',globalOpts.matModel)
        ssCurveLabels.xStr = '$\overline{\epsilon_{xx}}^{A}$ [$mm.m^{-1}$]';
        ssCurveLabels.yStr = '$\overline{\sigma_{xx}}^{A}$ [$MPa$]';
        hf = func_plotIdentStrVid_Field(plotParams,savePath,pos,time,...
            stress,stress.virtGaugeAvg.x,strain.virtGaugeAvg.x,ssCurveLabels,fracture,strOpts,true);
        saveFile = [savePath,'\','StrIdent_Diagnostics'];
        %export_fig(saveFile,'-eps','-pdf','-c[Inf,Inf,Inf,Inf]',hf)
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')

    elseif strcmp('orthotropic',globalOpts.matModel)
        fprintf('WARNING: orthotropic strength identification not implemented.\n')

    elseif strcmp('orthotropicAngle',globalOpts.matModel)
        fprintf('WARNING: angled orthotropic strength identification not implemented.\n')

    else
        fprintf('WARNING: specifed material model not recognised.\n')
     end 
    
    %--------------------------------------------------------------------------
    % Get the max stress and its location and time over the whole FOV
    % Useful for specimens that did not actually fail
    identStrength.FOVlims.rangeX = grid.pxPerPeriod+1:size(stress.xLinearGauge,2)-grid.pxPerPeriod-1;
    identStrength.FOVlims.rangeY = grid.pxPerPeriod+1:size(stress.xLinearGauge,1)-grid.pxPerPeriod-1;
    identStrength.FOVlims.rangeT = floor(smoothingOpts.WATemporalKernal(1)/2):...
        size(stress.xLinearGauge,3)-floor(smoothingOpts.WATemporalKernal(1)/2);
    
    % Get the maximum and minimum stress guage values and their locs in
    % space and time over the whole FOV
    temp = stress.xAvg(identStrength.FOVlims.rangeX,identStrength.FOVlims.rangeT);
    [identStrength.FOVlims.SGMax,ind] = max(temp(:));
    [identStrength.FOVlims.SGMaxLocX,identStrength.FOVlims.SGMaxLocT]...
        = ind2sub(size(temp),ind);
    [identStrength.FOVlims.SGMin,ind] = min(temp(:));
    [identStrength.FOVlims.SGMinLocX,identStrength.FOVlims.SGMinLocT]...
        = ind2sub(size(temp),ind);
    clear temp
    
    % Correct for the pixels/frames cut from the edges
    identStrength.FOVlims.SGMaxLocX = identStrength.FOVlims.SGMaxLocX+identStrength.FOVlims.rangeX(1)-1;
    identStrength.FOVlims.SGMaxLocT = identStrength.FOVlims.SGMaxLocT+identStrength.FOVlims.rangeT(1)-1;
    identStrength.FOVlims.SGMinLocX = identStrength.FOVlims.SGMinLocX+identStrength.FOVlims.rangeX(1)-1;
    identStrength.FOVlims.SGMinLocT = identStrength.FOVlims.SGMinLocT+identStrength.FOVlims.rangeT(1)-1;
    
    %--------------------------------------------------------------------------
    % Store vectors of the stress vs time at the fracture location
    identStrength.stressVsTAccel.xxAvgOverY = squeeze(stress.xAvg(fracture.locX,:));
    identStrength.stressVsTAccel.xxAvgOverGauge = stress.virtGaugeAvg.x; 
    identStrength.stressVsTStrain.xxAvgOverY = squeeze(stress.QRecon.x(:,fracture.locX,:));
    identStrength.stressVsTStrain.xxAvgOverGauge = stress.QRecon.virtGaugeAvg.x;
    
    %--------------------------------------------------------------------------
    % Write identified values to the console
    fprintf('\nIdentified strength values:\n')
    Strength_FracLoc = fracture
    Strength_StressGauge = identStrength.stressGauge
    Strength_LinearGauge = identStrength.linStressGauge

    fprintf('Strength Identification COMPLETE.\n')
    close all
else
    identStrength = nan;
    virtualGauge = nan;
    fracture = nan;
end

%% IMAGE SEQUENCES - for powerpoint presentations
if globalOpts.plotPresentationImages 
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        choice = questdlg('Plot images sequences for presentation? (takes a long time)', ...
            'Plot Image Sequence?', 'Yes','No','No');
        switch choice
            case 'Yes'
                plotPresentationImageSeq = true;
            case 'No'
                plotPresentationImageSeq= false;
        end
    elseif strcmp(globalOpts.plotImageSeqs,'yes')
        plotPresentationImageSeq = true;
    else
        plotPresentationImageSeq = false; 
    end

    if plotPresentationImageSeq == true
        fprintf('\n--------------------------------------------------------------\n')
        fprintf('IMAGE SEQUENCES: powerpoint presentations\n')
        fprintf('--------------------------------------------------------------\n')
        
        % Setup the plot parameters for the presentation figures
        savePath = imagePath;
        
        % Specify the frames over which the data is plotted
        endFrame = fracture.strengthFrame+25;
        %endFrame = time.numFrames;
        if endFrame > time.numFrames
            endFrame = time.numFrames;
        end
        plotParams.tRange = 1:endFrame;
        
        % Specify the location of the SS curve to be plotted
        if exist('fracture','var') == 1
            ssCurve.locX = fracture.locX;
        else
            ssCurve.locX = round(length(pos.x)/2);
        end
        % Plot the image sequences for presentation purposes
        fprintf('Plotting presentation image sequences.\n')
        func_plotPresentationVideos(globalOpts,plotParams,savePath,pos,time,...
            strain,strainRate,accel,stress,identStiffSG,ssCurve)
    end
end

%% SAVE DATA - Processing and Config Variables to File
if globalOpts.autoSaveProcData
    saveProcessedData = true;
else
    choice = questdlg('Save processed data to file?', ...
    'Save Data?', 'Yes','No','No');
    switch choice
        case 'Yes'
            saveProcessedData = true;
        case 'No'
            saveProcessedData = false;
    end
end

if saveProcessedData
    fprintf('\n--------------------------------------------------------------\n')
    fprintf('SAVING DATA\n')
    fprintf('--------------------------------------------------------------\n')
    %--------------------------------------------------------------------------
    % Save a .mat and readme file of the processed data
    % create filenames for saving data and config files
    fprintf('Saving processed data to README and mat files...\n')
    today = date;
    % Setup path and various file names for saving
    saveFiles.path = imagePath;
    saveFiles.processedData = 'AllPostProcessedData.mat';
    saveFiles.configFile = 'ConfigVariables.mat';
    saveFiles.configVars = 2:6;
    saveFiles.processedVars = 5:16;
    saveFiles.README = strcat('README_',today,'.txt');
    func_saveAllData(saveFiles,test,specimen,material,time,pos,disp,accel,strain,strainRate,stress,...
        identStiffSG,identStiffVFMan,identStiffVFOpt,identStrength,fracture)
    fprintf('All data files saved.\n')
end

%% SAVE DATA - Print Identified Properties to Console and Save in Struct
fprintf('\n--------------------------------------------------------------\n')
fprintf('IDENTIFIED MATERIAL PROPERTIES\n')
fprintf('--------------------------------------------------------------\n')

% Create condensed info on the strain rate
identStrainRate.xAvgMin = strainRate.xAvgMin;
identStrainRate.xAvgMax = strainRate.xAvgMax;
if globalOpts.identifyStrength
    identStrainRate.xAvgMinPreFract = strainRate.xAvgMinPreFract;
    identStrainRate.xAvgMaxPreFract = strainRate.xAvgMaxPreFract;
else
    identStrainRate = nan;
end

fprintf('Identified stiffness values:\n')
IdentifiedStiffness_SG = identStiffSG
IdentifiedStiffness_VFMan = identStiffVFMan
IdentifiedStiffness_VFOpt = identStiffVFOpt
StrainRateMaxValues = identStrainRate

if globalOpts.identifyStrength
    fprintf('Identified strength values:\n')
    Strength_FracLoc = fracture
    Strength_StressGauge = identStrength.stressGauge
    Strength_LinearGauge = identStrength.linStressGauge
end

save([imagePath,'IdentifiedProperties.mat'],'identStiffSG','identStiffVFMan','identStiffVFOpt','fracture',...
    'identStrength','identStrainRate');

close all
fprintf('\nCOMPLETE.\n\n') 