function [identStiffVF,errStats] = func_manualVFIsoElastic(pos,time,material,accel,strain,debug,axLim)
% Author: Frances Davis
% Edited By: Lloyd Fletcher
% PhotoDyn Group, University of Southampton
% Date: 8/5/2017
%
% Uses manual global virtual fields to find modulus and poissons ratio for
% a homogenous isotropic material

%% Input Checking
if nargin < 6
    debug = false;
    axLim.nu = [0,0.5];
    axLim.E = [0,500];
end

%% Define the Virtual Fields
rho = material.rho;
X = pos.xGrid;
Y = pos.yGrid;

L = max(max(X));% in m
H = max(max(Y)); % in m

%Define the Virtual fields
%First Virtual Field
u1_VF1 = (L-X(:,:));
%u1_VF1 = (L-X(:,:)).*X(:,:);
e1_VF1 = -1.*ones(size(X));
%e1_VF1 = L-2*X(:,:);
u2_VF1 = zeros(size(X));
e2_VF1 = zeros(size(X));
e6_VF1 = zeros(size(X));

% Second Virtual Field
u2_VF2 = (L-X(:,:)).*Y(:,:)/H;
u1_VF2 = zeros(size(X));
e1_VF2 = u1_VF2;
e2_VF2 = (L-X(:,:))/H;
e6_VF2 = -Y(:,:)/H;


%% Loop on time to find the elastic paramters
Qt = zeros(2, time.numFrames);
stab_check = zeros(time.numFrames,1);

for kk = 1:time.numFrames
    
    %Reshape strain and acceleration
    Exx = reshape(strain.x(:,:,kk), [], 1);
    Eyy = reshape(strain.y(:,:,kk), [], 1);
    Exy = reshape(strain.s(:,:,kk), [], 1);
    
    a1 = reshape(accel.x(:,:,kk), [], 1);
    a2 = reshape(accel.y(:,:,kk), [], 1);
    
    %Reshape virtual fields
    u1_1 = reshape(u1_VF1, [], 1);
    e1_1 = reshape(e1_VF1, [], 1);
    u2_1 = reshape(u2_VF1, [], 1);
    e2_1 = reshape(e2_VF1, [], 1);
    e6_1 = reshape(e6_VF1, [], 1);
    
    u1_2 = reshape(u1_VF2, [], 1);
    e1_2 = reshape(e1_VF2, [], 1);
    u2_2 = reshape(u2_VF2, [], 1);
    e2_2 = reshape(e2_VF2, [], 1);
    e6_2 = reshape(e6_VF2, [], 1);
    
    A(2*kk-1:2*kk,:) = [-(nanmean(Exx.*e1_1)+ nanmean(Eyy.*e2_1) + nanmean(0.5*Exy.*e6_1)), ...
                        -(nanmean(Exx.*e2_1)+ nanmean(Eyy.*e1_1) - nanmean(0.5*Exy.*e6_1)); ...
                        -(nanmean(Exx.*e1_2)+ nanmean(Eyy.*e2_2) + nanmean(0.5*Exy.*e6_2)), ...
                        -(nanmean(Exx.*e2_2)+ nanmean(Eyy.*e1_2) - nanmean(0.5*Exy.*e6_2))];
    
    B(2*kk-1:2*kk,1) = [rho*(nanmean(a1.*u1_1)+nanmean(a2.*u2_1));...
                        rho*(nanmean(a1.*u1_2)+nanmean(a2.*u2_2))];
      
   Qt(:,kk) = A(2*kk-1:2*kk,:)\B(2*kk-1:2*kk,1); %Result in MPa 
   
   stab_check(kk) = cond(A(2*kk-1:2*kk,:));
   
end

%% Convert the Qs to E and nu
Q = A\B;

nu_t = Qt(2,:)./Qt(1,:);        % Poisson's ratio as a function of time
E_t = Qt(1,:).*(1-nu_t.^2);     % Modulus as a function of time

nu = Q(2)./Q(1);                 % Poisson's ratio?
ECalc = Q(1).*(1-nu^2)*10^-9;    % Modulus?    

% Determine the residual on the least-squares solution
res = sum((B - A*Q).*(B - A*Q))^.5;

%% Plot figures for debugging
if debug
    t = (1:time.numFrames)*time.step*10^6;

    % Plot for Poisson's Ratio
    figure; plot(t, nu_t, '-o'); xlabel('Time, \mus','FontSize',18, 'FontName', 'Times'); ylabel('Poisson''s Ratio, \nu','FontSize',18, 'FontName', 'Times');
    grid on;
    axis([0  t(end) axLim.nu])
    set(gca,'FontSize',18, 'FontName', 'Times')

    % Plot for E
    figure; plot(t, E_t*10^-9, '-o'); xlabel('Time, \mus','FontSize',18, 'FontName', 'Times'); ylabel('Young''s Modulus, E, GPa','FontSize',18, 'FontName', 'Times');
    grid on;
    axis([0  t(end) axLim.E])
    set(gca,'FontSize',18, 'FontName', 'Times')

    % Plot  of stability
    figure; semilogy(t, stab_check, '-o'); xlabel('Time, \mus','FontSize',18, 'FontName', 'Times'); ylabel('Condition Number with Respect to Inversion','FontSize',18, 'FontName', 'Times');
    set(gca,'FontSize',18, 'FontName', 'Times')
    axis([0  t(end) 1 1000])
end

%% Assign return variables
% Error Statistics
errStats.stabCheck = stab_check;
errStats.residual = res;

% Identified Values
identStiffVF.QxxVsT = Qt(1,:);
identStiffVF.QxyVsT = Qt(2,:);
identStiffVF.ExxVsT = E_t;
identStiffVF.nuxyVsT = nu_t;
%identStiffVF.nu = nu;
%identStiffVF.E = ECalc;

end

