function func_plotIdentStrengthImage(plotParams,savePath,pos,time,strain,strainRate,...
    disp,stress,window,fracture)
% Author: Jared Van-Blitterswyk
% PhotoDyn Group, University of Southampton
% Date: 22/3/2017
% Edited By: Lloyd Fletcher
%
% Plots image of fracture location and strength identification

% Create a struct of formatting properties for the figure
plotProps = func_initPlotPropsStruct(plotParams.formatType);
plotParams.cAxisType = 'MaxQuantile';

% Remove rigid body motion from displacement maps
[disp.xDef,~] = func_calcDispRemoveRigidBody(disp.x);

% Create cropped position vectors for plotting
plotRangeX = plotParams.cutPxX+1:length(pos.x)-plotParams.cutPxX;
plotRangeY = plotParams.cutPxY+1:length(pos.y)-plotParams.cutPxY;
plotXPos = pos.x(plotParams.cutPxX+1:end-plotParams.cutPxX)*10^3;
plotYPos = pos.y(plotParams.cutPxY+1:end-plotParams.cutPxY)*10^3;

% Create the vertical line to show the fracture plane on each image
fractLineX(1:length(plotYPos)) = pos.x(fracture.location(1))*10^3;
fractLineY(1:length(plotYPos)) = plotYPos;

% Calculate the colour bar ranges
if isfield(plotParams,'cAxisType')
    plotParams.cRange{1} = func_calcColourBarRange(plotParams.cAxisType,disp.xDef(plotRangeY,plotRangeX,:));
    plotParams.cRange{2} = func_calcColourBarRange(plotParams.cAxisType,strain.x(plotRangeY,plotRangeX,:));
    plotParams.cRange{3} = func_calcColourBarRange(plotParams.cAxisType,strainRate.x(plotRangeY,plotRangeX,:));
end

% Create and size the figure
hf = figure('Visible','On');
set(hf,'color',[1 1 1])
set(hf,'Position', [plotProps.locX,plotProps.locY,...
     1.5*plotParams.Cols*plotProps.sizePerFigX,1.5*plotParams.Rows*plotProps.sizePerFigY])
set(hf,'PaperPositionMode','auto')

%--------------------------------------------------------------------------
% Stress at the width section
subplot(plotParams.Rows,plotParams.Cols,1)
plot(plotYPos,stress.sgFail(1)*ones(1,length(pos.y))/1e06,'k--','linewidth',plotProps.lw,'markersize',plotProps.ms)
hold on
plot(plotYPos,stress.sgFail(2)*ones(1,length(plotYPos))/1e06,'r--','linewidth',plotProps.lw,'markersize',plotProps.ms)
plot(plotYPos,stress.sgFail(3)*ones(1,length(plotYPos))/1e06,'b--','linewidth',plotProps.lw,'markersize',plotProps.ms)
plot(plotYPos,stress.xFailD(plotRangeY,1)/1e06,'ko-','linewidth',plotProps.lw,'markersize',plotProps.ms)
plot(plotYPos,stress.xFailD(plotRangeY,2)/1e06','rx-','linewidth',plotProps.lw,'markersize',plotProps.ms)
plot(plotYPos,stress.xFailD(plotRangeY,3)/1e06,'b*-','linewidth',plotProps.lw,'markersize',plotProps.ms)
legend('SG 1','SG 2','SG 3','S_D 1','S_D 2','S_D 3')
xlabel('Y (mm)','fontsize',plotProps.hfs,'fontname',plotProps.ft)
xlim([0 pos.y(end)*10^3])
ylabel('\sigma_{xx} (MPa)','fontsize',plotProps.hfs,'fontname',plotProps.ft)
ylim([0 round(max(stress.xFailD(:)),2,'significant')*10^-6])
titleStr = {'Strength Identification',...
    [sprintf('x = %.2fmm, t = %.1f',pos.x(fracture.location(1))*10^3,time.vec(fracture.frame-1)*10^6),'\mus']};
title(titleStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
set(gca,'fontsize',plotProps.fs,'fontname',plotProps.ft,'linewidth',plotProps.lw)
grid on
hold off

%--------------------------------------------------------------------------
% Deformation Displacement
subplot(plotParams.Rows,plotParams.Cols,2)
imagesc(plotXPos,plotYPos,disp.xDef(plotRangeY,plotRangeX,fracture.frame-1)*1e03)
hold on
plot(fractLineX,fractLineY,'--k','linewidth',plotProps.lw)
hold off
xlabel('X (mm)','fontsize',plotProps.hfs,'fontname',plotProps.ft)
ylabel('Y (mm)','fontsize',plotProps.hfs,'fontname',plotProps.ft)
titleStr = {'\it{u_{x}}\rm [mm]',[sprintf('t = %.2f',time.vec(fracture.frame-1)*10^6),'\mus']};
title(titleStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
set(gca,'fontsize',plotProps.fs,'fontname',plotProps.ft,'linewidth',plotProps.lw)
colormap(jet)
colorbar
% Determine the colour bar range 
if isfield(plotParams,'cAxisType')
    if ~strcmp(plotParams.cAxisType,'Auto')
        if isfield(plotParams,'cRange')
            caxis(plotParams.cRange{1}*10^3)
        end
    end
end

%--------------------------------------------------------------------------
% Strain
subplot(plotParams.Rows,plotParams.Cols,3)
imagesc(plotXPos,plotYPos,strain.x(plotRangeY,plotRangeX,fracture.frame-1)*1e03)
hold on
plot(fractLineX,fractLineY,'--k','linewidth',plotProps.lw)
hold off
xlabel('X (mm)','fontsize',plotProps.hfs,'fontname',plotProps.ft)
ylabel('Y (mm)','fontsize',plotProps.hfs,'fontname',plotProps.ft)
titleStr = {'\it{\epsilon_{x}}\rm [mm.m^{-1}]',[sprintf('t = %.2f',time.vec(fracture.frame-1)*10^6),'\mus']};
title(titleStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
set(gca,'fontsize',plotProps.fs,'fontname',plotProps.ft,'linewidth',plotProps.lw)
colorbar
colormap(jet)
% Determine the colour bar range 
if isfield(plotParams,'cAxisType')
    if ~strcmp(plotParams.cAxisType,'Auto')
        if isfield(plotParams,'cRange')
            caxis(plotParams.cRange{2}*10^3)
        end
    end
end

%--------------------------------------------------------------------------
% Strain Rate
subplot(plotParams.Rows,plotParams.Cols,4)
imagesc(plotXPos,plotYPos,strainRate.x(plotRangeY,plotRangeX,fracture.frame-1))
hold on
plot(fractLineX,fractLineY,'--k','linewidth',plotProps.lw)
hold off
xlabel('X (mm)','fontsize',plotProps.hfs,'fontname',plotProps.ft)
ylabel('Y (mm)','fontsize',plotProps.hfs,'fontname',plotProps.ft)
titleStr = {'\it{d\epsilon_{x}/dt}\rm [s^{-1}]',[sprintf('t = %.2f',time.vec(fracture.frame-1)*10^6),'\mus']};
title(titleStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
set(gca,'fontsize',plotProps.fs,'fontname',plotProps.ft,'linewidth',plotProps.lw)
colorbar
% Determine the colour bar range 
if isfield(plotParams,'cAxisType')
    if ~strcmp(plotParams.cAxisType,'Auto')
        if isfield(plotParams,'cRange')
            caxis(plotParams.cRange{3})
        end
    end
end

% Save the figure to the processed data file
print(hf,[savePath,'\StrengthIdentification'],plotProps.format,'-r0')

end

