function [strStrainRate,hf] = func_plotStrainRateAtFracLoc(plotParams,savePath,time,virtualGauge,...
    fracture,strainRate)
% Author: Lloyd Fletcher
% PhotoDyn Group, University of Southampton
% Date: 22/6/2017
%
% Plots the strain rate at the identified fracture location vs time

    % Create path for saving the image 
    if exist(savePath,'file') ~= 7
        mkdir(savePath);
    end

    % Calculate the average strain rates over each region
    strStrainRate.atFracLocYFracLocX = squeeze(strainRate.x(fracture.locY,fracture.locX,:));
    strStrainRate.avgGaugeYFracLocX = squeeze(mean(strainRate.x(virtualGauge.yRange,fracture.locX,:)));
    strStrainRate.avgGaugeYGaugeX = squeeze(mean(mean(strainRate.x(virtualGauge.yRange,virtualGauge.xRange,:))));
    strStrainRate.avgYFracLocX = squeeze(mean(strainRate.x(:,fracture.locX,:)));
    strStrainRate.avgGaugeXYFracTime = strStrainRate.avgGaugeYGaugeX(fracture.strengthFrame);
    
    maxSR = max([max(strStrainRate.atFracLocYFracLocX ),max(strStrainRate.avgGaugeYFracLocX),max(strStrainRate.avgGaugeYGaugeX)]);
    minSR = min([min(strStrainRate.atFracLocYFracLocX ),min(strStrainRate.avgGaugeYFracLocX),min(strStrainRate.avgGaugeYGaugeX)]);
    SRAxisMax = round(maxSR,2,'significant');
    SRAxisMin = round(minSR,2,'significant');
    
    % Create a line for the fracture location
    fracLineY = linspace(SRAxisMin,SRAxisMax,10);
    fracLineT(1:length(fracLineY)) = time.vec(fracture.strengthFrame)*10^6;
    fracLineF(1:length(fracLineY)) = fracture.strengthFrame;
    
    % Create a struct of formatting properties for the figure
    plotProps = func_initPlotPropsStruct(plotParams.formatType);
    plotParams.Rows = 1;
    plotParams.Cols = 2;
    
    %----------------------------------------------------------------------
    % Create and size the figure
    hf = figure('Visible','On');
    set(hf,'color',[1 1 1])
    set(hf,'Position', [plotProps.locX,plotProps.locY,...
         1.5*plotParams.Cols*plotProps.sizePerFigX,1.5*plotParams.Rows*plotProps.sizePerFigY])
    set(hf,'PaperPositionMode','auto')
    
    %----------------------------------------------------------------------
    % Strain Rate @ Frac Loc vs Time
    tStr = ['d\epsilon_{x}/dt(t=',sprintf('%.1f',time.vec(fracture.strengthFrame)*10^6)...
        ,') = ',sprintf('%.0f',strStrainRate.avgGaugeYFracLocX(fracture.strengthFrame)),'s^{-1}'];
    yStr = 'd\epsilon_{x}/dt (s^{-1})';
    xStr = 'Time t (\mus)';
    subplot(plotParams.Rows,plotParams.Cols,1)
    plot(time.vec*10^6,strStrainRate.atFracLocYFracLocX ,...
        '-ok','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    hold on
    plot(time.vec*10^6,strStrainRate.avgYFracLocX,...
        '-^g','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    plot(time.vec*10^6,strStrainRate.avgGaugeYFracLocX,...
        '-xb','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    plot(time.vec*10^6,strStrainRate.avgGaugeYGaugeX,...
        '-+r','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    plot(fracLineT,fracLineY,'--k','linewidth',plotProps.lw)
    hold off
    title(tStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
    xlabel(xStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
    ylabel(yStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
    axis([0,time.vec(end)*10^6,SRAxisMin,SRAxisMax])
    legend({'Lyx','AyLx','VGyLx','VGyx','Frac'},'location','best')
    set(gca, 'fontsize', plotProps.fs,'fontname',plotProps.ft)
    set(gca,'XMinorTick','on','YMinorTick','on')
    box on
    grid on
    
    %----------------------------------------------------------------------
    % Strain Rate @ Frac Loc vs Frame
    tStr = ['d\epsilon_{x}/dt(f=',sprintf('%i',fracture.strengthFrame)...
        ,') = ',sprintf('%.0f',strStrainRate.avgGaugeYFracLocX(fracture.strengthFrame)),'s^{-1}'];
    yStr = 'd\epsilon_{x}/dt (s^{-1})';
    xStr = 'Frame';
    subplot(plotParams.Rows,plotParams.Cols,2)
    plot(strStrainRate.atFracLocYFracLocX ,...
        '-ok','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    hold on
    plot(strStrainRate.avgYFracLocX,...
        '-^g','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    plot(strStrainRate.avgGaugeYFracLocX,...
        '-xb','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    plot(strStrainRate.avgGaugeYGaugeX,...
        '-+r','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    plot(fracLineF,fracLineY,'--k','linewidth',plotProps.lw)
    hold off
    title(tStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
    xlabel(xStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
    ylabel(yStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
    axis([0,time.numFrames,SRAxisMin,SRAxisMax])
    legend({'Lyx','AyLx','VGyLx','VGyx','Frac'},'location','best')
    set(gca, 'fontsize', plotProps.fs,'fontname',plotProps.ft)
    set(gca,'XMinorTick','on','YMinorTick','on')
    box on
    grid on
    
    
    print(hf,[savePath,'\','0_StrIdent_StrainRate'],'-dpng','-r0')
    saveas(hf,[savePath,'\','0_StrIdent_StrainRate'],'fig')
end

