function hf = func_plotStressDiffVsTime(plotParams,savePath,time,virtualGauge,...
    strain,stress,fracture,stiffness)
% Author: Lloyd Fletcher
% PhotoDyn Group, University of Southampton
% Date: 20/6/2017
%
% Plots the difference between stress measures at the fracture location vs
% time

    % Create path for saving the image 
    if exist(savePath,'file') ~= 7
        mkdir(savePath);
    end

    % Reconstruct the stress using the input strains and assumptions
    if stiffness.method == 1
        stress.xQRecon = stiffness.Exx*strain.x;
    else
        stress.xQRecon = stiffness.Exx/(1-stiffness.nuxy^2)*(strain.x+stiffness.nuxy.*strain.y);
    end
      
    % Calculate the stress difference based on width averages
    stress.xQReconAvg = squeeze(mean(stress.xQRecon));
    widthAvgDiff = stress.xAvg(fracture.locX,:) - stress.xQReconAvg(fracture.locX,:);
    
    % Calculate the stress difference based on local gauge over the virtual
    % gauge area
    stressAvgFromStrainOverGaugeArea = squeeze(mean(mean(...
        stress.xQRecon(virtualGauge.yRange,virtualGauge.xRange,:))));   
    stressAvgFromLinSGOverGaugeArea = squeeze(mean(mean(...
        stress.xLinearGauge(virtualGauge.yRange,virtualGauge.xRange,:))));   
    stressAvgOverGaugeAreaDiff = stressAvgFromLinSGOverGaugeArea - stressAvgFromStrainOverGaugeArea;
    
    % Calculate the stress difference based on local guage over the
    % fracture line
    stressAvgFromStrainOverFracLocX = squeeze(mean(...
        stress.xQRecon(virtualGauge.yRange,fracture.locX,:)));   
    stressAvgFromLinSGOverFracLocX = squeeze(mean(...
        stress.xLinearGauge(virtualGauge.yRange,fracture.locX,:)));    
    stressAvgOverFracLocXDiff = stressAvgFromLinSGOverFracLocX - stressAvgFromStrainOverFracLocX;

    % Create a struct of formatting properties for the figure
    plotProps = func_initPlotPropsStruct(plotParams.formatType);
    plotParams.Rows = 2;
    plotParams.Cols = 3;
    
    %----------------------------------------------------------------------
    % Create and size the figure
    hf = figure('Visible','On');
    set(hf,'color',[1 1 1])
    set(hf,'Position', [plotProps.locX,plotProps.locY,...
         1.5*plotParams.Cols*plotProps.sizePerFigX,1.5*plotParams.Rows*plotProps.sizePerFigY])
    set(hf,'PaperPositionMode','auto')
    
    %----------------------------------------------------------------------
    % Width Averaged Stress Difference Subplots
    yStr = '\sigma_{x} (MPa)';
    subplot(plotParams.Rows,plotParams.Cols,1)
    plot(stress.xAvg(fracture.locX,:)*10^-6,...
        '-ok','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    hold on
    plot(stress.xQReconAvg(fracture.locX,:)*10^-6,...
        '-xb','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    hold off
    xlabel('Frame','fontsize',plotProps.hfs,'fontname',plotProps.ft)
    ylabel(yStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
    xlim([0,time.numFrames])
    legend({'SG','\sigma(\epsilon)_{avg}'},'location','northwest')
    set(gca, 'fontsize', plotProps.fs,'fontname',plotProps.ft)
    set(gca,'XMinorTick','on','YMinorTick','on')
    box on
    grid on
    
     yStr = '\Delta \sigma_{SG}-\sigma(\epsilon) (MPa)';
    subplot(plotParams.Rows,plotParams.Cols,4)
    plot(widthAvgDiff*10^-6,...
        '-or','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    xlabel('Frame','fontsize',plotProps.hfs,'fontname',plotProps.ft)
    ylabel(yStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
    xlim([0,time.numFrames])
    set(gca, 'fontsize', plotProps.fs,'fontname',plotProps.ft)
    set(gca,'XMinorTick','on','YMinorTick','on')
    box on
    grid on
    
    %----------------------------------------------------------------------
    % Line Averaged Stress Difference Subplots  
    yStr = 'L Avg \sigma_{x} (MPa)';
    subplot(plotParams.Rows,plotParams.Cols,2)
    plot(stressAvgFromLinSGOverFracLocX*10^-6,...
        '-ok','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    hold on
    plot(stressAvgFromStrainOverFracLocX*10^-6,...
        '-xb','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    hold off
    xlabel('Frame','fontsize',plotProps.hfs,'fontname',plotProps.ft)
    ylabel(yStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
    xlim([0,time.numFrames])
    legend({'SG,L','\sigma(\epsilon)_{L,avg}'},'location','northwest')
    set(gca, 'fontsize', plotProps.fs,'fontname',plotProps.ft)
    set(gca,'XMinorTick','on','YMinorTick','on')
    box on
    grid on
    
    yStr = '\Delta \sigma_{SG,L}-\sigma(\epsilon,L) (MPa)';
    subplot(plotParams.Rows,plotParams.Cols,5)
    plot(stressAvgOverFracLocXDiff*10^-6,...
        '-or','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    xlabel('Frame','fontsize',plotProps.hfs,'fontname',plotProps.ft)
    ylabel(yStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
    xlim([0,time.numFrames])
    set(gca, 'fontsize', plotProps.fs,'fontname',plotProps.ft)
    set(gca,'XMinorTick','on','YMinorTick','on')
    box on
    grid on
    
    %----------------------------------------------------------------------
    % Area Averaged Stress Difference Subplots
    
    yStr = 'A Avg \sigma_{x} (MPa)';
    subplot(plotParams.Rows,plotParams.Cols,3)
    plot(stressAvgFromLinSGOverGaugeArea*10^-6,...
        '-ok','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    hold on
    plot(stressAvgFromStrainOverGaugeArea*10^-6,...
        '-xb','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    hold off
    xlabel('Frame','fontsize',plotProps.hfs,'fontname',plotProps.ft)
    ylabel(yStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
    xlim([0,time.numFrames])
    legend({'SG,A','\sigma(\epsilon)_{A,avg}'},'location','northwest')
    set(gca, 'fontsize', plotProps.fs,'fontname',plotProps.ft)
    set(gca,'XMinorTick','on','YMinorTick','on')
    box on
    grid on
    
     yStr = '\Delta \sigma_{SG,A}-\sigma(\epsilon,A) (MPa)';
    subplot(plotParams.Rows,plotParams.Cols,6)
    plot(stressAvgOverGaugeAreaDiff*10^-6,...
        '-or','linewidth',plotProps.lw,'markersize',plotProps.ms/2)
    xlabel('Frame','fontsize',plotProps.hfs,'fontname',plotProps.ft)
    ylabel(yStr,'fontsize',plotProps.hfs,'fontname',plotProps.ft)
    xlim([0,time.numFrames])
    set(gca, 'fontsize', plotProps.fs,'fontname',plotProps.ft)
    set(gca,'XMinorTick','on','YMinorTick','on')
    box on
    grid on
    
    print(hf,[savePath,'\','0_StrIdent_StressDiff'],'-dpng','-r0')
    saveas(hf,[savePath,'\','0_StrIdent_StressDiff'],'fig')
end

