%% Inertial Impact Experiment - Data Processing
% Author: Lloyd Fletcher
% PhotoDyn Group, University of Southampton
% Date: 19/6/2017
%
% Takes input images processes them with the grid method to obtain displacement 
% and then performs post processing using the stress gauge method to obtain
% Qxx and the tensile strength
%
% This code uses the grid processing tool box that can be found at:
% www.thegridMethodOpts.net, developed by Grediac

clc
clear all
close all

fprintf('--------------------------------------------------------------\n')
fprintf('INERTIAL IMPACT EXPERIMENTS - IMAGE DATA PROCESSING\n')
fprintf('--------------------------------------------------------------\n')
% Use hard coded data path for processing or select file with ui
hardCodePath = false;

% TURNS SMOOTHING ON/OFF, useful for processing no-noise image deformation data
% without smoothing
smoothingOn = true;
imageDef = false;

% Add the path for the grid method code and other useful functions:
funcPath = [pwd,'\Functions\'];
addpath(funcPath);
addpath([funcPath,'\GridMethodToolbox\']);

%% Setup the Data Structs to Hold the Test Information
fprintf('Creating data structures for test information.\n')

% Information about the test for the readme file
test.description = 'CFRP In-Plane Tranverse Tension';
test.specNum = 1; % specimen number
test.impactVel = 30; % impact speed [m/s]
test.wgLength = 50; % length of waveguide [mm]
test.projLength = 25; % length of projectile [mm]
test.projMaterial = 'Al 6061-T6';
test.testdate = 'Unknown'; % date test conducted

% Information about the frame rate
time.frameRate = 2e6;
time.step = 1/time.frameRate;
time.cutFrames = 3; % Frames to cut from the end of the accel/stress to avoid numerical diff effects
time.numFrames = 128;

% Material Data Struct - Nominal Target Properties
if imageDef
material.name = 'CF_IP_Model';
material.Exx = 10e9;
material.Ey = 135e9;
material.rho = 1600;
material.nuxy = 0.022;
material.nuyx = material.nuxy*(material.Ey/material.Exx);
material.Qxx = material.Exx/(1-material.nuxy*material.nuyx);
else
material.name = 'Gurit SE70 Carbon Fibre Pre-Preg [90]x15';
material.Exx = 7.29e9;
material.Ey = 137e9;
material.rho = 1530;
material.nuxy = 0.022;
material.nuyx = material.nuxy*(material.Ey/material.Exx);
material.Qxx = material.Exx/(1-material.nuxy*material.nuyx);    
end

% Specimen Data Struct - Nominal Geometry
specimen.length = 70e-3;
specimen.height = 45e-3;
specimen.thickness = 4.75e-3;
specimen.volume = specimen.length*specimen.height*specimen.thickness;
specimen.mass = specimen.volume*material.rho;
specimen.freeEdge = 'Right';

% Grid Data Struct
grid.name = 'Swiss_09mm_5pxpp';
grid.pitch = 0.9e-3;
grid.length = 70e-3;
grid.height= 45e-3;
grid.pxPerPeriod = 5;
grid.mPerPx = grid.pitch/grid.pxPerPeriod;
grid.numXPeriods = grid.length/grid.pitch;

% Struct to control how the grid images are processed
gridMethodOpts.windowFlag = 1; % 0 = gauss, 1 = bi-triang
gridMethodOpts.windowWidth = 1; % Multiplier for grid pitch to set window width
gridMethodOpts.dispCalcMethod = 2;
gridMethodOpts.temporalUnwrap = true;
gridMethodOpts.correctDefects = false;
gridMethodOpts.correctSDx = 3;
gridMethodOpts.debug  = true; % Plots diagnostic figures to debug errors

%% Load the image files
fprintf('Loading reference image from the select test data folder.\n')
if ~hardCodePath
    [imageFile,imagePath] = uigetfile({'*.*','All Files'},'Select the first image in the sequence');
else   
    % Swiss grid specimen 4 - Used for sensistivity study
    imageFile = 'Camera_15_21_38_001.tiff';
    %imagePath = 'E:\Matlab_WorkingDirectory\1_ImpactTest_ProcessingToolbox\CF_UD_InPlane_TransverseTension_SwissGrid09mm\Specimen4\';
    imagePath = 'C:\Users\Lloyd\Documents\UofS_Matlab\ImpactTest_Processing_Toolbox\CF_UD_InPlane_TransverseTension_SwissGrid09mm\Specimen4\';
    
    % Image Deformation Test Data Folder
    %imageFile = 'DefGridImage_001.tiff';
    %imagePath = 'E:\Matlab_WorkingDirectory\Experiment_Processing_CF\TestGrid_CFIP_M05mm_G09mm_400x250_16bit_SS3_C100\';   
end

%% Grid Method Processing
fprintf('\n--------------------------------------------------------------\n')
fprintf('GRID METHOD PROCESSING\n')
fprintf('--------------------------------------------------------------\n')

%--------------------------------------------------------------------------
% GRID IMAGE PROCESSING

% Check if the images have already been processed otherwise process the data
gridDataSavePath = imagePath;
gridDataFile = 'GridMethod_ProcessedData.mat';

fprintf('Checking for existing processed data file.\n')
processGridImages = true;
if exist([gridDataSavePath,gridDataFile],'file') == 2
    choice = questdlg('Processed grid data file found, process again?', ...
	'Process Grid Images', 'Yes','No','No');
    switch choice
        case 'Yes'
            processGridImages = true;
        case 'No'
            processGridImages = false;
    end
    % If we are not processing the images we should load the existing file
    if processGridImages == false
        fprintf('Loading processed grid data from: %s.\n',gridDataFile)
        load([gridDataSavePath,gridDataFile]);
    end
end

if processGridImages
    fprintf('Processing images using the grid method toolbox.\n')
    % Process the image squence with the grid method toolbox
    [grid,pos,disp] = func_gridMethodImageProcessing(imagePath,imageFile,grid,gridMethodOpts);    
    
    % Save the data to file so a reprocess is not necessary
    fprintf('Saving processed grid data to: %s.\n',gridDataFile)
    save([gridDataSavePath,gridDataFile],'grid','pos','disp') 

    fprintf('Grid Method Processing Complete.\n')
end

%--------------------------------------------------------------------------
% Update Geometry and Number of Frames Based on Displacement Matrix Size
fprintf('Updating geometry variables based on the size of displacement field matrices.\n')

% Check that the nominal specimen dimensions and the dimensions of the
% selected grid window are the same, if they are not update them
checkLength = grid.mPerPx*size(disp.x,2);
if checkLength ~= specimen.length
    specimen.length = checkLength;
    grid.length = checkLength;
end
checkHeight = grid.mPerPx*size(disp.x,1);
if checkHeight ~= specimen.height
    specimen.height = checkHeight;
    grid.height = checkHeight;
end

% Create the time vector based on the number of frames in the disp struct
time.numFrames = size(disp.x,3);
time.vec = 0:time.step:(size(disp.x,3)-1)*time.step;

%--------------------------------------------------------------------------
% Post Grid Method Defect Correction
if gridMethodOpts.correctDefects
    % TODO: reshape to 2D matrix with columns as space and width as time
    [disp.x,disp.y] = func_DefectRepair(disp.x,disp.y,grid.pxPerPeriod,gridMethodOpts.correctSDx);
end

% Currently the rotations are unused so remove them to save RAM
disp = rmfield(disp,'rot');

%% Setup the Post-Processing Options Structures
fprintf('\n--------------------------------------------------------------\n')
fprintf('POST PROCESSING - Stiffness Identification\n')
fprintf('--------------------------------------------------------------\n')
fprintf('Creating data structures for smoothing and extrapolation options.\n')

%--------------------------------------------------------------------------
% Smoothing Options 
if smoothingOn
    smoothingOpts.spatSmooth = true;
    smoothingOpts.WATempSmooth = true;
    smoothingOpts.FFTempSmooth = true;
else
    smoothingOpts.spatSmooth = false;
    smoothingOpts.WATempSmooth = false;
    smoothingOpts.FFTempSmooth = false;
end
% Descriptors
smoothingOpts.strainMethod = 'Spatial smooth only (on disp) then diff in space.';
smoothingOpts.accelAvgMethod = 'Average over width first, temporal smooth only (on disp), then double diff in time.';
smoothingOpts.FFAccelMethod = 'Temporal smooth full-field disp, then spatial smooth full-field disp before double diff in time';
% Spatial Smoothing Options
smoothingOpts.spatialFilt = 'gauss';
smoothingOpts.spatialKernal = [25,25]; % Specify kernal size in px
smoothingOpts.spatialEdgeMode = 'symmetric';
% Width Averaged Temporal Smoothing Options
smoothingOpts.WATemporalFilt = 'sgolay';
smoothingOpts.WATemporalKernal = [17,3];
smoothingOpts.WATemporalPad = true; 
% Full-Field Temporal Smoothing Options
smoothingOpts.FFTemporalFilt = 'sgolay';
smoothingOpts.FFTemporalKernal = [17,3];
smoothingOpts.FFTemporalPad = true;

%--------------------------------------------------------------------------
% Data Extrapolation Options
% Fix NaNs in the displacement field
extrapOpts.fixNaNs = true;
extrapOpts.fixNaNKernal = 10;    % Radius for averaging to replace NaNs in px 

% Extrapolation over the length to replace missing pitches
extrapOpts.extrapWidthAvgData = true;
extrapOpts.dispPx = 5;
if smoothingOpts.spatSmooth
    extrapOpts.strainPx = 5+floor(smoothingOpts.spatialKernal(1)/2);
else
    extrapOpts.strainPx = 5;
end
extrapOpts.dispMethod = 'linear';
extrapOpts.strainMethod = 'linear';
% Extrapolation options for the full-field data used with the VFM
extrapOpts.extrapFullFieldData = true;
extrapOpts.FFDispPx = 5;
if smoothingOpts.spatSmooth
    extrapOpts.FFStrainPx = 5+floor(smoothingOpts.spatialKernal(1)/2);
else
    extrapOpts.FFStrainPx = 5;
end
extrapOpts.FFDispMethod = 'linear';
extrapOpts.FFStrainMethod = 'linear';

%--------------------------------------------------------------------------
% Setup a plot parameters struct - cosmetic properties of diagnostic plots
plotParams.formatType = 'presentation';
plotParams.imageAxis = true;
plotParams.cutEdgePx = true;
plotParams.cutPxX = grid.pxPerPeriod+round(smoothingOpts.spatialKernal(1)/2);
plotParams.cutPxY = 0;

%% Post-Processing Experimental Data
% Smooth the displacement data and then calculate acceleration and strains
% Extrapolate the data to account for the missing pitch on the edges
% Use the smoothed data and the stress gauge approach/VFM to identify the
% in-plane stiffness components. 

%--------------------------------------------------------------------------
% Load the Reference Image and Determine Where the Free Edge is
fprintf('Obtaining the free edge location.\n')
[freeEdge,disp] = func_getFreeEdge(imagePath,imageFile,disp);

%--------------------------------------------------------------------------
% Smooth and Calculate Acceleration and Strain
fprintf('Calculating acceleration and strain from the displacement fields.\n')
[disp,vel,accel,strain,strainRate] = func_smoothCalcAccelStrain(...
    pos,time,material,grid,disp,smoothingOpts,extrapOpts,true);

%--------------------------------------------------------------------------
% STRESS GAUGE APPROACH:
fprintf('Calculating width averaged stress with the stress gauge approach.\n')
% Standard Stress Gauge Method
[stress.xAvg,~] = func_stressGaugeProcess(material,time,pos,accel);

% First Moment of the Stress Gauge
fprintf('Calculating the axial stress with the linear stress gauge approach.\n')
stress = func_linearStressGaugeProcess(specimen,material,pos,accel,stress);

%--------------------------------------------------------------------------
% STRESS GAUGE APPROACH: find Qxx by fitting the stress-strain curve
fprintf('Identifying Qxx by linearly fitting the stress-strain curves.\n')
% Set the options for fitting the stress strain curves
identOpts.fitCompOnly = true;
identOpts.defaultRange = 1:time.numFrames;
identOpts.method = 1;   % Uniaxial stress assumption
identOpts.avgQxxLRange = round(0.25*length(pos.x)):round(0.75*length(pos.x));
% Fit the stress strain curves to obtain Qxx
identStiffSG = func_identQxxFromStressStrainCurve(identOpts,stress,strain);
identStiffSG.ExxVsL = identStiffSG.QxxVsL;   % No poisson effect

% Calculate the average Ex over the specified range of the specimen
identStiffSG.ExxAvgOverL = nanmean(identStiffSG.ExxVsL(identOpts.avgQxxLRange));

%--------------------------------------------------------------------------
% FULL VFM: Simple Global VF, Uniaxial assumption
fprintf('Identifying Qxx with manual virtual fields.\n')
identOpts.avgOverTRange = 20:70; % approx 10us to 35us - stable for image def data
% Define manual virtual fields
virtualField.ux_star = @(x) x - specimen.length;
virtualField.epsx_star = @(x) 1;
identStiffVF.ExxVsT = func_manualVFforUniaxialTest(virtualField,pos,material,accel,strain);

% Calculate an average stiffness over the specified time range
identStiffVF.ExxAvgOverT = nanmedian(identStiffVF.ExxVsT(identOpts.avgOverTRange));

%//////////////////////////////////////////////////////////////////////////
% FULL VFM: Special Optimised VF, Uniaxial Assumption
fprintf('Identifying Qxx with reduced optimised virtual fields.\n')
identOpts.avgOverTRange = 20:70; % approx 10us to 35us - stable for image def data
virtualField.meshX = 5; % number of nodes in the x-direction for optimised VF
virtualField.meshY = 1; % number of nodes in the y-direction for optimised VF
identStiffVFOpt.ExxVsT = func_specialOptimisedVFforUniaxialTest(virtualField, pos, strain, accel, specimen, material);

% Calculate an average stiffness over the specified time range
identStiffVFOpt.ExxAvgOverT = nanmedian(identStiffVFOpt.ExxVsT(identOpts.avgOverTRange));
%//////////////////////////////////////////////////////////////////////////

fprintf('Identified stiffness values:\n')
IdentifiedStiffness_SG = identStiffSG
IdentifiedStiffness_VF =identStiffVF
IdentifiedStiffness_VFOpt =identStiffVFOpt

%% Plot Diagnostic Figures of the Post-Processing Results
fprintf('\n--------------------------------------------------------------\n')
fprintf('POST PROCESSING - Diagnostic Image Sequences\n')
fprintf('--------------------------------------------------------------\n')

%--------------------------------------------------------------------------
% Setup the plot parameters structure
plotParams.formatType = 'diagnostic';
plotParams.imageAxis = true;
plotParams.cutEdgePx = true;
plotParams.cutPxX = grid.pxPerPeriod+round(smoothingOpts.spatialKernal(1)/2);
plotParams.cutPxY = 0;
plotParams.cAxisType = 'Specified';

plotParams.cAxisDisp = [-0.25,0.25];
plotParams.cAxisAccel = [-6e6,6e6];
plotParams.cAxisStrain = [-20,15];
plotParams.cAxisRawStrain = [-20,20];
plotParams.cAxisStrainRate = [-3000,3000];

%--------------------------------------------------------------------------
% Create Video Sequences of Disp, Strain, Accel and Strain Rate
% NOTE: these images are required for the crack location identification
fprintf('Plotting image sequence of disp, accel, strain and strain rate.\n')
fprintf('NOTE: image sequence used to identify crack location for strength identification.\n')
func_plotAllFullFieldVideos(plotParams,imagePath,pos,time,disp,strain,strainRate,accel);

%--------------------------------------------------------------------------
% Plot all other diagnostic images
plotParams.cAxisType = 'Auto';
% Create path for saving the image sequence
savePath = [imagePath,'ProcessedData_DiagnosticFigs\'];
% Check if the save path exits, if it does ask the user if it
% needs to be plotted again
if exist(savePath,'file') == 7
    choice = questdlg('Folder for diagnostic figures exists, plot again?', ...
    'Plot Image Sequence?', 'Yes','No','No');
    switch choice
        case 'Yes'
            plotDiagFigs = true;
        case 'No'
            plotDiagFigs = false;
    end
else
    % If the directory does not exist, create it and plot the image seq
    mkdir(savePath);
    plotDiagFigs = true;
end

if plotDiagFigs
    plotAllDiagFigs = false;
    if plotAllDiagFigs
        fprintf('Ploting diagnostic figures.\n')
        % Plot full-field heat maps: disp and strain in x direction
        plotParams.frames = ceil(time.numFrames.*[0.1,0.3,0.5,0.6,0.7,0.9]);
        plotParams.Rows = 2;
        plotParams.Cols = 3;
        func_plotAllFullFieldMaps(plotParams,savePath,pos,time,disp,accel,strain,strainRate);

        % Plot the input pulse from the average accel over the full-field
        hf = func_plotInputPulse(time,specimen,accel);
        print(hf,[savePath,'\','InputPulse'],'-dpng','-r0')

        % Plot average over width data vs time for different x locs
        fprintf('\tPlotting average width data vs time for different specimen locations.\n')
        plotParams.locXmm = grid.pitch/2+grid.pitch*round(grid.numXPeriods*[0.1,0.2,0.3,0.4,0.5,0.6,0.7,0.8]);
        plotParams.Rows = 2;
        plotParams.Cols = 4;
        func_plotAllAvgDataVsTime(plotParams,savePath,time,pos,disp,accel,strain,stress)

        % Plot average over width data vs length for different frames/times
        fprintf('\tPlotting average width data vs length for different frames/times.\n')
        plotParams.frames = round(time.numFrames.*[0.1,0.25,0.5,0.6,0.7,0.8,0.9,1]);
        plotParams.Rows = 2;
        plotParams.Cols = 4;
        func_plotAllAvgDataVsLength(plotParams,savePath,time,pos,disp,accel,strain,stress)

        % Close some windows to free some RAM.
        close all
    end
    
    % Plot field averages over time on a single plot
    fprintf('\tPlotting field averages over time.\n')
    plotParams.Rows = 2;
    plotParams.Cols = 2;
    plotParams.tRange = 1:time.numFrames;
    hf = func_plotFieldAvgVsTime(plotParams,time,pos,disp,accel,strain,strainRate);
    saveFile = [savePath,'\','FieldAveragesVsTime'];
    print(hf,saveFile,'-dpng','-r0')
    saveas(hf,saveFile,'fig')
    
    % Plot stress-strain curves at several locations
    fprintf('\tPlotting stress-strain curves.\n')
    plotParams.Rows = 2;
    plotParams.Cols = 4;
    plotParams.locXmm = grid.pitch/2+grid.pitch*round(grid.numXPeriods*[0.1,0.2,0.3,0.4,0.5,0.6,0.7,0.8]);
    labelStrs.x = 'Avg Strain, \epsilon_{x} (mm/m)';
    labelStrs.y = 'Avg Stress, \sigma_{x} (MPa)';
    plotParams.tRange = 1:time.numFrames;
    hf = func_plotStressStrainCurves(labelStrs,plotParams,time,{pos},{stress},{strain},1);
    saveFile = [savePath,'\','Full_StressStrainCurves'];
    print(hf,saveFile,'-dpng','-r0')
    saveas(hf,saveFile,'fig')

    % Plot identified Qxx vs length
    fprintf('\tPlotting identified Qxx over the length of the specimen.\n')
    QxxPlotOpts.targPc = 0.1;
    QxxPlotOpts.rangePc = 5*QxxPlotOpts.targPc;
    QxxPlotOpts.method = 1;
    QxxPlotOpts.plotAverage = true;
    hf = func_plotQxxVsLength(QxxPlotOpts,pos,material,identStiffSG);
    saveFile = [savePath,'\','QxxVsLength'];
    print(hf,saveFile,'-dpng','-r0')
    saveas(hf,saveFile,'fig')
    
     % Plot identified Qxx vs length
    fprintf('\tPlotting identified Qxx over time using manual global VFs.\n')
    QxxPlotOpts.targPc = 0.1;
    QxxPlotOpts.rangePc = 10*QxxPlotOpts.targPc;
    QxxPlotOpts.method = 1;
    QxxPlotOpts.plotAverage = true;
    hf = func_plotQxxVsTime(QxxPlotOpts,time,material,identStiffVF);
    saveFile = [savePath,'\','QxxVsTime_ManGlobVF'];
    print(hf,saveFile,'-dpng','-r0')
    saveas(hf,saveFile,'fig')
    
    % Plot identified Qxx vs length
    fprintf('\tPlotting identified Qxx over time using special optimised VFs.\n')
    QxxPlotOpts.targPc = 0.1;
    QxxPlotOpts.rangePc = 10*QxxPlotOpts.targPc;
    QxxPlotOpts.method = 1;
    QxxPlotOpts.plotAverage = true;
    hf = func_plotQxxVsTime(QxxPlotOpts,time,material,identStiffVFOpt);
    saveFile = [savePath,'\','QxxVsTime_SpecOptVF'];
    print(hf,saveFile,'-dpng','-r0')
    saveas(hf,saveFile,'fig')
end

%% Strength Identification
fprintf('\n--------------------------------------------------------------\n')
fprintf('POST PROCESSING - Strength Identification\n')
fprintf('--------------------------------------------------------------\n')
plotParams.formatType = 'presentation';
% Create path for saving the image sequence
strIdentSavePath = [imagePath,'ImageSeq_StrengthIdentification\'];

%--------------------------------------------------------------------------
% Identify Fracture Location
% TODO: check if a fracture file already exists, if so load it. Otherwise,
% find the fracture location

% Check if the fracture file exists, if it does ask to load it
loadFractFile = false;
fractFile = 'fracture.mat';
fractFilePath = imagePath;
if exist([fractFilePath,fractFile],'file') == 2
    % Load the file and present the parameters:
    load([fractFilePath,fractFile])
    dlgStr = {'Fracture file found with:',...
        ['locX: ',sprintf('%i',fracture.locX)],...
        ['locY: ',sprintf('%i',fracture.locY)],...
        'Load file?'};
    
    choice = questdlg(dlgStr,'Load fracture file?', 'Yes','No','No');
    switch choice
        case 'Yes'
            loadFractFile = true;
        case 'No'
            loadFractFile = false;
            clear fracture
    end
end

% If the fracture location doesn't already exist, let the user input it
if ~loadFractFile
    % Check if the user wants to manually input this data or not
    inputFractLoc = false;
    choice = questdlg('Manually input fracture location and time?', ...
    'Process Grid Images', 'Manual','Graphic Pick','Manual');
    switch choice
        case 'Manual'
            inputFractLoc = true;
        case 'Graphic Pick'
            inputFractLoc = false;
    end

    if inputFractLoc
        fprintf('Obtaining fracture location (x,y) from user input by manual input.\n')
        fracture = func_manualInputFractureLoc(pos,time);
    else
        fprintf('Obtaining fracture location (x,y) from user input by picking.\n')
        fracture = func_findFractureLoc(imagePath,pos,time,disp,accel,strain,strainRate);
    end
    % Save the fracture location to file
    save([fractFilePath,fractFile],'fracture')
end


%--------------------------------------------------------------------------
% Create the Virtual Gauge for 'Local' strength identification
virtualGauge.Xpx = 10;
virtualGauge.Ypx = 20;
virtualGauge = func_createVirtualGauge(virtualGauge,fracture,strain);

% Specify the stiffness values for the stress reconstruction from strain 
strIdentStiffness.method = 1; % Neglect the poisson effect
strIdentStiffness.Exx = (identStiffSG.ExxAvgOverL+identStiffVFOpt.ExxAvgOverT)/2;
strIdentStiffness.nuxy = 0;
identStrength.strIdentStiffness = strIdentStiffness;

% Specify the method for determining the strength
% Manual - allows user to specify strength frame
% Automated - takes the maximum value of the stress gauge
identStrength.method = 'Automated';

%--------------------------------------------------------------------------
% Plot Diagnostic Image Sequences for the Selected Fracture Location
fprintf('Plotting diagnostic image sequence at selected fracture location.\n')
%func_plotIdentStrengthVideo(plotParams,imagePath,pos,time,disp,accel,...
%    strain,strainRate,stress,fracture,identStiffSG)
plotParams.stressLims = [-175,75];
func_plotIdentStrengthVideo_v2(plotParams,strIdentSavePath,pos,time,disp,accel,...
    strain,strainRate,stress,fracture,strIdentStiffness);

%--------------------------------------------------------------------------
% Plot the difference between the acceleration and strain stress measures
fprintf('Plotting the difference between the stress gauge and stress from strain vs time.\n\n');
func_plotStressDiffVsTime(plotParams,strIdentSavePath,time,virtualGauge,...
    strain,stress,fracture,strIdentStiffness);

% Grab the average stress and strain in the gauge area
identStrength.avgStressGA = squeeze(mean(mean(...
    stress.xLinearGauge(virtualGauge.yRange,virtualGauge.xRange,:)))); 
identStrength.avgStrainGA = squeeze(mean(mean(...
    strain.x(virtualGauge.yRange,virtualGauge.xRange,:))));
identStrength.avgStressFromStrainGA = identStrength.avgStrainGA.*strIdentStiffness.Exx;

%--------------------------------------------------------------------------
% Manually or Automatically Identify the Strength
if strcmp(identStrength.method,'Manual')
    % MANUAL Strength Identification
    fprintf('ANALYSE STRESS DIFFERENCE TO DETERMINE FRACTURE POINT.\n');
    fprintf('Press any key to continue.\n\n');
    pause;

    % Get the Fracture Frame/Time
    fprintf('Obtaining fracture frame from user input by manual input.\n')
    fracture.strengthFrame = func_manualInputFractureFrame(pos,time);

    % Identify strength
    fprintf('Identifying strength from stress gauge and strains using the virtual gauge area.\n')
    [identStrength.stressGauge,identStrength.linStressGauge,identStrength.fromStrain] =...
        func_identifyStrength(time,virtualGauge,fracture,stress,strain,strainRate,strIdentStiffness);
else
    % AUTOMATED Strength Identification
    [identStrength.stressGauge,identStrength.linStressGauge,fracture] = ...
            func_identifyStrengthAutomated(virtualGauge,fracture,stress);
    identStrength.fromStrain = -1;
end
 
%--------------------------------------------------------------------------
% Plot the strain rate at the fracture location up to the fracture frame
fprintf('Obtaining fracture frame from user input by manual input.\n')
strainRate.strStrainRate = func_plotStrainRateAtFracLoc(plotParams,strIdentSavePath,time,virtualGauge,...
    fracture,strainRate);

% Update the maximum strain rate variables
if smoothingOpts.spatSmooth == true
    edgePx = grid.pxPerPeriod+round(smoothingOpts.spatialKernal(1)/2)+1;
else
    edgePx = grid.pxPerPeriod+1;
end
xRange = edgePx:size(strainRate.xAvg,1) - edgePx;
tRange = 1:fracture.strengthFrame;
% Ignore the edges where the data is corrupted by smoothing effects
strainRate.xAvgMinPreFract = min(min(strainRate.xAvg(xRange,tRange)));
strainRate.xAvgMaxPreFract = max(max(strainRate.xAvg(xRange,tRange)));

%--------------------------------------------------------------------------
% Write identified values to the console
fprintf('\nIdentified strength values:\n')
Strength_FracLoc = fracture
Strength_StressGauge = identStrength.stressGauge
Strength_LinearGauge = identStrength.linStressGauge
Strength_fromStrain = identStrength.fromStrain

fprintf('Strength Identification COMPLETE.\n')
close all

%% Save All Processing Config Variables to File
choice = questdlg('Save processed data to file? (takes a long time)', ...
'Plot Image Sequence?', 'Yes','No','No');
switch choice
    case 'Yes'
        saveProcessedData = true;
    case 'No'
        saveProcessedData = false;
end

if saveProcessedData
    fprintf('\n--------------------------------------------------------------\n')
    fprintf('SAVING DATA\n')
    fprintf('--------------------------------------------------------------\n')
    %--------------------------------------------------------------------------
    % Save a .mat and readme file of the processed data
    % create filenames for saving data and config files
    fprintf('Saving processed data to README and mat files...\n')
    today = date;
    % Setup path and various file names for saving
    saveFiles.path = imagePath;
    saveFiles.processedData = 'AllPostProcessedData.mat';
    saveFiles.configFile = 'ConfigVariables.mat';
    saveFiles.configVars = [2:6,17:20];
    saveFiles.processedVars = 5:16;
    saveFiles.README = strcat('README_',today,'.txt');
    func_saveAllData(saveFiles,test,specimen,material,time,pos,disp,accel,strain,strainRate,stress,...
        identStiffSG,identStiffVF,identStiffVFOpt,identStrength,fracture,gridMethodOpts,smoothingOpts,extrapOpts,identOpts) % Var num 12 starts on this row

    fprintf('All data files saved.\n')
end

%% Print Identified Properties to Screen
fprintf('\n--------------------------------------------------------------\n')
fprintf('IDENTIFIED MATERIAL PROPERTIES\n')
fprintf('--------------------------------------------------------------\n')

fprintf('Identified stiffness values:\n')
IdentifiedStiffness_SG = identStiffSG
IdentifiedStiffness_VF =identStiffVF
IdentifiedStiffness_VFOpt =identStiffVFOpt

fprintf('Identified strength values:\n')
Strength_FracLoc = fracture
Strength_StrainRate = strainRate
Strength_StressGauge = identStrength.stressGauge
Strength_LinearGauge = identStrength.linStressGauge
Strength_fromStrain = identStrength.fromStrain

% Create condensed info on the strain rate
identStrainRate.xAvgMin = strainRate.xAvgMin;
identStrainRate.xAvgMax = strainRate.xAvgMax;
identStrainRate.xAvgMinPreFract = strainRate.xAvgMinPreFract;
identStrainRate.xAvgMaxPreFract = strainRate.xAvgMaxPreFract;
identStrStrainRate = strainRate.strStrainRate;

save([imagePath,'IdentifiedProperties.mat'],'identStiffSG','identStiffVF','identStiffVFOpt','fracture',...
    'virtualGauge','identStrength','identStrainRate','identStrStrainRate');

close all
fprintf('\nCOMPLETE.\n\n')