%% Gas Gun Designer - Simulink Model - v 1.0
% Created by: Lloyd Fletcher, PhotoDyn Research Group
% Date Edited: 14/9/2018
%
% This program runs a simulink model of a pneumatic gas gun to determine
% the pressure velocity curve. The opening area curve of the firing valve
% is modelled as linear. Further details of the model can be found in:
% GasGunSimulinkModel.slx

clc;
clear all;
format compact;

%% Load the simulink model
mdl = 'GasGunSimulinkModel';
isModelOpen = bdIsLoaded(mdl);
open_system(mdl);

%% Create Simulation Parameters
% Create variables for multiple simulations
firingTime = 0.5;                       % s, time at which the valve opening signal is sent

% Projectile Parameters:
projectileMass = 0.38;                 % kg

% Barrel Parameters:
barrelLength = 3;                       % m
barrelDiameter = 50e-3;                 % m
barrelCSA = 0.25*pi*barrelDiameter^2;   % m^2

% Valve Parameters:
valveDiameter = 50e-3;                  % m                  
valveCSA = 0.25*pi*valveDiameter^2;     % m^2
valveTime = 200e-3;                     % s
valveSlope = valveCSA/valveTime;        % m^2/s
valveCd = 0.64;

% Pressure Reservoir Parameters:
vesselVolume = 20*10^-3;                % m^3
P0_sweep = 0.25:0.25:8;                 % reservoir pressure (bar)
iterations = length(P0_sweep);

% pre-allocate sim storage variable
simout(iterations)  = Simulink.SimulationOutput;

% Push all Parameters onto the Model
set_param([mdl '/Projectile'],'mass',num2str(projectileMass));
set_param([mdl '/Barrel'],'Area',num2str(barrelCSA));
set_param([mdl '/Firing Valve'],'Cd',num2str(valveCd));
set_param([mdl '/Pressure Vessel'],'Volume',num2str(vesselVolume));

set_param([mdl '/Valve Opening'],'Slope',num2str(valveSlope));
set_param([mdl '/Valve Limit'],'UpperLimit',num2str(valveCSA));

%% Run the simulation for each parameter
for i = 1:iterations
    load_system(mdl);
    set_param([mdl '/Pressure Source (bar)'],'pressure_difference_reference',num2str(P0_sweep(i)));
    simout(i) = sim(mdl,'SimulationMode','normal','StopTime', '1.0');
end
clc

%% Process Simulation Data
% Obtain the velocity at the length of the barrel, or the largest
% projectile velocity if the piston does not reach the barrel length
for i = 1:iterations
    si = simout(i);
    projVel= si.get('simlog').Projectile.v.series.values;
    projDisp = si.get('simlog').Barrel.position.series.values;
    time = si.get('simlog').Barrel.position.series.time;
    
    % Remove the 0's before the firing valve opens
    mask = projVel>1;
    projVel = projVel(mask);
    projDisp = projDisp(mask);
    time = time(mask);
    
    % Find the maximum velocity over the simulation time and mask 
    [maxVal,ind] = max(projVel);
    projDisp = projDisp(1:ind);
    projVel = projVel(1:ind);
    time = time(1:ind);
    
    % If the projectile makes it out of the barrel, get the velocity at
    % the barrel length, if not get the max velocity
    if max(projDisp) > barrelLength
        exitVel(i) = interp1(projDisp,projVel,barrelLength);
    else
        exitVel(i) = max(projVel);
    end
end

%% Figure for Manual

%--------------------------------------------------------------------------
% Model Comparison
% Find the index of the pressure vector less than 3bar
pMask = logical(P0_sweep <= 3);

% Ideal model 100% efficiency
vesselVolume = 20*0.001; % 20 L
gamma = 1.4;
pVecIdeal = P0_sweep*10^5;
eta = 1;
vVecIdeal = eta*sqrt((2*pVecIdeal*vesselVolume)./(projectileMass*(gamma-1))*...
    (1-(vesselVolume/(vesselVolume+barrelLength*barrelCSA))^(gamma-1)));

% Plot formatting properties
plotProps.sizePerFigXcm = 10;
plotProps.sizePerFigYcm = 7;
plotProps.units = 'centimeters';
plotProps.hfs = 11;
plotProps.fs = 10;
plotProps.ft = 'times';

% Create the figure
hf = figure('Visible','On');
set(hf,'Resize','on','Units',plotProps.units,'PaperPositionMode','auto')
set(hf,'Position', [0,0,plotProps.sizePerFigXcm,plotProps.sizePerFigYcm])
set(hf,'Color',[1 1 1])

hold on
plot(pVecIdeal*10^-5,vVecIdeal,'-k')
plot(pVecIdeal*10^-5,0.75*vVecIdeal,'--k')
plot(pVecIdeal*10^-5,0.5*vVecIdeal,'-.k')
plot(P0_sweep,exitVel,'-b')
hold off
xlabel('Reservoir Pressure ($bar$)',...
    'fontsize',plotProps.hfs,'fontname',plotProps.ft,'interpreter','LaTeX')
ylabel('Projectile Velocity ($m.s^{-1}$)',...
    'fontsize',plotProps.hfs,'fontname',plotProps.ft,'interpreter','LaTeX')
legend({'$\eta=1$','$\eta=0.75$','$\eta=0.5$','Simulink'},...
    'location','northwest','interpreter','LaTeX','fontsize',8)
set(gca,'fontsize', plotProps.fs,'fontname',plotProps.ft)
set(gca,'XMinorTick','on','YMinorTick','on')
box on
grid on

