%% IMAGE BASED INERTIAL IMPACT (IBII) TEST : Data Processing - v1.0r
% Authors: Lloyd Fletcher, Jared Van-Blitterswyk
% PhotoDyn Group, University of Southampton
% http://photodyn.org/
% Date Created: 12th Dec. 2017 - v0.1
% Date Edited: 6th Jun. 2019 - v1.0r release
%
% Takes input IBII test images processes them with the grid method code 
% to obtain displacement fields. Further kinematic fields are derived from 
% the displacement fields (acceleration, strain) after applying smoothing 
% to the displacement fields. Kinematic fields are then used to identify 
% material properties including: stiffness and strength. This version
% includes two material models: reduced orthotropic and isotropic.
%
% This code uses the grid processing tool box that can be found at:
% www.thegridmethod.net, developed by Grediac et al.
%
% The following papers describe the IBII method:
%[1] L. Fletcher, J. Van-Blitterswyk, F. Pierron, A Novel Image-Based 
%   Inertial Impact Test (IBII) for the Transverse Properties of 
%   Composites at High Strain Rates, J. Dynamic Behavior Mater. (2019). 
%   doi:10.1007/s40870-019-00186-y.
%[2] L. Fletcher, F. Pierron, An image-based inertial impact (IBII) test 
%   for tungsten carbide cermets, J. Dynamic Behavior Mater. 4 (2018) 
%   481504. doi:10.1007/s40870-018-0172-4.
%[3] J. Van Blitterswyk, L. Fletcher, F. Pierron, Image-based inertial 
%   impact test for composite interlaminar tensile properties, J. 
%   Dynamic Behavior Mater. 4 (2018) 543572. doi:10.1007/s40870-018-0175-1.
%
% This work is licensed under the Creative Commons Attribution-
% NonCommercial-ShareAlike 4.0 International License. To view a copy of 
% this license, visit http://creativecommons.org/licenses/by-nc-sa/4.0/ or 
% send a letter to Creative Commons, PO Box 1866, Mountain View, CA 94042, 
% USA.
%
% If there are any issues with this code please contact: 
% Lloyd Fletcher: l.c.fletcher@soton.ac.uk / lloydcolinfletcher@gmail.com 

clc
clear all
close all

fprintf('--------------------------------------------------------------\n')
fprintf('IMAGE-BASED INERTIAL IMPACT (IBII): Data Processing v1.0r\n')
fprintf('--------------------------------------------------------------\n')

% Flag for processing pure FE data for validation purposes, bypassess the
% grid method processing module and directly loads FE kinematic data
FEValidMode = false;
calcKinFieldsFromDisp = ~FEValidMode;

%% INITIALISE: Add path for processing functions
fprintf('Adding pathes for processing functions.\n')
% Add the path for the grid method code and other useful functions:
funcPath = [pwd,'\Functions\'];

% If the default path is not found we should find it
if exist(funcPath,'file') ~= 7
    hWarn = warndlg('Folder for processing functions not found.','Function folder not found');
    waitfor(hWarn);
    funcPath = uigetdir(pwd,'Locate Processing Function Folder');
end
addpath(funcPath);
addpath([funcPath,'GridMethodToolbox\']);

%% LOAD RAW DATA FILES: Images or FE
hardCodePath = false; % Useful for running a single file repeatedly for debugging
if ~FEValidMode    
    % LOAD RAW DATA FILES: Raw .tiff images
    fprintf('Loading reference image from the selected test data folder.\n')
    if ~hardCodePath
        [imageFile,imagePath] = uigetfile({'*.*','All Files'},'Select the first image in the sequence');
    else   
        imageFile = 'DefGridImage_001.tiff';
        imagePath = 'E:\Matlab_WorkingDirectory\1_IBIITest_Data\TestData_ID_CFIPQIso_HalfPulse\'; 
    end
else
    % LOAD RAW DATA FILES: FE Validation Data
    fprintf('Loading FE data from selected .mat file.\n')
    if ~hardCodePath
        [FEDataFile,FEDataPath] = uigetfile({'*.*','All Files'},'Select the FE data file');
    else   
        FEDataFile = '';
        FEDataPath = [pwd,'\'];    
    end
    load([FEDataPath,FEDataFile])
    imagePath = FEDataPath;
    imageFile = FEDataFile;   
end

%% INITIALISE: Load Processing Parameter Data Structures
fprintf('Loading processing parameters file.\n')
initPath = imagePath;
initFile = 'processingParameters.mat';
if exist([initPath,initFile],'file') ~= 2
    hWarn = warndlg('Processing parameter file does not exist.','Processing parameters not found');
    waitfor(hWarn);
    [initFile,initPath,~] = uigetfile('*.mat','Locate processing parameter file');
end
% Load the processing parameters from file
load([initPath,initFile])

% Store the FE valid mode parameter
globalOpts.FEValidMode = FEValidMode;
globalOpts.calcKinFieldsFromDisp = calcKinFieldsFromDisp;

%% IMAGE PROCESSING: Use the Grid Method to extract displacement fields
if ~FEValidMode
    % Process the raw tiff images using the grid method code developed by
    % Grediac et al.
    fprintf('\n--------------------------------------------------------------\n')
    fprintf('GRID METHOD PROCESSING\n')
    fprintf('--------------------------------------------------------------\n')

    %--------------------------------------------------------------------------
    % GRID IMAGE PROCESSING

    % Check if the images have already been processed otherwise process the data
    gridDataSavePath = imagePath;
    gridDataFile = 'GridMethod_ProcessedData.mat';

    fprintf('Checking for existing processed data file.\n')
    processGridImages = true;
    if exist([gridDataSavePath,gridDataFile],'file') == 2
        if gridMethodOpts.autoLoadProccessedDataFile
            processGridImages = false;
        else   
            choice = questdlg('Processed grid data file found, process again?', ...
            'Process Grid Images', 'Yes','No','No');
            switch choice
                case 'Yes'
                    processGridImages = true;
                case 'No'
                    processGridImages = false;
            end
        end
    end

    if processGridImages
        fprintf('Processing images using the grid method toolbox.\n')
        % Process the image squence with the grid method toolbox
        [grid,pos,disp] = func_gridMethodImageProcessing(imagePath,imageFile,...
            grid,gridMethodOpts,imageNoise);    

        % Save the data to file so a reprocess is not necessary
        fprintf('Saving processed grid data to: %s.\n',gridDataFile)
        save([gridDataSavePath,gridDataFile],'grid','pos','disp') 

        fprintf('Grid Method Processing Complete.\n')
    else
        fprintf('Loading processed grid data from: %s.\n',gridDataFile)
            load([gridDataSavePath,gridDataFile]);
    end

    %--------------------------------------------------------------------------
    % Update Geometry and Number of Frames Based on Displacement Matrix Size
    fprintf('Updating geometry variables based on the size of displacement field matrices.\n')
    [specimen,grid] = func_updateSpecGeom(specimen,grid,disp);

    % Currently the rotations are unused so remove them to save RAM
    disp = rmfield(disp,'rot');
end

%--------------------------------------------------------------------------
% Create the time vector based on the number of frames in the disp struct
time.numFrames = size(disp.x,3);
time.vec = 0:time.step:(size(disp.x,3)-1)*time.step;

%% POST-PROCESSING: Smoothing and Kinematic Field Derivation
% Smooth the displacement data and then calculate acceleration and strains
% Extrapolate the data to account for the missing pitch on the edges

fprintf('\n--------------------------------------------------------------\n')
fprintf('POST PROCESSING: Smoothing and Kinematic Field Calculation\n')
fprintf('--------------------------------------------------------------\n')
% Calculate the kinematic fields from displacement fields using
% displacements from images or displacements from FE data
if calcKinFieldsFromDisp
    %--------------------------------------------------------------------------
    % Load the Reference Image and Determine Where the Free Edge is
    fprintf('Obtaining and setting the free edge location.\n')
    [freeEdge,specimen,disp] = func_getFreeEdge(globalOpts.hardCodeFreeEdge,...
        imagePath,imageFile,specimen,disp);

    %--------------------------------------------------------------------------
    % Smooth and Calculate Strain
    fprintf('Calculating strain from the displacement fields.\n')
    [disp,strain,strainRate] = func_smoothCalcStrain(globalOpts,pos,time,...
        grid,disp,smoothingOpts,extrapOpts);
    
    %--------------------------------------------------------------------------
    % Smooth and Calculate Acceleration
    fprintf('Calculating acceleration from the displacement fields.\n')
    [disp,~,accel] = func_smoothCalcAccel(pos,time,grid,disp,smoothingOpts,...
        extrapOpts,diffOpts);
    
    % Remove some 3D fields from the structs to save memory.
    if globalOpts.reduceMemory
        disp.x = disp.extrap.x;
        disp.y = disp.extrap.y;
        disp = rmfield(disp,'extrap');
        disp = rmfield(disp,'tSmooth');
    end
else
    fprintf('Using kinematic fields directly from FE data.\n')
    % For pure FE data the kinematic fields can be taken directly from the
    % FE calculation
    disp.xAvg = func_avgFFVarOverWidth(disp.x);
    accel.xAvg = func_avgFFVarOverWidth(accel.x);
    strain.xAvg = func_avgFFVarOverWidth(strain.x);
    strain.yAvg = func_avgFFVarOverWidth(strain.y);
    [strainRate,~] = func_calculateStrainRate(strain,time,smoothingOpts,true);
    strainRate.xAvg = func_avgFFVarOverWidth(strainRate.x);
    strainRate = func_calcMaxStrainRate(smoothingOpts,grid,strainRate);
    disp.extrap.x = disp.x;
    disp.extrap.y = disp.y;
end

%% POST-PROCESSING - Image Sequences of Kinematic Fields
fprintf('\n--------------------------------------------------------------\n')
fprintf('POST PROCESSING: Image Sequences\n')
fprintf('--------------------------------------------------------------\n')
plotProps = func_initPlotPropsStruct(plotParams.formatType);

%--------------------------------------------------------------------------
% Create Video Sequences of Disp, Strain, Accel and Strain Rate
% NOTE: these images are required for the crack location identification
if strcmp(globalOpts.plotImageSeqs,'no')
    fprintf('Image sequence plotting disabled.\n')
else
    fprintf('Plotting image sequence of disp, accel, strain and strain rate.\n')
    fprintf('NOTE: image sequence used to identify crack location for failure stress identification.\n')
    func_plotAllFullFieldVideos(globalOpts,plotParams,imagePath,pos,time,...
        disp,strain,strainRate,accel);
end
close all

%% POST-PROCESSING: VFM Stiffness Identification
% Use the smoothed data and the VFM to identify the in-plane stiffness. 
fprintf('\n--------------------------------------------------------------\n')
fprintf('POST PROCESSING: VFM Stiffness Identification\n')
fprintf('--------------------------------------------------------------\n')

%--------------------------------------------------------------------------
% VFM: MANUAL
fprintf('Identifying stiffnesses with manual virtual fields.\n')
fprintf('Selected material model is: %s.\n',globalOpts.matModel) 

if strcmp('isotropic',globalOpts.matModel)
    % Create the virtual fields
    VFs = func_VFDynInitManIsoLinElas(VFOpts,pos,accel,strain);
    % Use the virtual fields for stiffness identification
    identStiffVFMan = func_VFDynManIsoLinElas(VFOpts,pos,time,material,...
        VFs,strain,accel);

    % Calculate the median to work out the 'average' identified value
    identStiffVFMan.QxxAvgOverT = nanmean(identStiffVFMan.QxxVsT(VFOpts.avgQVsTRange));
    identStiffVFMan.QxyAvgOverT = nanmean(identStiffVFMan.QxyVsT(VFOpts.avgQVsTRange));
    identStiffVFMan.ExxAvgOverT = nanmean(identStiffVFMan.ExxVsT(VFOpts.avgQVsTRange));
    identStiffVFMan.NuxyAvgOverT = nanmean(identStiffVFMan.NuxyVsT(VFOpts.avgQVsTRange));

elseif strcmp('orthotropicReduced',globalOpts.matModel)
    % Process kinematic data with manual virtual fields
    identStiffVFMan.ExxVsT = func_VFDynManReducedLinElas(VFOpts,pos,material,accel,strain);
    
    % Calculate an average stiffness over the specified time range
    identStiffVFMan.ExxAvgOverT = nanmean(identStiffVFMan.ExxVsT(VFOpts.avgQVsTRange));    
    
    % For this case Qxx approx Exx
    identStiffVFMan.QxxVsT = identStiffVFMan.ExxVsT;
    identStiffVFMan.QxxAvgOverT = identStiffVFMan.ExxAvgOverT;
    
elseif strcmp('orthotropic',globalOpts.matModel)
    fprintf('WARNING: orthtropic manual virtual fields have not been implemented.\n')
    identStiffVFMan = nan;
    
elseif strcmp('orthotropicAngle',globalOpts.matModel)
    fprintf('WARNING: angled orthtropic manual virtual fields have not been implemented.\n')
    identStiffVFMan = nan;
else
    fprintf('WARNING: specifed material model not recognised.\n')
    identStiffVFMan = nan;
end 

%--------------------------------------------------------------------------
% VFM: PIECE-WISE SPECIAL OPTIMISED
if globalOpts.processOptVF
    fprintf('Identifying stiffnesses with piece-wise optimised virtual fields.\n') 
    fprintf('Selected material model is: %s.\n',globalOpts.matModel) 
    
    if strcmp('isotropic',globalOpts.matModel)
        % Use isotropic virtual fields to get Qxx and Qxy
        fprintf('Identifying Qxx and Qxy with optimised virtual fields.\n')
        [identStiffVFOpt,VFOptDiag] = func_VFDynPWSpecOptIsoLinElas(VFOpts,...
            pos,specimen,material,accel,strain);

        % Calculate the median to work out the 'average' identified value
        identStiffVFOpt.QxxAvgOverT = nanmean(identStiffVFOpt.QxxVsT(VFOpts.avgQVsTRange));
        identStiffVFOpt.QxyAvgOverT = nanmean(identStiffVFOpt.QxyVsT(VFOpts.avgQVsTRange));
        identStiffVFOpt.ExxAvgOverT = nanmean(identStiffVFOpt.ExxVsT(VFOpts.avgQVsTRange));
        identStiffVFOpt.NuxyAvgOverT = nanmean(identStiffVFOpt.NuxyVsT(VFOpts.avgQVsTRange));

    elseif strcmp('orthotropicReduced',globalOpts.matModel)
        % Use reduced optimised virtual fields to obtain Qxx
        fprintf('Identifying Qxx with reduced optimised virtual fields.\n')
        [identStiffVFOpt,VFOptDiag] = func_VFDynPWSpecOptReducedLinElas(VFOpts,...
            pos,strain,accel,specimen, material);

        % Calculate an average stiffness over the specified time range
        identStiffVFOpt.ExxAvgOverT = nanmean(identStiffVFOpt.ExxVsT(VFOpts.avgQVsTRange));
        
        % For this case Qxx is approx Exx
        identStiffVFOpt.QxxVsT = identStiffVFOpt.ExxVsT;
        identStiffVFOpt.QxxAvgOverT = identStiffVFOpt.ExxAvgOverT;
        
    else
        fprintf('WARNING: specifed material model not recognised.\n')
        identStiffVFOpt = nan;
    end
else
    fprintf('Optimised virtual fields processing disabled.\n')
    identStiffVFOpt = nan;
end

%% POST-PROCESSING: Stress Metrics and Stiffness Identification
% Use the smoothed data and perform stress gauge calculations for stiffness
% and failure stress identification. 
fprintf('\n--------------------------------------------------------------\n')
fprintf('POST PROCESSING: Stress-Strain Curves\n')
fprintf('--------------------------------------------------------------\n')

%--------------------------------------------------------------------------
% STANDARD STRESS GAUGE: CALCULATION
fprintf('Calculating average stress with the stress-gauge equation.\n')
% Standard Stress Gauge equation calculation
[stress.xAvg,~] = func_stressGaugeProcess(material,time,pos,accel.xAvg);

% First moment of the stress-gauge
fprintf('Calculating the average stress with the linear stress-gauge equation.\n')
stress = func_linearStressGaugeProcess(specimen,material,pos,accel,stress);

%--------------------------------------------------------------------------
% STANDARD STRESS GAUGE: find stiffness(es) by fitting the stress-strain curves
fprintf('Identifying stiffness(es) by fitting the stress-strain curves.\n')
if strcmp('orthotropicReduced',globalOpts.matModel)
    fprintf('Linearly fitting the stress and strain (e_xx) components to obtain Qxx.\n')
    % Fit the stress strain curves to obtain Qxx/Exx
    % identStiffSG = func_identQxxFromStressStrainCurve(stressGaugeOpts,stress,strain);
    [identStiffSG.QxxVsL,identStiffSG.QxxLinFitCoeffs,identStiffSG.QxxFitFrameRange]...
        = func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stress.xAvg,strain.xAvg);
    
    % Set the range over which the average stiffness is identified
    stressGaugeOpts.avgQVsLRange = round(stressGaugeOpts.avgQVsLRangePc(1)*length(pos.x))...
        :round(stressGaugeOpts.avgQVsLRangePc(2)*length(pos.x));
    % Calculate Identified Averages
    identStiffSG.QxxAvgOverL = nanmean(identStiffSG.QxxVsL(stressGaugeOpts.avgQVsLRange));
    identStiffSG.ExxVsL = identStiffSG.QxxVsL;
    identStiffSG.ExxAvgOverL = nanmean(identStiffSG.ExxVsL(stressGaugeOpts.avgQVsLRange));
    
else % Default to isotropic material model
    fprintf('Linearly fitting the stress and strain (e_xx + nu_xy*e_yy) components to obtain Qxx.\n')
    % Calculate the axial strain including the poisson effect
    if strcmp('VFOpt',stressGaugeOpts.strainCalcNuxy)
        identStiffSG.strainCalcNuxy = identStiffVFOpt.NuxyAvgOverT;
    elseif strcmp('VFMan',stressGaugeOpts.strainCalcNuxy)
        identStiffSG.strainCalcNuxy = identStiffVFMan.NuxyAvgOverT;
    else % Assume the QS value
        identStiffSG.strainCalcNuxy = material.nuxy;
    end
    strain.xnyAvg = strain.xAvg + identStiffSG.strainCalcNuxy*strain.yAvg;
    
    % Fit the stress strain curves to obtain Qxx
    [identStiffSG.QxxVsL,identStiffSG.QxxLinFitCoeffs,identStiffSG.QxxFitFrameRange] = ...
        func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stress.xAvg,strain.xnyAvg);
    
    % Set the range over which the average stiffness is identified
    stressGaugeOpts.avgQVsLRange = round(stressGaugeOpts.avgQVsLRangePc(1)*length(pos.x))...
        :round(stressGaugeOpts.avgQVsLRangePc(2)*length(pos.x));
    % Calculate Identified Averages
    identStiffSG.QxxAvgOverL = nanmean(identStiffSG.QxxVsL(stressGaugeOpts.avgQVsLRange));
    identStiffSG.ExxVsL = identStiffSG.QxxVsL.*(1-identStiffSG.strainCalcNuxy^2);
    identStiffSG.ExxAvgOverL = nanmean(identStiffSG.ExxVsL(stressGaugeOpts.avgQVsLRange));
end

%--------------------------------------------------------------------------
% Display the identified stiffnesses
fprintf('Identified stiffness values:\n')
IdentifiedStiffness_SG = identStiffSG
IdentifiedStiffness_VFMan =identStiffVFMan
IdentifiedStiffness_VFOpt =identStiffVFOpt

%% POST-PROCESSING: Diagnostic Figures for Stiffness Identification
fprintf('\n--------------------------------------------------------------\n')
fprintf('POST PROCESSING: Stiffness Identification Diagnostic Figures\n')
fprintf('--------------------------------------------------------------\n')

%--------------------------------------------------------------------------
% Plot Diagnostic Figures of Identified Stiffnesses
savePath = [imagePath,'DiagnosticFigures\'];
if strcmp(globalOpts.plotImageSeqs,'prompt')
    % Check if the save path exits, if it does ask the user if it
    plotDiagFigs = func_checkIfPathExistsDlg(savePath,...
        'Folder for diagnostic figures exists, plot again?','Plot Diagnostics?');
elseif strcmp(globalOpts.plotImageSeqs,'yes') || strcmp(globalOpts.plotImageSeqs,'diagnostic')
    plotDiagFigs = true;
    if exist(savePath,'file') ~= 7
        mkdir(savePath);
    end
else
    plotDiagFigs = false; 
end

if plotDiagFigs
    fprintf('Ploting diagnostic figures.\n')
    
    %----------------------------------------------------------------------
    % Plot General Diagnostics: Fields Averages and Loading Pulse
    func_plotGeneralFieldDiagnostics(savePath,plotParams,...
        specimen,time,pos,disp,accel,strain,strainRate,stress)
    
    %----------------------------------------------------------------------
    % MANUAL VF: Stiffness against time
    fprintf('\tPlotting identified stiffness components over time with manual virtual fields.\n')
    if strcmp('orthotropicReduced',globalOpts.matModel)
        func_plotAllIdentStiffVFManRedOrtho(savePath,plotParams,time,material,...
            VFPlotOpts,identStiffVFMan,VFOpts)
    else % Isotropic case by default 
        func_plotAllIdentStiffVFManIso(savePath,plotParams,time,material,...
            VFPlotOpts,identStiffVFMan,VFOpts)
    end
    
    %----------------------------------------------------------------------
    % OPTIMISED VF: Stiffness diagnostic vs time
    if globalOpts.processOptVF
        fprintf('\tPlotting identified stiffness components over time with optimised virtual fields.\n')
        if strcmp('orthotropicReduced',globalOpts.matModel)
            func_plotAllIdentStiffVFOptRedOrtho(savePath,plotParams,time,material,...
                VFPlotOpts,identStiffVFOpt,VFOpts) 
        else % Isotropic case by default
            func_plotAllIdentStiffVFOptIso(savePath,plotParams,time,material,...
                VFPlotOpts,identStiffVFOpt,VFOpts)
        end
    end
    
    %----------------------------------------------------------------------
    % STRESS-GAUGE: Plot stress-strain curves at several locations
    fprintf('\tPlotting stress-strain curves.\n')
    if strcmp('orthotropicReduced',globalOpts.matModel)
        func_plotSSCurvesSGRedOrtho(savePath,plotParams,ssCurvePlotParams,...
            pos,stress,strain,identStiffSG)
    else % Isotropic case as default for stress-gauge
        func_plotSSCurvesSGIso(savePath,plotParams,ssCurvePlotParams,...
            pos,stress,strain,identStiffSG)
    end
    
    %----------------------------------------------------------------------
    % STRESS-GAUGE: Plot identified Stiffness vs Length
    fprintf('\tPlotting identified stiffness over the length of the specimen.\n')
    ssCurveIdentPlotOpts.indLRange = 1:length(pos.x);

    if strcmp('orthotropicReduced',globalOpts.matModel)
        func_plotSGIdentStiffVsLOrthoRed(savePath,plotParams,ssCurveIdentPlotOpts,...
            pos,material,identStiffSG,stressGaugeOpts)
    else % Isotropic case as default for stress-gauge
        func_plotSGIdentStiffVsLIso(savePath,plotParams,ssCurveIdentPlotOpts,...
            pos,material,identStiffSG,stressGaugeOpts)
    end
end
close all

%% POST-PROCESSING: Strength Identification
if globalOpts.identifyStrength
    fprintf('\n--------------------------------------------------------------\n')
    fprintf('POST PROCESSING: Strength Identification\n')
    fprintf('--------------------------------------------------------------\n')
    
    %--------------------------------------------------------------------------
    % Obtain Fracture Location
    % Check if the fracture file exists, if it does ask to load it
    loadFractFile = false;
    fractFile = 'fracture.mat';
    fractFilePath = imagePath;
    if exist([fractFilePath,fractFile],'file') == 2
        if strOpts.autoLoadFractureFile
            loadFractFile = true;
            load([fractFilePath,fractFile])
        else
            % Load the file and present the parameters:
            load([fractFilePath,fractFile])
            dlgStr = {'Fracture file found with:',...
                ['locX: ',sprintf('%i',fracture.locX)],...
                ['locY: ',sprintf('%i',fracture.locY)],...
                'Load file?'};

            choice = questdlg(dlgStr,'Load fracture file?', 'Yes','No','No');
            switch choice
                case 'Yes'
                    loadFractFile = true;
                case 'No'
                    loadFractFile = false;
                    clear fracture
            end
        end
    end

    % If the fracture location doesn't already exist, let the user input it
    if ~loadFractFile
        % Check if the user wants to manually input this data or not
        inputFractLoc = false;
        choice = questdlg('Manually input fracture location and time?', ...
        'Fracture Location Input', 'Manual','Graphic Pick','Manual');
        switch choice
            case 'Manual'
                inputFractLoc = true;
            case 'Graphic Pick'
                inputFractLoc = false;
        end

        if inputFractLoc
            fprintf('Obtaining fracture location (x,y) from user input by manual input.\n')
            fracture = func_manualInputFractureLoc(pos,time);
        else
            fprintf('Obtaining fracture location (x,y) from user input by picking.\n')
            fracture = func_findFractureLoc(imagePath,pos,time,disp,accel,strain,strainRate);
        end
        % Save the fracture location to file
        save([fractFilePath,fractFile],'fracture')
    end

    %--------------------------------------------------------------------------
    % Iniatialise failure stress processing options
    strOpts.virtualGauge = func_createVirtualGauge(strOpts.virtualGauge,fracture,strain);
    
    % Depending on material model set the stiffness matrix for stres
    % calculation from strain
    if strcmp('isotropic',globalOpts.matModel)
        if strcmp('SG',strOpts.stiffnessMethod)
            strOpts.stiffnessQxx = identStiffSG.QxxAvgOverL;
            strOpts.stiffnessQxy = material.Qxy;
        elseif strcmp('VF',strOpts.stiffnessMethod)
            strOpts.stiffnessQxx = identStiffVFOpt.QxxAvgOverL;
            strOpts.stiffnessQxy = identStiffVFOpt.QxyAvgOverL; 
        else
            strOpts.stiffnessQxx = nanmean([identStiffSG.QxxAvgOverL,identStiffVFOpt.QxxAvgOverT]);
            strOpts.stiffnessQxy = identStiffVFOpt.QxyAvgOverT;
        end
        strOpts.stiffnessQMat = [strOpts.stiffnessQxx,strOpts.stiffnessQxy,0;...
                                     strOpts.stiffnessQxy,strOpts.stiffnessQxx,0;...
                                     0,0,(strOpts.stiffnessQxx-strOpts.stiffnessQxy)/2]; 
    elseif strcmp('orthotropicReduced',globalOpts.matModel)
        if strcmp('SG',strOpts.stiffnessMethod)
            strOpts.stiffnessQxx = identStiffSG.QxxAvgOverL;
        elseif strcmp('VF',strOpts.stiffnessMethod)
            strOpts.stiffnessQxx = identStiffVFOpt.QxxAvgOverL;
        else
            strOpts.stiffnessQxx = nanmean([identStiffSG.QxxAvgOverL,identStiffVFOpt.QxxAvgOverT]);
        end
        strOpts.stiffnessQMat = [strOpts.stiffnessQxx,0,0;...
                                 0,0,0;...
                                 0,0,0]; 
    else
        fprintf('WARNING: specifed material model not recognised.\n')
    end

    %--------------------------------------------------------------------------
    % Calculate the stress from the strains using stiffness matrix
    fprintf('Calculating stress from strain using the identified stiffness matrix.\n')
    [stress.QRecon.x,~,~] = func_calcStressFromStrain2D(strOpts.stiffnessQMat,...
        strain.x,strain.y,strain.s);
    
    fprintf('Calculating average stress and strain in the virtual gauge area at the fracture location.\n')
    [stress,strain] = func_calcAvgStressStrainInGaugeArea(globalOpts,strOpts,stress,strain);
    
    % Plot the difference between the acceleration and strain stress measures
    fprintf('Plotting the difference between the stress gauge and stress from strain vs time.\n');
    hf = func_plotStressDiffVsTime(plotParams,savePath,time,stress,fracture);
    saveFile = [savePath,'\','FailStrIdent_StressDiff']; 
    func_saveFigureMultFormat(hf,saveFile,plotProps,plotParams)

    %--------------------------------------------------------------------------
    % Manually or Automatically Identify the Strength
    if strcmp(strOpts.method,'Manual')
        % MANUAL Strength Identification
        % User manually specifies the time at which the stress and strain
        % measures diverge for failure stress identification
        fprintf('ANALYSE STRESS DIFFERENCE TO DETERMINE FRACTURE POINT.\n');
        fprintf('Press any key to continue.\n');
        pause;

        % Get the Fracture Frame/Time
        fprintf('Obtaining fracture frame from user input by manual input.\n')
        fracture.strengthFrame = func_manualInputFractureFrame(pos,time);

        % Identify failure stress
        fprintf('Identifying failure stress from stress gauge and strains using the virtual gauge area.\n')
        [identStrength.stressGauge,identStrength.linStressGauge,identStrength.fromStrain] =...
            func_identifyStrength(time,strOpts.virtualGauge,fracture,stress,strain,strainRate);
    else
        % AUTOMATED failure stress Identification
        [identStrength.stressGauge,identStrength.linStressGauge,fracture] = ...
                func_identifyStrengthAutomated(strOpts.virtualGauge,fracture,stress);
    end

    %--------------------------------------------------------------------------
    % DIAGNOSTIC IMAGE SEQ 1: Width averaged stress measures and fields
    fprintf('Plotting diagnostic image sequence 1: width average stress and fields.\n')
    plotParams.cutPxY = 0;

    strImageSeqSavePath = [imagePath,'ImageSeq5_FailStrIdent_WidthAvg\'];
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        plotImageSeq = func_checkIfPathExistsDlg(strImageSeqSavePath,...
            'Width averaged failure stress identification image sequence folder found, plot again?','Plot Image Sequence?');
    elseif strcmp(globalOpts.plotImageSeqs,'yes') || strcmp(globalOpts.plotImageSeqs,'diagnostic')
        plotImageSeq = true;
        if exist(strImageSeqSavePath,'file') ~= 7
            mkdir(strImageSeqSavePath);
        end
    else
        plotImageSeq = false; 
    end

    % Video of width average stress, raw strain, strain, accel, strain rate
    if plotImageSeq
        func_plotIdentStrVid_WAvg(plotParams,strImageSeqSavePath,pos,time,...
            disp,accel,strain,strainRate,stress,fracture);
    end

    %--------------------------------------------------------------------------
    % DIAGNOSTIC IMAGE SEQ 2: Stress fields and local stress measures/curve
    fprintf('Plotting diagnostic image sequence 2: stress fields and stress measures.\n')

    strImageSeqSavePath = [imagePath,'ImageSeq6_FailStrIdent_StressFields\'];
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        plotImageSeq = func_checkIfPathExistsDlg(strImageSeqSavePath,...
            'Stress field based failure stress identification image sequence folder found, plot again?','Plot Image Sequence?');
    elseif strcmp(globalOpts.plotImageSeqs,'yes') || strcmp(globalOpts.plotImageSeqs,'diagnostic')
        plotImageSeq = true;
        if exist(strImageSeqSavePath,'file') ~= 7
            mkdir(strImageSeqSavePath);
        end
    else
        plotImageSeq = false; 
    end

    % Video of stress fields from linear gauge and strain, local 
    % stress-strain curve and stress difference 
    if plotImageSeq
        if strcmp('isotropic',globalOpts.matModel)
            ssCurveLabels.xStr = '$\overline{\epsilon_{xx} + \nu \epsilon_{yy}}^{VG}$ [$mm.m^{-1}$]';
            ssCurveLabels.yStr = '$\overline{\sigma_{xx}}^{VG}$ [$MPa$]';
            func_plotIdentStrVid_Field(plotParams,strImageSeqSavePath,pos,time,...
                stress,stress.virtGaugeAvg.x,strain.virtGaugeAvg.xny,ssCurveLabels,...
                fracture,strOpts,false,false);

        elseif strcmp('orthotropicReduced',globalOpts.matModel)
            ssCurveLabels.xStr = '$\overline{\epsilon_{xx}}^{VG}$ [$mm.m^{-1}$]';
            ssCurveLabels.yStr = '$\overline{\sigma_{xx}}^{VG}$ [$MPa$]';
            func_plotIdentStrVid_Field(plotParams,strImageSeqSavePath,pos,time,...
                stress,stress.virtGaugeAvg.x,strain.virtGaugeAvg.x,ssCurveLabels,...
                fracture,strOpts,false,false);
        else
            fprintf('WARNING: specifed material model not recognised.\n')
        end 
    end
    
    %--------------------------------------------------------------------------
    % Plot the strain rate at the fracture location up to the fracture frame
    fprintf('Calculating maximum strain rate at the fracture location.\n')
    fprintf('Plotting the strain rate at the fracture location.\n')
    strainRate.strStrainRate = func_plotStrainRateAtFracLoc(plotParams,savePath,...
        time,strOpts.virtualGauge,fracture,strainRate);
    
    % Update the maximum strain rate variables
    if smoothingOpts.spatialSmooth == true
        edgePx = grid.pxPerPeriod+round(smoothingOpts.spatialKernal(1)/2)+1;
    else
        edgePx = grid.pxPerPeriod+1;
    end
    xRange = edgePx:size(strainRate.xAvg,1) - edgePx;
    tRange = 1:fracture.strengthFrame;
    % Ignore the edges where the data is corrupted by smoothing effects
    strainRate.xAvgMinPreFract = min(min(strainRate.xAvg(xRange,tRange)));
    strainRate.xAvgMaxPreFract = max(max(strainRate.xAvg(xRange,tRange)));
    identStrainRate.strStrainRate = strainRate.strStrainRate;
    
    %--------------------------------------------------------------------------
    % FINAL DIAG FIG: Plot the stress fields maps and local stress-strain curves
    fprintf('Plotting diagnostic images at fracture time: stress fields and stress measures.\n')
     if strcmp('isotropic',globalOpts.matModel)
        ssCurveLabels.xStr = '$\overline{\epsilon_{xx} + \nu \epsilon_{yy}}^{VG}$ [$mm.m^{-1}$]';
        ssCurveLabels.yStr = '$\overline{\sigma_{xx}}^{VG}$ [$MPa$]';
        hf = func_plotIdentStrVid_Field(plotParams,savePath,pos,time,...
            stress,stress.virtGaugeAvg.x,strain.virtGaugeAvg.xny,ssCurveLabels,...
            fracture,strOpts,true,false);
        saveFile = [savePath,'\','FailStrIdent_Diagnostics'];
        func_saveFigureMultFormat(hf,saveFile,plotProps,plotParams)

    elseif strcmp('orthotropicReduced',globalOpts.matModel)
        ssCurveLabels.xStr = '$\overline{\epsilon_{xx}}^{VG}$ [$mm.m^{-1}$]';
        ssCurveLabels.yStr = '$\overline{\sigma_{xx}}^{VG}$ [$MPa$]';
        hf = func_plotIdentStrVid_Field(plotParams,savePath,pos,time,...
            stress,stress.virtGaugeAvg.x,strain.virtGaugeAvg.x,ssCurveLabels,...
            fracture,strOpts,true,false);
        saveFile = [savePath,'\','FailStrIdent_Diagnostics'];
        func_saveFigureMultFormat(hf,saveFile,plotProps,plotParams)
    else
        fprintf('WARNING: specifed material model not recognised.\n')
     end 
    
    %--------------------------------------------------------------------------
    % Get the max stress and its location and time over the whole FOV
    % Useful for specimens that did not actually fail to determine the max
    % stress sustained without fracture.
    identStrength = func_getMaxStressOverFOV(grid,smoothingOpts,stress,identStrength);

    %--------------------------------------------------------------------------
    % Store vectors of the stress vs time at the fracture location
    identStrength.stressVsTAccel.xxAvgOverY = squeeze(stress.xAvg(fracture.locX,:));
    identStrength.stressVsTAccel.xxAvgOverGauge = stress.virtGaugeAvg.x; 
    identStrength.stressVsTStrain.xxAvgOverY = squeeze(stress.QRecon.x(:,fracture.locX,:));
    identStrength.stressVsTStrain.xxAvgOverGauge = stress.QRecon.virtGaugeAvg.x;
    
    %--------------------------------------------------------------------------
    % Write identified values to the console
    fprintf('\nIdentified failure stress values:\n')
    Strength_FracLoc = fracture
    Strength_StressGauge = identStrength.stressGauge
    Strength_LinearGauge = identStrength.linStressGauge

    fprintf('Strength Identification COMPLETE.\n')
    close all
else
    identStrength = nan;
    virtualGauge = nan;
    fracture = nan;
end

%% IMAGE SEQUENCES: For powerpoint presentations
if globalOpts.plotPresentationImages 
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        choice = questdlg('Plot images sequence for presentations?', ...
            'Plot Image Sequence?', 'Yes','No','No');
        switch choice
            case 'Yes'
                plotPresentationImageSeq = true;
            case 'No'
                plotPresentationImageSeq= false;
        end
    elseif strcmp(globalOpts.plotImageSeqs,'yes') || strcmp(globalOpts.plotImageSeqs,'diagnostic')
        plotPresentationImageSeq = true;
    else
        plotPresentationImageSeq = false; 
    end

    if plotPresentationImageSeq == true
        fprintf('\n--------------------------------------------------------------\n')
        fprintf('IMAGE SEQUENCES: For Presentations\n')
        fprintf('--------------------------------------------------------------\n')
        
        % Setup the plot parameters for the presentation figures
        savePath = imagePath;
        
        % Specify the frames over which the data is plotted
        endFrame = fracture.strengthFrame+5;
        %endFrame = time.numFrames;
        if endFrame > time.numFrames
            endFrame = time.numFrames;
        end
        plotParams.tRange = 1:endFrame;
        
        % Specify the location of the SS curve to be plotted
        if exist('fracture','var') == 1
            ssCurve.locX = fracture.locX;
        else
            ssCurve.locX = round(length(pos.x)/2);
        end
        % Plot the image sequences for presentation purposes
        fprintf('Plotting presentation image sequences.\n')
        func_plotPresentationVideos(globalOpts,plotParams,savePath,pos,time,...
            strain,strainRate,accel,stress,identStiffSG,ssCurve)
    end
end

%% SAVE DATA: Processing and Config Variables to File
if ~isfield(globalOpts,'saveProcData')
    globalOpts.saveProcData = 'yes';
end

if strcmp(globalOpts.saveProcData,'yes')
    if globalOpts.autoSaveProcData
        saveProcessedData = true;
    else
        choice = questdlg('Save processed data to file?', ...
        'Save Data?', 'Yes','No','No');
        switch choice
            case 'Yes'
                saveProcessedData = true;
            case 'No'
                saveProcessedData = false;
        end
    end

    if saveProcessedData
        fprintf('\n--------------------------------------------------------------\n')
        fprintf('SAVING DATA\n')
        fprintf('--------------------------------------------------------------\n')
        %--------------------------------------------------------------------------
        % Save a .mat and readme file of the processed data
        % create filenames for saving data and config files
        fprintf('Saving processed data to a mat file...\n')
        savePath = imagePath; % Save the processed data to the save directory as the input image
        saveFile = 'AllProcessedData.mat';
        save([savePath,saveFile],'test','specimen','material','time','pos',...
            'disp','accel','strain','strainRate','stress',...
            'identStiffSG','identStiffVFMan','identStiffVFOpt',...
            'identStrength','fracture')
        fprintf('All data saved.\n')
    end
end
%% SAVE DATA: Print Identified Properties to Console and Save in Struct
clc
fprintf('\n--------------------------------------------------------------\n')
fprintf('IDENTIFIED MATERIAL PROPERTIES\n')
fprintf('--------------------------------------------------------------\n')

% Create condensed info on the strain rate
identStrainRate.xAvgMin = strainRate.xAvgMin;
identStrainRate.xAvgMax = strainRate.xAvgMax;
if globalOpts.identifyStrength
    identStrainRate.xAvgMinPreFract = strainRate.xAvgMinPreFract;
    identStrainRate.xAvgMaxPreFract = strainRate.xAvgMaxPreFract;
else
    identStrainRate = nan;
end

fprintf('Identified stiffness values:\n')
IdentifiedStiffness_SG = identStiffSG
IdentifiedStiffness_VFMan = identStiffVFMan
IdentifiedStiffness_VFOpt = identStiffVFOpt
StrainRateMaxValues = identStrainRate

if globalOpts.identifyStrength
    fprintf('Identified strength values:\n')
    Strength_FracLoc = fracture
    Strength_StressGauge = identStrength.stressGauge
    Strength_LinearGauge = identStrength.linStressGauge
end

save([imagePath,'IdentifiedProperties.mat'],'identStiffSG','identStiffVFMan','identStiffVFOpt','fracture',...
    'identStrength','identStrainRate');

close all
fprintf('\nCOMPLETE.\n\n') 