%% IMAGE BASED INERTIAL RELEASE (IBIR) TEST: QS Stiffness Identification - v0.1b
% Author: Lloyd Fletcher
% PhotoDyn Group, University of Southampton
% http://photodyn.org/
% Date Created: 10th May 2019 - v0.1b
% Date Edited: 10th May 2019 - v1.0b
%
% This code uses the grid processing tool box that can be found at:
% www.thegridmethod.net, developed by Grediac et al.

clc
clear all
close all

fprintf('--------------------------------------------------------------\n')
fprintf('IMAGE-BASED INERTIAL RELEASE (IBIR): QS Data Processing v1.0b\n')
fprintf('--------------------------------------------------------------\n')

%% INITIALISE: Add path for processing functions
fprintf('Adding pathes for processing functions.\n')
% Add the path for the grid method code and other useful functions:
%funcPath = 'E:\Matlab_WorkingDirectory\ProcessingFunctions_2019\';
funcPath = [pwd,'\Functions\'];

% If the default path is not found we should find it
if exist(funcPath,'file') ~= 7
    hWarn = warndlg('Folder for processing functions not found.','Function folder not found');
    waitfor(hWarn);
    funcPath = uigetdir(pwd,'Locate Processing Function Folder');
end
addpath(funcPath);
addpath([funcPath,'GridMethodToolbox\']);

%% LOAD RAW DATA FILES: QS Images and Dyn Static Images
specNum = 1;
hardCodePath = true; % Useful for running a single file repeatedly for debugging

% Static reference image
fprintf('Loading static reference image from the selected test data folder.\n')
if ~hardCodePath
    [SRImageFile,SRImagePath] = uigetfile({'*.*','All Files'},'Select the static reference image.');
else
    if specNum == 1
        SRImageFile = 'Camera_12_42_32_100.tiff';
        SRImagePath = [pwd,'\Data_PMMATest1_StaticReference_QuasiStatic\'];
    elseif specNum == 2
        SRImageFile = 'Camera_13_10_34_100.tiff';
        SRImagePath = [pwd,'\Data_PMMATest2_StaticReference_QuasiStatic\'];
    elseif specNum == 3
        SRImageFile = 'Camera_13_23_05_100.tiff';
        SRImagePath = [pwd,'\Data_PMMATest2_StaticReference_QuasiStatic\'];
    end
end

% Loaded 'static' image from dynamic sequence
fprintf('Loading dynamic reference image from the selected test data folder.\n')
if ~hardCodePath
    [DynImageFile,DynImagePath] = uigetfile({'*.*','All Files'},'Select the deformed image.');
else
    if specNum == 1
        DynImageFile = 'Camera_12_49_40_001.tiff';
        DynImagePath = [pwd,'\Data_PMMARelease1_Dyn_1Mfps_UniAxial\']; 
    elseif specNum ==2
        DynImageFile = 'Camera_13_13_32_001.tiff';
        DynImagePath = [pwd,'\Data_PMMARelease2_Dyn_1Mfps_UniAxial\']; 
    elseif specNum == 3
        DynImageFile = 'Camera_13_27_16_001.tiff';
        DynImagePath = [pwd,'\Data_PMMARelease3_Dyn_2Mfps_UniAxial\'];
    end
end

%% INITIALISE: Load Processing Parameter Data Structures
fprintf('Loading processing parameters file.\n')
initPath = SRImagePath;
initFile = 'processingParameters.mat';
if exist([initPath,initFile],'file') ~= 2
    hWarn = warndlg('Processing parameter file does not exist.','Processing parameters not found');
    waitfor(hWarn);
    [initFile,initPath,~] = uigetfile('*.mat','Locate processing parameter file');
end
% Load the processing parameters from file
load([initPath,initFile])

%% INITIALISE: Set the specimen specific Data

if specNum == 1
    force.xRes = 1328;
    specimen.height = 9.97e-3;
    specimen.thickness = 2.77e-3;
elseif specNum == 2
    force.xRes = 967;
    specimen.height = 9.99e-3;
    specimen.thickness = 2.88e-3;
elseif specNum == 3
    force.xRes = 1011;
    specimen.height = 9.99e-3;
    specimen.thickness = 2.91e-3;
end

%% IMAGE PROCESSING: Use the Grid Method to extract displacement fields

%--------------------------------------------------------------------------
% 1) Load the reference and deformed images

% Load the reference image 
refImage = imread([SRImagePath,SRImageFile]);
refImage = double(refImage);

% Load the deformed image
defImage = imread([DynImagePath,DynImageFile]);
defImage = double(defImage);

% Create an image stack
imageStack(:,:,1) = refImage;
imageStack(:,:,2) = defImage;

% If this is image deformation data we might want to add some noise
if imageNoise.addNoise
    for ff = 1:size(imageStack,3)
        imageStack(:,:,ff) = func_addNoiseToImagesStruct(imageStack(:,:,ff),imageNoise);
    end
end

%--------------------------------------------------------------------------
% 2) Select specimen ROI to mask images
fprintf('Getting specimen region of interest (ROI).\n')

specLocFile = 'specimenLocation.mat';
loadSpecLocFile = false;
% Check if the specimen location file already exists, if it does prompt to load it
if exist([SRImagePath,specLocFile],'file') == 2
    choice = questdlg('Specimen location (ROI) file found:', ...
    'Process Grid Images', 'Load ROI file','Reselect ROI','Load ROI file');
    switch choice
        case 'Load ROI file'
            loadSpecLocFile = true;
        case 'Reselect ROI'
            loadSpecLocFile = false;
    end
end

if loadSpecLocFile
    fprintf('Loading existing ROI file.\n')
    load([SRImagePath,specLocFile])
else
    % User selects rectangular ROI using the imcrop function
    hf = figure; 
    imshow(refImage, []);
    title('Select specimen region of interest (ROI).')
    [~,~,tempImage,Rect] = imcrop(hf);
    specimenLoc.bottomLeft = ceil([Rect(1),Rect(2)]);
    specimenLoc.topRight = floor([Rect(1)+Rect(3),Rect(2)+Rect(4)]);   

    close all
    range.x = (specimenLoc.bottomLeft(1):specimenLoc.topRight(1));
    range.y = (specimenLoc.bottomLeft(2):specimenLoc.topRight(2));
    
    % Save the location to file for quick preload later
    save([SRImagePath,specLocFile],'specimenLoc')
end

% Assign the position of the specimen to the pos struct for return to main
pos.specimenLoc.bottomLeft = specimenLoc.bottomLeft;
pos.specimenLoc.topRight = specimenLoc.topRight;

%--------------------------------------------------------------------------
% 3) Input Grid Parameters
fprintf('Input grid parameters.\n')
gridData = inputdlg({'Number of pixels per period:','Grid pitch: (m)'}, ...
             'Grid analysis parameters', 1, {num2str(grid.pxPerPeriod), num2str(grid.pitch)} );
grid.pxPerPeriod = str2double(gridData{1}); 
grid.pitch = str2double(gridData{2});
grid.mPerPx = grid.pitch/grid.pxPerPeriod;

%--------------------------------------------------------------------------
% 4) Build the Analysis Window
%Build Window
analysisWindow = build_window(gridMethodOpts.windowFlag,...
    gridMethodOpts.windowWidth*grid.pxPerPeriod); % See documentation
% 0 = Gaussian window, default
% 1 = Bi-triangular window, localised phenom, low noise

%--------------------------------------------------------------------------
% 5) Process Images
% Pre-alloc vars for speed
[sy,sx] = size(refImage);
st = size(imageStack,3);
disp.x = zeros([sy,sx,st]); 
disp.y = zeros([sy,sx,st]);
disp.rot = zeros([sy,sx,st]);
phi.x = zeros([sy,sx,st]);
phi.y = zeros([sy,sx,st]);

% Spatial Unwrapping
fprintf('Spatial unwrapping.\n')
for ff = 1:size(imageStack,3)
    % Calculation of the phase and spatial phase unwrapping
    [phi.x(:,:,ff),phi.y(:,:,ff),phaseMod.x(:,:,ff),phaseMod.y(:,:,ff)] = ...
        LSA(imageStack(:,:,ff), analysisWindow, grid.pxPerPeriod);
    phi.x(:,:,ff) = unwrap2D(single(phi.x(:,:,ff)));
    phi.y(:,:,ff) = unwrap2D(single(phi.y(:,:,ff)));
end

% Temporal Unwrapping
% CAUTION: this only works for a sequence of images with small increments in
% the phase, uses average phase over the FOV to calc rigid body motion
if gridMethodOpts.temporalUnwrap
    range.x = (specimenLoc.bottomLeft(1):specimenLoc.topRight(1));
    range.y = (specimenLoc.bottomLeft(2):specimenLoc.topRight(2));
    phi.x_nuw = phi.x;
    phi.y_nuw = phi.y;
    
    fprintf('Temporal unwrappping.\n')
    threshold = pi;
    phase = func_temporalUnwrap(phi,threshold,'field',range);

    for ff = 1:size(phi.x,3)
        phi.x(:,:,ff) = phi.x(:,:,ff) + 2*pi*phase.x(ff); 
        phi.y(:,:,ff) = phi.y(:,:,ff) + 2*pi*phase.y(ff);
    end
end

if gridMethodOpts.debug
    figure;
    hold on
    plot(squeeze(mean(mean(phi.x(range.y,range.x,:)))),'-+b')
    plot(squeeze(mean(mean(phi.x_nuw(range.y,range.x,:)))),'-xr')
    xlabel('Frame')
    ylabel('Mean Phase Over ROI')
    legend('Temp Unwrapped','No Unwrap')
    hold off    
end

% Calculate Displacements and Strains
fprintf('Calculating displacement and strain components.\n')
disp.x = zeros(size(phi.x));
disp.y = zeros(size(phi.x));
disp.rot = zeros(size(phi.x));

% Calculate a window over which the specimen exists + an extra pitch to
% avoid nans - currently uses the range variable
% X Range
if (min(range.x)-grid.pxPerPeriod) < 1
    startInd = 1;
else
    startInd = min(range.x)-grid.pxPerPeriod;
end
if (max(range.x)+grid.pxPerPeriod) > size(phi.x,2)
    endInd = size(phi.x,2);
else
    endInd = max(range.x)+grid.pxPerPeriod;
end
dispRangeX = startInd:endInd;

% Y Range
if (min(range.y)-grid.pxPerPeriod) < 1
    startInd = 1;
else
    startInd = min(range.y)-grid.pxPerPeriod;
end
if (max(range.y)+grid.pxPerPeriod) > size(phi.y,1)
    endInd = size(phi.y,1);
else
    endInd = max(range.y)+grid.pxPerPeriod;
end
dispRangeY = startInd:endInd;

for ff = 2:size(imageStack,3)  
    [disp.x(dispRangeY,dispRangeX,ff), disp.y(dispRangeY,dispRangeX,ff),~,~,~, disp.rot(dispRangeY,dispRangeX,ff)]...
        = calculate_U_EPS(grid.pxPerPeriod,squeeze(phi.x(dispRangeY,dispRangeX,1)),squeeze(phi.y(dispRangeY,dispRangeX,1)),...
        squeeze(phi.x(dispRangeY,dispRangeX,ff)),squeeze(phi.y(dispRangeY,dispRangeX,ff)),gridMethodOpts.dispCalcMethod,100);  
end

%--------------------------------------------------------------------------
% 6) Convert Displacement and Crop to ROI 
% Convert the displacement to mm
fprintf('Cropping to ROI and updating position/size vectors.\n')
if grid.asymmPitch
    disp.x = disp.x*(grid.pitchX/grid.pxPerPeriod); 
    disp.y = disp.y*(grid.pitchY/grid.pxPerPeriod);
else
    disp.x = disp.x*(grid.pitch/grid.pxPerPeriod); 
    disp.y = disp.y*(grid.pitch/grid.pxPerPeriod);
end

% Crop the image to the ROI
disp.x = disp.x(range.y,range.x,:);
disp.y = disp.y(range.y,range.x,:);

if grid.asymmPitch
    % Create the position mesh grid
    grid = rmfield(grid,'mPerPx');
    grid.mPerPxX = grid.pitchX/grid.pxPerPeriod;
    grid.mPerPxY = grid.pitchY/grid.pxPerPeriod;
    pos.x = grid.mPerPxX/2:grid.mPerPxX:(grid.mPerPxX*size(disp.x,2));
    pos.y = grid.mPerPxY/2:grid.mPerPxY:(grid.mPerPxY*size(disp.x,1));
    [pos.xGrid,pos.yGrid] = meshgrid(pos.x,pos.y);
    pos.xStep = grid.mPerPxX;
    pos.yStep = grid.mPerPxY;

    % Update grid size parameters
    grid.length = size(disp.x,2)*grid.pitchX/grid.pxPerPeriod;
    grid.height = size(disp.y,1)*grid.pitchY/grid.pxPerPeriod; 
else
    % Create the position mesh grid
    pos.x = grid.mPerPx/2:grid.mPerPx:(grid.mPerPx*size(disp.x,2));
    pos.y = grid.mPerPx/2:grid.mPerPx:(grid.mPerPx*size(disp.x,1));
    [pos.xGrid,pos.yGrid] = meshgrid(pos.x,pos.y);
    pos.xStep = grid.mPerPx;
    pos.yStep = grid.mPerPx;

    % Update grid size parameters
    grid.length = size(disp.x,2)*grid.pitch/grid.pxPerPeriod;
    grid.height = size(disp.y,1)*grid.pitch/grid.pxPerPeriod;
end

%--------------------------------------------------------------------------
% Ask user about grid rotation and correct if needed
fprintf('Correcting for any grid rotation.\n')
[grid,disp] = func_rotateDisp(grid,disp,~(gridMethodOpts.hardCodeRotAngle));

%--------------------------------------------------------------------------
% Tranpose fields if the images are in portrait
if isfield(grid,'transposeFields')
    if grid.transposeFields
        temp = disp.x;
        disp.x = permute(disp.y,[2,1,3]);
        disp.y = permute(temp,[2,1,3]);
        temp = pos.xGrid;
        pos.xGrid = permute(pos.yGrid,[2,1,3]);
        pos.yGrid = permute(temp,[2,1,3]);
        temp = pos.x;
        pos.x = pos.y;
        pos.y = temp;
    end
end

figure; imagesc(disp.x(6:end-5,6:end-6,2)); axis image; colorbar;
figure; imagesc(disp.y(6:end-5,6:end-6,2)); axis image; colorbar;

%% POST-PROCESSING: Strain and Stiffness Calculation

%--------------------------------------------------------------------------
% Smooth and Calculate Strain
fprintf('Calculating strain from the displacement fields.\n')
[disp,strain,~] = func_smoothCalcStrain(globalOpts,pos,time,...
    grid,disp,smoothingOpts,extrapOpts);

cutPx = 6;
cutPxX = 0;
% Remove the lost pixels of bad grid data
strain.xSAvg = squeeze(nanmean(nanmean(strain.x(cutPx:end-cutPx,cutPx+cutPxX:end-cutPx-cutPxX,:))));
strain.ySAvg = squeeze(nanmean(nanmean(strain.y(cutPx:end-cutPx,cutPx+cutPxX:end-cutPx-cutPxX,:))));
strainXSAvg = strain.xSAvg(2)*10^3
strainYSAvg = strain.ySAvg(2)*10^3

figure; imagesc(disp.x(6:end-5,6:end-6,2)); axis image; colorbar;
figure; imagesc(disp.y(6:end-5,6:end-6,2)); axis image; colorbar;

figure; imagesc(strain.x(6:end-5,6:end-6,2)); axis image; colorbar;
figure; imagesc(strain.y(6:end-5,6:end-6,2)); axis image; colorbar;

%--------------------------------------------------------------------------
% STIFFNESS IDENTIFICATION: Qxx and Qxy using manual VFM
specimen.CSA = specimen.height*specimen.thickness;
stress.xAvg = force.xRes/specimen.CSA

ECheck = stress.xAvg / strain.xSAvg(2)
nuCheck = -strain.ySAvg(2) / strain.xSAvg(2)



