% Script to generate figures in "Design and Evaluation of Personal Audio 
% Systems based on Speech Privacy Constraints"
% 
% Daniel Wallace 
% D.Wallace@soton.ac.uk 
% 

clearvars
close all
clc

%% Figure 1 
% Front view of loudspeaker array. Groups of 9 elements
% were selected from a 27-channel array to form two arrays with
% different horizontal element spacing. Narrow and wide sub-
% arrays are indicated with solid and dotted lines respectively.

% 27-channel array source coordinates 
N = 27;                                % Number of Drivers
dx = 35.14e-3;                         % Horizontal spacing between drivers
dz = 30.40e-3;                         % Vertical spacing between drivers
n = zeros(N,3);                        % Line array coordinates
n(:,1) = ((-N/2)+0.5:(N/2-0.5))*dx;    % x coordinates
n(1:2:end,3) = dz/2;                   % odd  z coordinates
n(2:2:end,3) = -dz/2;                  % even z coordinates    
sourceRadius = sqrt(0.00085/pi);       % Find source radius from area

% Produce circles for driver outlines 
t = linspace(0,2*pi,100);
x = sourceRadius*sin(t);
y = sourceRadius*cos(t);

% Initialise Figure 1 and draw driver outlines 
figure(1)
hold off
for i = 1:27
    plot(n(i,1)+x,n(i,3)+y,'Color',[0.7,0.7,0.7])
    hold on
end
axis equal
axis off

% Draw edge of loudspeaker array enclosure
plot([-0.47,-0.52,-0.52,0.52,0.52,0.47,-0.47],[-0.05,0.01,0.05,0.05,0.01,-0.05,-0.05],'k-')
xlim([-0.53,0.53])
ylim([-0.08,0.06])

% Select drivers in "Narrow" and "Wide" arrays 
driverArray = logical([[0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0];
                       [0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0]]);
          
% Set colour and line style for "Narrow" and "Wide" arrays. 
cols = {[0,0,0],[0.3,0.3,0.3]}; 
ls = {'-','--'};     

% For narrow, then wide 
for j = 1:2
    nj = n(driverArray(j,:)',:);
    % Create circles of different radii depending on index j
    t = linspace(0,2*pi,100);
    x = sourceRadius*sin(t)*1.17^j; 
    y = sourceRadius*cos(t)*1.17^j;
    % Plot circles with centres displaced by coordinates n
    for i = 1:9
        plot(nj(i,1)+x,nj(i,3)+y,'Color',cols{j},'LineWidth',2,'LineStyle',ls{j})
    end
end

% Annotations: 
% For delta_n 
plot([dx,dx],[dz/2,-2.5*dz],'k--')
plot([2*dx,2*dx],[-dz/2,-2.5*dz],'k--')
text(1.5*dx,-2.1*dx,'\delta\it_n','FontSize',32,'HorizontalAlignment','Center','FontName','Times')
text(0.25*dx,-1.9*dx,'\rightarrow','HorizontalAlignment','Left','FontSize',30)
text(2*dx,-1.9*dx,'\leftarrow','HorizontalAlignment','Left','FontSize',30)
% For delta_w
plot([6*dx,6*dx],[-dz/2,-2.5*dz],'k--')
plot([9*dx,9*dx],[dz/2,-2.5*dz],'k--')
text(7.5*dx,-2.1*dx,'3\delta\it_n= \rm\delta\it_w','Fontsize',32,'HorizontalAlignment','Center','FontName','Times')
text(5.25*dx,-1.9*dx,'\rightarrow','HorizontalAlignment','Left','FontSize',30)
text(9*dx,-1.9*dx,'\leftarrow','HorizontalAlignment','Left','FontSize',30)

% Figure formatting 
set(gcf,'Position',[411, 435, 1562, 250])
set(gca,'position',[0, -0.1, 1, 1.2])

%% Figure 2 
% Plan view of the personal audio system geometry, 
% showing source and microphone locations.

% Extract data 
fig2_data = csvread('Figure_2_data.csv',1,0);
brightSetupLocs = fig2_data(:,1:3);
brightEvalLocs = fig2_data(:,4:6);
darkSetupLocs = fig2_data(:,7:9);
darkEvalLocs = fig2_data(:,10:12);
narrowSourceLocs = fig2_data(1:9,13:15);
wideSourceLocs = fig2_data(1:9,16:18);

% Custom plot colours
carrot = [0.93,0.57,0.13];
purple = [128, 66, 244]/255;

% Plot data
figure(2)
hold off
plot3(narrowSourceLocs(:,1),narrowSourceLocs(:,2),narrowSourceLocs(:,3),'Marker','+','LineStyle','none','Color',carrot)
hold on 
plot3(darkSetupLocs(:,1),darkSetupLocs(:,2),darkSetupLocs(:,3),'Color','b','marker','o','markersize',6,'MarkerFaceColor','b','Linestyle','none')
plot3(darkEvalLocs(:,1),darkEvalLocs(:,2),darkEvalLocs(:,3),'Color','b','marker','o','markersize',6,'MarkerFaceColor','w','Linestyle','none')
plot3(wideSourceLocs(:,1),wideSourceLocs(:,2),wideSourceLocs(:,3),'Marker','x','LineStyle','none','Color',purple)
plot3(brightSetupLocs(:,1),brightSetupLocs(:,2),brightSetupLocs(:,3),'Color','r','marker','s','markersize',8,'MarkerFaceColor','r','Linestyle','none')
plot3(brightEvalLocs(:,1),brightEvalLocs(:,2),brightEvalLocs(:,3),'Color','r','marker','s','markersize',8,'MarkerFaceColor','w','Linestyle','none')

% Plot formatting  
view(2)
xlabel('Metres')
ylabel('Metres')
axis equal
ylim([-0.1,1])
xlim([-0.78,0.78])
set(gca,'FontSize',14,'Fontname','Times')
grid on
box on

% Arrow indicating distance away from source
plot([-0.7,0.5],[0.72,0.72],'--','Color',[0.5,0.5,0.5])
plot([-0.7,-0.42],[0,0],'--','Color',[0.5,0.5,0.5])
annotation('doublearrow',[0.155,0.155],[0.375,0.81],'Color',[0.5,0.5,0.5])
text(-0.67,0.35,'0.72 m','FontName','Times','FontSize',14,'Color',[0.5,0.5,0.5])

% Arrow indicating width between zones 
plot([-0.5,-0.5],[0.72,1.0],'--','Color',[0.5,0.5,0.5])
plot([0.5,0.5],[0.72,1.0],'--','Color',[0.5,0.5,0.5])
annotation('doublearrow',[0.26,0.82],[0.95,0.95],'Color',[0.5,0.5,0.5])
text(-0.07,0.90,'1.0 m','FontName','Times','FontSize',14,'Color',[0.5,0.5,0.5])

% Microphone grid indication
plot([0.5,0.5],[0.45,0.56],'--','Color',[0.5,0.5,0.5])
plot([0.5+0.072,0.5+0.072],[0.45,0.56],'--','Color',[0.5,0.5,0.5])% 
annotation('arrow',[0.70,0.82],[0.68,0.68],'Color',[0.5,0.5,0.5])
annotation('arrow',[0.912,0.86],[0.68,0.68],'Color',[0.5,0.5,0.5])
text(0.25,0.45,'0.072 m','FontName','Times','FontSize',14,'Color',[0.5,0.5,0.5])

% Legend and plot formatting
hl = legend({'Narrow Source Elements','Dark Optimisation Mics','Dark Evaluation Mics','Wide Source Elements','Bright Optimisation Mics','Bright Evaluation Mics'},'Location','SouthOutside','NumColumns',2);
set(gcf,'Position',[   1000         462         560         516])
set(gca,'Position',[ 0.100    0.2400    0.8750    0.8150]) 
set(hl,'Position',[0.1513    0.0480    0.7732    0.1560])
set(gcf,'renderer','Painters')

%% Figure 3 
% Acoustic contrast measurements for
% the narrow and wide loudspeaker array configurations.

% Extract data
fig3_data = csvread('Figure_3_data.csv',1,0);
f = fig3_data(:,1);
narrow_AC = fig3_data(:,2);
wide_AC = fig3_data(:,3);
% Custom plot colours
carrot = [0.93,0.57,0.13];
purple = [128, 66, 244]/255;

% Plot lines 
figure(3) 
semilogx(f,narrow_AC,'-','Color',carrot,'linewidth',2)
hold on
semilogx(f,wide_AC,'--','Color',purple,'linewidth',2)

% Apply formatting 
xlabel('Freq (Hz)')
ylabel('$C$ (dB)','interpreter','latex')
legend('Narrow','Wide')
xlim([100,8000])
ylim([0,25])
xticks(125*2.^(0:6))
set(gca,'FontSize',14,'FontName','Times')
grid on

%% Figure 4
% Relative Sound Pressure Level with
% tonal signals focussed to the bright zone (square markers)
% at 1.2, 2.4 and 4.8 kHz for each source array configuration.
% Dimensions are in metres.

% Extract coordinate data for grid plots
XX = csvread('Figure_4_xGrid.csv');
YY = csvread('Figure_4_yGrid.csv');

% Extract data for plotting positions of microphones 
% Uses data from Figure 2
fig2_data = csvread('Figure_2_data.csv',1,0);
brightSetupLocs = fig2_data(:,1:3);
brightEvalLocs = fig2_data(:,4:6);
darkSetupLocs = fig2_data(:,7:9);
darkEvalLocs = fig2_data(:,10:12);
sourceLocs(:,:,1) = fig2_data(1:9,13:15); % Narrow source locations 
sourceLocs(:,:,2) = fig2_data(1:9,16:18); % Wide source locations

% Custom plot colours
carrot = [0.93,0.57,0.13];
purple = [128, 66, 244]/255;

% Define colours and markers for narrow and wide source arrays 
cols = {carrot,purple};
markers = {'+','x'};
widths = {'n','w'};
freqs = [1200,2400,4800];
plotLabels = {'a','b';'c','d';'e','f'};

figure(4)
for i = 1:2              % For each width 
    width = widths{i};   % Select width 
    for j = 1:3          % For each frequency 
        freq = freqs(j); % Select frequency  
        % Read in data for pressure grid 
        pressureGrid = csvread(sprintf('Figure_4_pressureGrid_%d_%s.csv',freq,width));
        subplot(3,2,((j-1)*2+i))   % Index of subplot 
        hold off
        pcolor(XX,YY,pressureGrid) % Plot data in pseudocolour plot
        
        % Plot source and microphone locations 
        hold on 
        plot3(sourceLocs(:,1,i),sourceLocs(:,2,i),sourceLocs(:,3,i),'Marker',markers{i},'LineStyle','none','Color',cols{i})
        hold on 
        plot(brightSetupLocs(:,1),brightSetupLocs(:,2),'Color','r','marker','s','markersize',6,'MarkerFaceColor',[1,0.6,0.6],'Linestyle','none')
        plot(darkSetupLocs(:,1),darkSetupLocs(:,2),'Color','b','marker','o','markersize',4,'MarkerFaceColor',[0.6,0.6,1],'Linestyle','none')
        plot(brightEvalLocs(:,1),brightEvalLocs(:,2),'Color','r','marker','s','markersize',6,'MarkerFaceColor','w','Linestyle','none')
        plot(darkEvalLocs(:,1),darkEvalLocs(:,2),'Color','b','marker','o','markersize',4,'MarkerFaceColor','w','Linestyle','none')    
        
        % Plot formatting 
        view(2)             % Top-down view
        shading interp      % Interpolate over grid 
        colormap viridis    % Colour map with perceptually accurate grading
        caxis([-40,0])      % Colour axis limits 
        
        % Display colour bar on right hand plots 
        if i == 2
            colorbar
        else  % Display frequency on left hand plots 
            toShow = sprintf('\\textbf{f = %d Hz}',freq);
            ylabel({toShow},'interp','latex')
        end
        
        % Display titles on top row of plots 
        if j == 1
            if i == 1
                ht = title('Narrow Array');
                set(ht,'Position',[1.6085e-06 1.159 0])
            else
                ht = title('Wide Array');
                set(ht,'Position',[3.7907e-06 1.159 0])
            end
        end
        
        % Axis scaling and labelling
        axis equal
        ylim([-0.05,1.05])
        xlim([-0.7,0.7])
        xlabel(sprintf('(%s)',plotLabels{j,i}))
        set(gca,'Fontsize',16,'FontName','Times')
    end
end
% Overall figure sizing
set(gcf,'Position',[ 1000         255         674         723])

%% Figure 6
% Distribution of Speech Reception Thresholds (SRTs) 
% achieved for each array and masker configuration. 
% N=21 participants. Red plusses indicate outliers, identified as the
% results which lie greater than 1.5 times the box length from the edges 
% of the box (approximately 2.7 sigma). The reference SRT for the 
% closed-set English matrix test is represented with a dashed horizontal 
% line.

% Extract SRT data from each of N=21 participants
fig6_data = csvread('Figure_6_data.csv',1,0);
labels = {'$$W_{S}$$','$$W_{A}$$','$$W_\infty$$','$$N_{S}$$','$$N_{A}$$','$$N_\infty$$'};

figure(6)
boxplot(fig6_data,labels)   % Box plot of columnwise data
hold on
refSRT = -8.9;              % Reference SRT for test
plot([0,7],[refSRT, refSRT],'k--')
% Plot formatting
grid on     
ylabel('SRT (dB)','interpreter','latex')
set(gca,'TickLabelInterpreter','latex','Fontsize',14);

%% Figure 7
% Surface plots of SII and Loudness with variation in dark zone 
% signal-to-noise ratio and masking signal cut-off frequency. 
% Upper Row: SII in dark zone.
% Middle Row: SII in bright zone. 
% Lower Row: Loudness of masker in dark zone. 
% White line: SIId = 0.05. 
% Black line: SIIb = 0.75. 
% Arrows indicate regions of the parameter space where intelligibility #
% constraints are met.

% Extract coordinate data for surface plots 
cutoffs = csvread('Figure_7_cutoff.csv'); % LPF cut-off frequency, 
                                             % X axis grid on surface plots
SNRds = csvread('Figure_7_SNRd.csv');      % SNR in dark zone, 
                                             % Y axis grid on surface plots   
% Extract colour data for surface plots 
SII_d_n = csvread('Figure_7_SII_dark_narrow.csv');
SII_b_n = csvread('Figure_7_SII_bright_narrow.csv');
L_d_n   = csvread('Figure_7_Loudness_dark_narrow.csv');
SII_d_w = csvread('Figure_7_SII_dark_wide.csv');
SII_b_w = csvread('Figure_7_SII_bright_wide.csv');
L_d_w   = csvread('Figure_7_Loudness_dark_wide.csv');

% Parameters for contour lines 
% The contour lines on each plot can be generated from the surface data 
% using contour(), or be plotted directly from the data in *.csv files: 
% If useContourCSVData = true, plots use data from CSV files.
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %
useContourCSVData = true; 
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %

lc = [0.05,0.05]; % Lower contour, dark zone constraint SII_d < 0.05
uc = [0.75,0.75]; % Upper contour, bright zone constraint SII_b > 0.75
contour_bright_narrow = csvread('Figure_7_contour_bright_narrow.csv',1,0);
contour_bright_wide = csvread('Figure_7_contour_bright_wide.csv',1,0);
contour_dark_narrow = csvread('Figure_7_contour_dark_narrow.csv',1,0);
contour_dark_wide = csvread('Figure_7_contour_dark_wide.csv',1,0);

% Initialise Figure
figure(7)
set(gcf,'Position',[  1000          49         758         929])

% Initialise subplot for dark zone SII, narrow array 
subplot(3,2,1)                  
pcolor(cutoffs,SNRds,SII_d_n)     % Pseudo-colour plot of data, 
% Surface plot formatting 
shading interp
hold on 
% Plot contours 
if useContourCSVData
    plot(contour_dark_narrow(:,1),contour_dark_narrow(:,2),'linestyle','-','color','w','linewidth',2)
else
    contour(cutoffs,SNRds,SII_d_n,lc,'linestyle','-','color','w','linewidth',2)
end
xlabel('$f_c$ (Hz)','interp','latex')
ylabel({'\textbf{SII (Dark Zone)}','','$\mathrm{SNR}_d$ (dB)'},'interp','latex')
set(gca,'xscale','log','xtick',[1000,2000,4000,8000,16000],'Fontsize',15,'FontName','Times');
caxis([0,1])
axis square
colormap viridis
ht = title('Narrow Array');
set(ht,'Position',[4.0000e+03 5 0])
annotation('arrow',[0.338258575197889,0.343007915567282],[0.848261571582345,0.81485468245425],'Linewidth',2,'color','w')
annotation('arrow',[0.253298153034301,0.279683377308707],[0.804090419806243,0.783638320775027],'Linewidth',2,'color','w')

% Initialise subplot for dark zone SII, wide array 
subplot(3,2,2) 
pcolor(cutoffs,SNRds,SII_d_w) % Pseudo-colour plot of data, 
% Surface plot formatting 
shading interp
hold on
% Plot contours 
if useContourCSVData
    plot(contour_dark_wide(:,1),contour_dark_wide(:,2),'linestyle','-','color','w','linewidth',2)
else
    contour(cutoffs,SNRds,SII_d_w,lc,'linestyle','-','color','w','linewidth',2)
end
% Labels, annotations and formatting
xlabel('$f_c$ (Hz)','interp','latex')
ylabel('$\mathrm{SNR}_d$ (dB)','interp','latex')
set(gca,'xscale','log','xtick',[1000,2000,4000,8000,16000],'Fontsize',15,'FontName','Times');
caxis([0,1])
axis square
colormap viridis
h = colorbar;
set(h,'Position',[0.9010 0.720 0.0224 0.2050])
ht2 = title('Wide Array');
set(ht2,'Position',[4.0000e+03 5 0])
annotation('line',[0.923,0.902],[0.73,0.73],'linewidth',2,'color','w')
annotation('arrow',[0.722955145118733,0.753298153034301],[0.790173304628632,0.778256189451022],'Linewidth',2,'color','w')
annotation('arrow',[0.763852242744064,0.783641160949868],[0.850453175457479,0.826695371367059],'Linewidth',2,'color','w')

% Initialise subplot for bright zone SII, Narrow Array
subplot(3,2,3)
pcolor(cutoffs,SNRds,SII_b_n) % Plot surface data 
shading interp 
hold on 
if useContourCSVData
    plot(contour_bright_narrow(:,1),contour_bright_narrow(:,2),'linestyle','-','color','k','linewidth',2)
else
    contour(cutoffs,SNRds,SII_b_n,uc,'linestyle','-','color','k','linewidth',2)
end
% Labels, annotations and formatting
xlabel('$f_c$ (Hz)','interp','latex')
ylabel({'\textbf{SII (Bright Zone)}','','$\mathrm{SNR}_d$ (dB)'},'interp','latex')
set(gca,'xscale','log','xtick',[1000,2000,4000,8000,16000],'Fontsize',15,'FontName','Times');
caxis([0,1])
axis square
colormap viridis
annotation('arrow',[0.229551451187335,0.216358839050132],[0.52629601722282,0.554359526372443],'Linewidth',2,'color','k')
annotation('arrow',[0.320580474934037,0.319261213720317],[0.532754574811625,0.565123789020452],'Linewidth',2,'color','k')

% Initialise subplot for bright zone SII, Wide Array
subplot(3,2,4)
pcolor(cutoffs,SNRds,SII_b_w) % plot surface data 
shading interp
hold on 
if useContourCSVData
    plot(contour_bright_wide(:,1),contour_bright_wide(:,2),'linestyle','-','color','k','linewidth',2)
else
    contour(cutoffs,SNRds,SII_b_w,uc,'linestyle','-','color','k','linewidth',2)
end
% Labels, annotations and formatting
xlabel('$f_c$ (Hz)','interp','latex')
ylabel('$\mathrm{SNR}_d$ (dB)','interp','latex')
set(gca,'xscale','log','xtick',[1000,2000,4000,8000,16000],'Fontsize',15,'FontName','Times');
caxis([0,1])
axis square
colormap viridis
g = colorbar;
set(g,'Position',[0.9010 0.420 0.0224 0.2050])
annotation('line',[0.923,0.902],[0.57,0.57],'linewidth',2)
annotation('arrow',[0.679419525065963,0.658311345646438],[0.501691065662002,0.532831001076426],'Linewidth',2,'color','k')
annotation('arrow',[0.763852242744063,0.759894459102902],[0.527448869752422,0.561894510226049],'Linewidth',2,'color','k')

% Initialise subplot for dark zone loudness, narrow array
subplot(3,2,5)
pcolor(cutoffs,SNRds,L_d_n) % Plot surface data
shading interp
caxis([60,100])
xlabel('$f_c$ (Hz)','interp','latex')
ylabel({'\textbf{Loudness (Dark Zone)}','','$\mathrm{SNR}_d$ (dB)'},'interp','latex')
set(gca,'xscale','log','xtick',[1000,2000,4000,8000,16000],'Fontsize',15,'FontName','Times');
colormap(gca,'plasma')
hold on 
if useContourCSVData
    plot(contour_dark_narrow(:,1),contour_dark_narrow(:,2),'linestyle','-','color','w','linewidth',2)
    plot(contour_bright_narrow(:,1),contour_bright_narrow(:,2),'linestyle','-','color','k','linewidth',2)
else
    contour(cutoffs,SNRds,SII_d_n,lc,'linewidth',2,'color','w')
    contour(cutoffs,SNRds,SII_b_n,uc,'linewidth',2,'color','k')
end
axis square
% Plot differences between experimental SRTs and SII=0.05
plot(4000,-15,'c.','markersize',15) % N_s, SRT 
plot([4000,4000],[-15,-13],'c:','linewidth',1.5)
plot(8000,-12,'c.','markersize',15) % N_a, SRT
plot(16000,-10,'c.','markersize',15) % N_inf, SRT
plot([16000,16000],[-10,-12],'c:','linewidth',1.5)

% Initialise subplot for dark zone loudness, wide array 
subplot(3,2,6)
pcolor(cutoffs,SNRds,L_d_w) % Plot surface data
shading interp
caxis([60,100])
xlabel('$f_c$ (Hz)','interp','latex')
ylabel('$\mathrm{SNR}_d$ (dB)','interp','latex')
set(gca,'xscale','log','xtick',[1000,2000,4000,8000,16000],'Fontsize',15,'FontName','Times');
colormap(gca,'plasma')
hold on 
if useContourCSVData
    plot(contour_dark_wide(:,1),contour_dark_wide(:,2),'linestyle','-','color','w','linewidth',2)
    plot(contour_bright_wide(:,1),contour_bright_wide(:,2),'linestyle','-','color','k','linewidth',2)
else
    contour(cutoffs,SNRds,SII_d_w,lc,'linewidth',2,'color','w')
    contour(cutoffs,SNRds,SII_b_w,uc,'linewidth',2,'color','k')
end
axis square
hcaw = colorbar(gca);
set(hcaw,'Position',[0.9010 0.120 0.0224 0.2050])
% Plot differences between experimental SRTs and SII=0.05
plot(4000,-21,'c.','markersize',15) % W_s, SRT 
plot([4000,4000],[-21,-16],'c:','linewidth',1.5)
plot(2400,-32,'c.','markersize',15) % W_a, SRT
plot([2400,2400],[-32,-27.5],'c:','linewidth',1.5)
plot(16000,-17,'c.','markersize',15) % W_inf, SRT
plot([16000,16000],[-17,-11.8],'c:','linewidth',1.5)
