function res = Annoyance(signal,FS,show)
    % Calculates the Psychoacoustic Annoyance of a signal
    %
    % INPUT: 
    %   signal    : acoustic signal, monophonic (Pa)
    %   FS        : sampling frequency (Hz)
    %   show      : true/false to display output to screen
    % 
    % OUTPUT:
    % res   : Structure containing fields
    %   A   : Annoyance 
    %   L   : Loudness exceeded for 5% of the time (sones) 
    %   S   : Sharpness of signal (acum) 
    %   F   : Fluctuation Strength of Signal (vacil) 
    %   R   : Roughness of Signal (asper)   
    % 
    % REFERENCE: 
    % Zwicker E. and Fastl H., "Psychoacoustics: Facts and models",
    % 2nd Edition, Springer-Verlag, Berlin, 1999
    %
    % DEPENDENCIES: 
    % Genesis Loudness Toolbox - GENESIS S.A. 
    % 
    % CONTACT: 
    % Daniel Wallace: ORCID ID: https://orcid.org/0000-0003-0212-5395
    
    % Loudness 
    loudRes = Loudness_TimeVaryingSound_Zwicker(signal,FS,'mic','free',5,0.03,show);
    L = loudRes.Nx; % Loudness exceeded for 5% of the signal duration
    
    % Sharpness 
    S = Sharpness(signal,FS,'mic','free',5,0.03,show);
    
    % ******* Fluctuation Strength ********
    F = FluctuationStrength(signal,FS);
    
    % ************* Roughness *************
    R = Roughness(signal,FS);
    
    % ******** Annoyance *********    
    if S > 1.75
        ws = (S-1.75)*(0.25*log10(L+10));
    else
        ws = 0;
    end
    wfr = 2.18/(L^0.4)*(0.4*F+0.6*R);
    A = L * (1 + sqrt (ws^2 + wfr^2));
    
    if show
        fprintf('Loudness    = %6.3f \n',L)
        fprintf('Sharpness   = %6.3f \n',S)
        fprintf('Roughness   = %6.3f \n',R)
        fprintf('Fluctuation = %6.3f \n',F)
        fprintf('Annoyance   = %6.3f \n',A)
    end
    
    % outputs 
    res.R = R;
    res.S = S;
    res.L = L;
    res.F = F;
    res.A = A;