function F = FluctuationStrength(signal,FS)
    % F = FluctuationStrength(signal,FS)
    % Calculates the fluctuation strength of a signal in vacil.
    %
    % INPUT:
    % signal - acoustic signal, monophonic (Pa)
    % FS - sampling frequency (Hz)
    %
    % OUTPUT: 
    % F - fluctuation strength - an approximation of the fluctuation
    % strength of the signal.
    %
    % ALGORITHM:
    % Function uses Loudness_Sharpness to calculate the
    % instantaneous specific loudness in 240 fractional bark bands. These 
    % are aggregated by summing them into the 24 bark bands. The 24 signals
    % are smoothed with a moving average filter with time constant 1/10th
    % second to remove some peaks. Sensation of fluctuation is greatest at 
    % 4 Hz, so smoothing should not affect strong variation here. The peaks
    % and troughs of the 24 bark band signals are collected, and for each 
    % peak, the level difference between the peak and the nearest and 
    % deepest trough (looking one trough to the right and one to the left) 
    % is collected. These are averaged across the signals and then summed 
    % over bark bands to give the Fluctuation Strength in Vacil.
    %
    % REFERENCE 
    % Zwicker E. and Fastl H., "Psychoacoustics: Facts and models",
    % 2nd Edition, Springer-Verlag, Berlin, 1999
    % 
    % DEPENDENCIES: 
    % Genesis Loudness Toolbox - GENESIS S.A. 
    % 
    % CONTACT: 
    % Daniel Wallace - ORCID ID: https://orcid.org/0000-0003-0212-5395

    % Get Instantaneous Specific Loudness
    length_sec = length(signal)/FS;
    res = Loudness_TimeVaryingSound_Zwicker(signal,FS,'mic','free',5,0.03,false);
    ISL = res.InstantaneousSpecificLoudness;
    FS_loudness = 500; % Sample rate of Instantaneous Specific Loudness
    % ISLs must be averaged into Bark bands from 1/10th fractional bands
    ISLb = zeros(size(ISL,1),24);
    for k = 1:10
        ISLb = ISLb + ISL(:,k:10:240); % Instantaneous Specific Loudness (1 bark bands)
    end
    % Apply temporal smoothing
    ISLbs = zeros(size(ISLb));
    for j = 1:24
        % Instantaneous Specific Loudness, bark band, smoothed. 
        % use moving average filter with 10Hz response time.
        ISLbs(:,j) = smooth(ISLb(:,j),0.1*FS_loudness);
    end
    % Ignore the start and end 1/8th second as this region is not smoothed
    % by MA filter
    cut = floor(0.125*FS_loudness):length(ISL)-floor(0.125*FS_loudness);
    ISLbs_cut = ISLbs(cut,:);
    % Get peaks and troughs of the cuts, per band
    LDb = zeros(1,24);
    for bark = 1:24
        currentSig = ISLbs_cut(:,bark);
        % Expect about four peaks per second,
        % Get maximum peaks first
        % Expect spacing greater than 1/10th second 
        [pk,pkloc] = findpeaks(currentSig,'NPeaks',floor(4*length_sec),...
            'SortStr','descend','MinPeakDistance',0.1*FS_loudness);
        % find dips in same way 
        [dip,diploc] = findpeaks(max(currentSig)-currentSig,'NPeaks',...
            floor(4*length_sec),'SortStr','descend','MinPeakDistance',0.1*FS_loudness); 
        % for each peak, get the level difference to the closest trough
        % to the left, and the closest to the right
        % not guaranteed to have the same number of peaks and troughs.
        if isempty(pk) % if no peaks in this band
            LDb(bark) = 0; % 
        else
            LD = zeros(length(pk),1);
            for pkidx = 1:length(pk) % for each peak 
                % find the closest dip in each direction
                currentPeakLocation = pkloc(pkidx);
                currentPeakHeight = pk(pkidx);
                % identify indices of dips to the right and left of current
                right = currentPeakLocation < diploc;
                left = currentPeakLocation > diploc;
                diploc_right = diploc(right);
                diploc_left = diploc(left);
                dip_right = dip(right);
                dip_left = dip(left);
                % find closest left and right peaks and troughs, and their
                % positions within diploc_right and diploc_left
                [~,positionIndiplocright] = ...
                    min(abs(currentPeakLocation-diploc_right));
                [~,positionIndiplocleft] = ...
                    min(abs(currentPeakLocation-diploc_left));
                %get level differences between the beaks and troughs
                Ld_left = currentPeakHeight - (max(currentSig)-dip_left(positionIndiplocleft));
                Ld_right = currentPeakHeight - (max(currentSig)-dip_right(positionIndiplocright));
                if ~(isempty(Ld_left) && isempty(Ld_right)) % if anything is found
                    LD(pkidx) = max([Ld_left,Ld_right]);    % save the level differences
                else
                    LD(pkidx) = 0; % no level difference here
                end
            end  
            LDb(bark) = mean(LD);
            % extract a scalar quantity from the array of Level Differences. 
        end
    end
    F = sum(LDb)/32; % calibration factor for 1 vacil
    % Correct for SPL 
    SPLDiff= SPL(signal) - 60;
    F = F * 2.^-(SPLDiff/20);
    F = F-0.4;
    F = max(0,F); % limiting and scaling
end