function S = Sharpness(signal, FS, type, fieldtype, x_ratio, t_duration, show)
    % S = Sharpness(signal, FS, type, fieldtype, x_ratio, t_duration, show)
    % Calculates the Sharpness of a signal in acum
    % 
    % INPUT:
    %   signal    : acoustic signal, monophonic (Pa)
    %   FS        : sampling frequency (Hz)
    %   type      : (optional parameter) 'mic' (default value) for omnidirectional sound
    %               recording, 'head' for dummy head measurement
    %   fieldtype : 'free' (default) for free field, 'diffuse' for diffuse field
    %   x_ratio   : percentage x used to compute Nx and Lx (percent) - default value is 5 % - see output
    %   t_duration: duration t used to compute Nt and Lt (second) - default value is 0.030 sec. - see output
    %   show      : optional parameter for some figures display.
    %            May be false (disable, default value) or true (enable).
    % 
    % OUTPUT: 
    %   S         : Sharpness of signal in acum
    %
    % REFERENCE:
    % Zwicker E. and Fastl H., "Psychoacoustics: Facts and models",
    % 2nd Edition, Springer-Verlag, Berlin, 1999
    %
    % DEPENDENCIES: 
    % Genesis Loudness Toolbox - GENESIS S.A. 
    % 
    % CONTACT:
    % Daniel Wallace: ORCID ID: https://orcid.org/0000-0003-0212-5395

    % Get loudness 
    res = Loudness_TimeVaryingSound_Zwicker(signal, FS, type, fieldtype, x_ratio, t_duration, show);

    n = length(res.time); % Number of temporal bins. 
    c = 0.11; % factor to scale Sharpness to 1 acum for a 60dB 1kHz critical band tone
    Nprime = res.InstantaneousSpecificLoudness;  % Matrix of loudness per bark band and per 0.02 second 
    barkAxis = res.barkAxis; % 
    InstantaneousSharpness = zeros(1,n); % initialise array - same size as InstantaneousLoudness
    g = @(z) 1.*(z<=14)+(0.00012.*z.^4 - 0.0056.*z.^3 + 0.1.*z.^2 - 0.81.*z + 3.51).*(z>14);
    % g(z) is defined in Zwicker and Fastl - Psychoacoustics: Facts and Models 
    % This weights each Bark band's contribution to sharpness towards high frequencies.
    for i = 1:n %for each time period
        InstantaneousSharpness(i) = c*sum(Nprime(i,:).*g(barkAxis).*barkAxis)./sum(Nprime(i,:));
    end
    % Arithmetic Mean of instantaneous sharpness correlates with sensation of sharpness 
    % See Ref [2]
    S = mean(InstantaneousSharpness,'omitnan');
end