%% IMAGE BASED INERTIAL IMPACT (IBII) TEST - Data Processing - v3.0ir
% Authors: Lloyd Fletcher, Jared Van-Blitterswyk
% PhotoDyn Group, University of Southampton
% http://photodyn.org/
% Date Created: 12th Dec. 2017 - v1.0ir
% Date Edited: 11th Mar. 2019 - v3.0ir
%
% Description: Takes input images processes them with the grid method to 
% obtain displacement fields. Further kinematic fields are derived from the
% displacement fields (acceleration, strain). Kinematic fields are then
% used to identify material properties including: stiffness and strength
%
% This code uses the grid processing tool box that can be found at:
% www.thegridmethod.net, developed by Grediac et al.
%
% The following papers describe the IBII method:
% [1] L. Fletcher, J. Van-Blitterswyk, F. Pierron, A Novel Image-Based 
%       Inertial Impact Test (IBII) for the Transverse Properties of 
%       Composites at High Strain Rates, J. Dynamic Behavior Mater. (2019). 
%       doi:10.1007/s40870-019-00186-y.
% [2] L. Fletcher, F. Pierron, An image-based inertial impact (IBII) test 
%       for tungsten carbide cermets, J. Dynamic Behavior Mater. 4 (2018) 
%       481504. doi:10.1007/s40870-018-0172-4.
% [3] J. Van Blitterswyk, L. Fletcher, F. Pierron, Image-based inertial 
%        impact test for composite interlaminar tensile properties, J. 
%        Dynamic Behavior Mater. 4 (2018) 543572. 
%        doi:10.1007/s40870-018-0175-1.
%
% Version 3.0: Contains support for the generalised stress-strain curves
% for isotropic and orthotropic elasticity. Removed old edge extrapolation
% method and consolidated to the new method developed by Jared. Fixed the 
% data save section and added a flag to reduce the number of fields 
% retained to save RAM and disk space. Other minor functionality
% improvements and code streamlining.
% NOTE: relies on the 2019 processing functions folder. 
%
% This work is licensed under the Creative Commons Attribution-
% NonCommercial-ShareAlike 4.0 International License. To view a copy of 
% this license, visit http://creativecommons.org/licenses/by-nc-sa/4.0/ or 
% send a letter to Creative Commons, PO Box 1866, Mountain View, CA 94042, 
% USA.
%
% If there are any issues with this code please contact: 
% Lloyd Fletcher: l.c.fletcher@soton.ac.uk / lloydcolinfletcher@gmail.com 
%
% Modifications
% FE interpolation to slice to validate equilibrium
% 17/08/2019 - Rotation equations for u1 and u2 disp fields
% 04/09/2019 - to account for off-axis composites with fibres pointing
% downwards at the impact edge. See FBD derivations for explanation.
%
%
clc
clear all
close all

fprintf('--------------------------------------------------------------\n')
fprintf('IMAGE-BASED INERTIAL IMPACT (IBII): Data Processing v3.0ir\n')
fprintf('--------------------------------------------------------------\n')

%% Switches

hardCodePath = false;
FEValidMode = false; % Load FE data files. Not sure if this fully works %%%%%%%%%%%%%%
calcKinFieldsFromDisp = true;
switches.Q12CorrectionExpData = 'off'; % Turn on to perform an e11 correction for the E22 identification. Note: uses experimental fibre strains, which can be noisy.
switches.interpolateFEStressToSlice = 'off'; % Set to 'on' with FE valid mode = 1 and calcKinFieldsFromDisp = 1 to interpolate FE stresses to slices and plot fields and material data. Not sure if this is in the correct place.  %%%%%%%

% Save options
saveAllWorkspaceVariables = true; % Set to true for saving all angled slice data for the orthoAngle case.

%% INITIALISE: Add path for processing functions
fprintf('Adding paths for processing functions.\n')

if ~hardCodePath
    funcPath = ([uigetdir('E:\experimentalData\ModulusPaper\ProcessingScripts\code1_IBII_TestProcessor\','Select path to processing functions'), '\']);
else
    % Add the path for the grid method code and other useful functions:
    funcPath = 'E:\IBIIProcessingCode_v3.0ir\ProcessingFunctions_2019\';

    % If the default path is not found we should find it
    if exist(funcPath,'file') ~= 7
        hWarn = warndlg('Folder for processing functions not found.','Function folder not found');
        waitfor(hWarn);
        funcPath = uigetdir(pwd,'Locate Processing Function Folder');
    end
end

% Add paths to Matlab
addpath(funcPath);
addpath([funcPath,'GridMethodToolbox\']);
addpath([funcPath,'Export_Fig\']);
addpath([funcPath,'linspecer\']);

%% LOAD RAW DATA FILES: Images or FE

if ~FEValidMode
    % LOAD RAW DATA FILES: Raw .tiff images
    fprintf('Loading reference image from the selected test data folder.\n')
    if ~hardCodePath
        [imageFile,imagePath] = uigetfile({'*.*','All Files'},'Select the first image in the sequence');
    else   
        imageFile = 'DefGridImage_001.tiff';
        imagePath = 'E:\Matlab_WorkingDirectory\1_IBIITest_Data\TestData_ID_CFIPQIso_HalfPulse\'; 
    end
else
    % LOAD RAW DATA FILES: FE Validation Data
    fprintf('Loading FE data from selected .mat file.\n')
    if ~hardCodePath
        [FEDataFile,FEDataPath] = uigetfile({'*.*','All Files'},'Select the FE data file');
    else   
        FEDataFile = '';
        FEDataPath = [pwd,'\'];    
    end
    load([FEDataPath,FEDataFile])
    imagePath = FEDataPath;
    imageFile = FEDataFile;
end

%% INITIALISE: Load Processing Parameter Data Structures
fprintf('Loading processing parameters file.\n')
initPath = imagePath;
initFile = 'processingParameters_v3_0ir.mat'; 

if exist([initPath,initFile],'file') ~= 2
    hWarn = warndlg('Processing parameter file does not exist.','Processing parameters not found');
    waitfor(hWarn);
    [initFile,initPath,~] = uigetfile('*.mat','Locate processing parameter file');
end

% Load the processing parameters from file
load([initPath,initFile])

% Store the FE valid mode parameter
globalOpts.FEValidMode = FEValidMode;
globalOpts.calcKinFieldsFromDisp = calcKinFieldsFromDisp;

%% PRE-PROCESSING: Grid Method Processing of Raw Images
if ~FEValidMode
    % Process the raw tiff images using the grid method code developed by
    % Grediac et al.
    fprintf('\n--------------------------------------------------------------\n')
    fprintf('GRID METHOD PROCESSING\n')
    fprintf('--------------------------------------------------------------\n')

    %--------------------------------------------------------------------------
    % GRID IMAGE PROCESSING

    % Check if the images have already been processed otherwise process the data
    gridDataSavePath = imagePath;
    gridDataFile = 'GridMethod_ProcessedData.mat';

    fprintf('Checking for existing processed data file.\n')
    processSliceData = true;
    if exist([gridDataSavePath,gridDataFile],'file') == 2
        fprintf('Grid method data file found.\n')
        if gridMethodOpts.autoLoadProccessedDataFile
            processSliceData = false;
        else   
            choice = questdlg('Processed grid data file found, process again?', ...
            'Process Grid Images', 'Yes','No','No');
            switch choice
                case 'Yes'
                    fprintf('.\n')
                    processSliceData = true;
                case 'No'
                    processSliceData = false;
            end
        end
    end

    if processSliceData
        fprintf('Processing images using the grid method toolbox.\n')
        % Process the image squence with the grid method toolbox
        [grid,pos,disp] = func_gridMethodImageProcessing(imagePath,imageFile,...
            grid,gridMethodOpts,imageNoise);

        % Save the data to file so a reprocess is not necessary
        fprintf('Saving processed grid data to: %s.\n',gridDataFile)
        save([gridDataSavePath,gridDataFile],'grid','pos','disp')

        fprintf('Grid Method Processing Complete.\n')
    else
        fprintf('Loading processed grid data from: %s.\n',gridDataFile)
            load([gridDataSavePath,gridDataFile]);
    end


    %--------------------------------------------------------------------------
    % Update Geometry and Number of Frames Based on Displacement Matrix Size
    fprintf('Updating geometry variables based on the size of displacement field matrices.\n')
    [specimen,grid] = func_updateSpecGeom(specimen,grid,disp);

    % Currently the rotations are unused so remove them to save RAM
    disp = rmfield(disp,'rot');  
end

%--------------------------------------------------------------------------
 % Create both angled slice structs for tranverse and parallel fibre cuts.
if strcmp('orthotropicAngle',globalOpts.matModel)

    % Create both angled slice structs for tranverse and parallel fibre cuts.
    angledSlice1 = func_initAngledSliceStruct(specimen,pos,angledSlice1);
    angledSlice2 = func_initAngledSliceStruct(specimen,pos,angledSlice2);

end

%% POST-PROCESSING: Smoothing and Kinematic Field Derivation
% Smooth the displacement data and then calculate acceleration and strains
% Extrapolate the data to account for the missing pitch on the edges

fprintf('\n--------------------------------------------------------------\n')
fprintf('POST PROCESSING: Smoothing and Kinematic Field Calculation\n')
fprintf('--------------------------------------------------------------\n')

% Calculate the kinematic fields from displacement fields using
% displacements from images or displacements from FE data
if calcKinFieldsFromDisp
    %--------------------------------------------------------------------------
    % Load the Reference Image and Determine Where the Free Edge is
    fprintf('Obtaining and setting the free edge location.\n')
    [freeEdge,specimen,disp] = func_getFreeEdge(globalOpts.hardCodeFreeEdge,...
        imagePath,imageFile,specimen,disp);

    %--------------------------------------------------------------------------
    % Smooth and Calculate Strain
    fprintf('Calculating strain from the displacement fields.\n')
    [disp,strain,strainRate] = func_smoothCalcStrain(globalOpts,pos,time,...
        grid,disp,smoothingOpts,extrapOpts);

    %--------------------------------------------------------------------------
    % Smooth and Calculate Acceleration
    fprintf('Calculating acceleration from the displacement fields.\n')
    [disp,vel,accel] = func_smoothCalcAccel(pos,time,grid,disp,smoothingOpts,...
        extrapOpts,diffOpts);

    % Remove some 3D fields from the structs to save memory.
    if globalOpts.reduceMemory
        disp.x = disp.extrap.x;
        disp.y = disp.extrap.y;
        disp = rmfield(disp,'extrap');
        disp = rmfield(disp,'tSmooth');
    end
else
    fprintf('Using kinematic fields directly from FE data.\n')
    % For pure FE data the kinematic fields can be taken directly from the
    % FE calculation
    disp.xAvg = func_avgFFVarOverWidth(disp.x);
    accel.xAvg = func_avgFFVarOverWidth(accel.x);
    accel.yAvg = func_avgFFVarOverWidth(accel.y);                           % Added 05/09/2019 SP
    strain.xAvg = func_avgFFVarOverWidth(strain.x);
    strain.yAvg = func_avgFFVarOverWidth(strain.y);
    strain.sAvg = func_avgFFVarOverWidth(strain.s);
    [strainRate,~] = func_calculateStrainRate(strain,time,smoothingOpts,true);
    strainRate.xAvg = func_avgFFVarOverWidth(strainRate.x);
    strainRate = func_calcMaxStrainRate(smoothingOpts,grid,strainRate); % Add 'grid' as input var for processing FE data
    disp.extrap.x = disp.x;
    disp.extrap.y = disp.y;
end

%% --------------------------------------------------------------------------
% Rotate Kinematic Fields to Material Co-ords
% NOTE: only required for orthotropicAngle material model
if strcmp('orthotropicAngle',globalOpts.matModel)
    fprintf('Rotating kinematic fields into the material co-ords, angle = %0.2f.\n',material.rotAngle)
    
    % Note disp.mat11 is rotated relative to the x-axis.
    fprintf('Rotating the displacement fields.\n')
    [disp.mat11,disp.mat22] = func_rotateVector2D(disp.x,disp.y,material.rotAngle);
    
    fprintf('Rotating the acceleration fields.\n')
    [accel.mat11,accel.mat22] = func_rotateVector2D(accel.x,accel.y,material.rotAngle);
    
    fprintf('Rotating strain fields to the material co-ord system.\n')
    [strain.mat11,strain.mat22,strain.mat12] = func_rotateStrain2D(...
        strain.x,strain.y,strain.s,material.rotAngle);
    
    fprintf('Rotating strain rate fields to the material co-ord system.\n')
    [strainRate.mat11,strainRate.mat22,strainRate.mat12] = func_rotateStrain2D(...
        strainRate.x,strainRate.y,strainRate.s,material.rotAngle);
end

%% --------------------------------------------------------------------------
% Create material coordinate fields for the reduced orthotropic case.
% Note 1: cannot use rotation equations, as they give a sign error on the wavefront
% Note 2: SS curves based on GCSYS so signs wont reflect what is shown in MCSYS fields
if strcmp('orthotropicReduced',globalOpts.matModel)
    fprintf('Creating the UD90 case material coordinate fields.\n')
    disp.mat11 = -disp.y; % pos.y is upside down
    disp.mat22 = disp.x;
    accel.mat11 = -accel.y; % calculated from disp.y
    accel.mat22 = accel.x;
    strain.mat11 = -strain.y; % calculated from disp.y
    strain.mat22 = strain.x;
    strain.mat12 = -strain.s; % calculated from disp.y
    strainRate.mat11 = -strainRate.y;
    strainRate.mat22 = strainRate.x;
    strainRate.mat12 = -strainRate.s; % calculated from disp.y
end

%% --------------------------------------------------------------------------
% STANDARD STRESS GAUGE: CALCULATION
fprintf('Calculating average stress with the stress-gauge equation.\n')

% Standard Stress Gauge equation calculation - xAvg
[stress.xAvg,~] = func_stressGaugeProcess(material, time, pos, accel.x, accel.xAvg);

% Standard Stress Gauge equation calculation
[stress.sAvg,~] = func_stressGaugeProcess(material,time,pos,accel.y,accel.yAvg);

%% --------------------------------------------------------------------------

% Angled slice interpolation - Strain and accel calcs along slices

if strcmp('orthotropicAngle',globalOpts.matModel)
    fprintf('Calculating strain averages along both angled slices.\n')
    %//////////////////////////////////////////////////////////////////
    % Slice 1, along the fibres: strain calculation
    if angledSlice1.xMaxInd > 0 
        fprintf('Interpolating average strain over each angled slice for slice 1.\n')
        
        [strainSlice1.avg22,strainSlice1.field22] = ...
            func_calcRotatedFieldAvgAlongSlice(angledSlice1,pos,strain.mat22);
            
        [strainSlice1.avg12,strainSlice1.field12] = ...
            func_calcRotatedFieldAvgAlongSlice(angledSlice1,pos,strain.mat12);
        
        [strainSlice1.avg11,strainSlice1.field11] = ...
            func_calcRotatedFieldAvgAlongSlice(angledSlice1,pos,strain.mat11);
    
    else
        fprintf('Slice 1 is too long for the sample length. Cancelling calculation.\n')
        strainSlice1 = nan;
    end

    %//////////////////////////////////////////////////////////////////
    % Slice 2, transverse to the fibres: strain calculation
    if angledSlice2.xMaxInd > 0 
        fprintf('Interpolating average strain over each angled slice for slice 2.\n')
        [strainSlice2.avg11,strainSlice2.field11] = ...
            func_calcRotatedFieldAvgAlongSlice(angledSlice2,pos,strain.mat11);
        [strainSlice2.avg12,strainSlice2.field12] = ...
            func_calcRotatedFieldAvgAlongSlice(angledSlice2,pos,strain.mat12);
        [strainSlice2.avg22,strainSlice2.field22] = ...
            func_calcRotatedFieldAvgAlongSlice(angledSlice2,pos,strain.mat22);
    else
        fprintf('Slice 2 is too long for the sample length. Cancelling calculation.\n')
        strainSlice2 = nan;
    end

%% -------------------------------------------------------------------------
% Interpolate FE stress fields to slice
%--------------------------------------------------------------------------
    if strcmp('on',switches.interpolateFEStressToSlice)
        
        % Stress rotations
        fprintf('Rotating FE stress fields to the material co-ord system.\n')
        [stress.mat11, stress.mat22, stress.mat12] = func_rotateStrain2D(...
        stress.x, stress.y, stress.s, material.rotAngle);
        
        fprintf('Calculating FE stress averages along both angled slices.\n')
    
        %//////////////////////////////////////////////////////////////////
        % Slice 1, along the fibres: FE stress interpolation to slice
        if angledSlice1.xMaxInd > 0 
            fprintf('Interpolating average strain over each angled slice for slice 1.\n')

            [stressSlice1.avg22,stressSlice1.field22] = ...
                func_calcRotatedFieldAvgAlongSlice(angledSlice1,pos,stress.mat22);
                
            [stressSlice1.avg12,stressSlice1.field12] = ...
                func_calcRotatedFieldAvgAlongSlice(angledSlice1,pos,stress.mat12);

            [stressSlice1.avg11,stressSlice1.field11] = ...
                func_calcRotatedFieldAvgAlongSlice(angledSlice1,pos,stress.mat11);
           
        else
            fprintf('Slice 1 is too long for the sample length. Cancelling calculation.\n')
            strainSlice1 = nan;
        end

        %//////////////////////////////////////////////////////////////////
        % Slice 2, transverse to the fibres: FE stress interpolation to slice
        if angledSlice2.xMaxInd > 0 
            fprintf('Interpolating average strain over each angled slice for slice 2.\n')
            [stressSlice2.avg11,stressSlice2.field11] = ...
                func_calcRotatedFieldAvgAlongSlice(angledSlice2,pos,stress.mat11);
            [stressSlice2.avg12,stressSlice2.field12] = ...
                func_calcRotatedFieldAvgAlongSlice(angledSlice2,pos,stress.mat12);
            [stressSlice2.avg22,stressSlice2.field22] = ...
                func_calcRotatedFieldAvgAlongSlice(angledSlice2,pos,stress.mat22);
        else
            fprintf('Slice 2 is too long for the sample length. Cancelling calculation.\n')
            strainSlice2 = nan;
        end
    
    else
        %//////////////////////////////////////////////////////////////////
        % Slice 1, along the fibres: accel calculation
        if angledSlice1.xMaxInd > 0 
            fprintf('Calculating average stress along slice 1 with the angled stress-gauge at an angle of: %.2f.\n',material.rotAngle)

            [accelSlice1.surfAvg11,angledSlice1.surfArea,angledSlice1.Xcoord] = func_calcRotatedFieldAreaAvgAlongSlice(...
                angledSlice1,specimen,time,pos,accel.mat11);

            [accelSlice1.surfAvg22,~] = func_calcRotatedFieldAreaAvgAlongSlice(...
                angledSlice1,specimen,time,pos,accel.mat22);

            % Calculate the standard angled stress-gauge
            % Modified 04/09/2019 - to account for off-axis composites with fibres pointing
            % downwards at the impact edge. See FBD derivations for explanation. 
            if angledSlice1.angle > 0 
            stressSlice1.avg22 = -material.rho*angledSlice1.surfArea./...
                angledSlice1.length.*accelSlice1.surfAvg22; 

            stressSlice1.avg12 = -material.rho*angledSlice1.surfArea./...
                angledSlice1.length.*accelSlice1.surfAvg11;

            else 
            stressSlice1.avg22 = material.rho*angledSlice1.surfArea./...
                angledSlice1.length.*accelSlice1.surfAvg22;    

            stressSlice1.avg12 = material.rho*angledSlice1.surfArea./...
                angledSlice1.length.*accelSlice1.surfAvg11;
            end

        else
            fprintf('Slice 1 is too long for the sample length. Cancelling calculation.\n')
            accelSlice1 = nan;
            stressSlice1 = nan;
        end

        %% //////////////////////////////////////////////////////////////////
        % Slice 2, transverse to the fibres: accel calculation
        if angledSlice2.xMaxInd > 0
            fprintf('Calculating average stress along slice 2 with the angled stress-gauge at an angle of: %.2f.\n',material.rotAngle)

            [accelSlice2.surfAvg11,angledSlice2.surfArea, angledSlice2.Xcoord] = func_calcRotatedFieldAreaAvgAlongSlice(...
                angledSlice2,specimen,time,pos,accel.mat11);

            [accelSlice2.surfAvg22,~] = func_calcRotatedFieldAreaAvgAlongSlice(...
                angledSlice2,specimen,time,pos,accel.mat22);

            % Calculate the standard angled stress-gauge
            stressSlice2.avg11 = material.rho*angledSlice2.surfArea./...
                angledSlice2.length.*accelSlice2.surfAvg11;

            stressSlice2.avg12 = material.rho*angledSlice2.surfArea./...
                angledSlice2.length.*accelSlice2.surfAvg22;
        else
            fprintf('Slice 2 is too long for the sample length. Cancelling calculation.\n')
            accelSlice2 = nan;
            stressSlice2 = nan;
        end 
    end
end

%% --------------------------------------------------------------------------
% STANDARD STRESS GAUGE: find stiffness(es) by fitting the stress-strain curves
fprintf('Identifying stiffness(es) by fitting the stress-strain curves.\n')

if strcmp('orthotropicReduced',globalOpts.matModel)
    fprintf('Linearly fitting the stress and strain (e_xx) components to obtain Qxx.\n')
    % Fit the stress strain curves to obtain Qxx/Exx
    % identStiffSG = func_identQxxFromStressStrainCurve(stressGaugeOpts,stress,strain);
    [identStiffSG.QxxVsL, identStiffSG.QxxLinFitCoeffs, identStiffSG.QxxFitFrameRange]...
        = func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stress.xAvg,strain.xAvg);
    
    % Set the range over which the average stiffness is identified
    stressGaugeOpts.avgQVsLRange = round(stressGaugeOpts.avgQVsLRangePc(1)*length(pos.x))...
        :round(stressGaugeOpts.avgQVsLRangePc(2)*length(pos.x));
    
    % Calculate Identified Averages
    identStiffSG.QxxAvgOverL = nanmean(identStiffSG.QxxVsL(stressGaugeOpts.avgQVsLRange));
    identStiffSG.ExxVsL = identStiffSG.QxxVsL;
    identStiffSG.ExxAvgOverL = nanmean(identStiffSG.ExxVsL(stressGaugeOpts.avgQVsLRange));
    
    fprintf('Linearly fitting the stress and strain (e_xy) components to obtain Qss.\n')
    % Fit the stress strain curves to obtain Qss/Gxy
    [identStiffSG.QssVsL,identStiffSG.QssLinFitCoeffs, identStiffSG.QssFitFrameRange]...
        = func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stress.sAvg,strain.sAvg);
    
    % Set the range over which the average stiffness is identified
    stressGaugeOpts.avgQVsLRange = round(stressGaugeOpts.avgQVsLRangePc(1)*length(pos.x))...
        :round(stressGaugeOpts.avgQVsLRangePc(2)*length(pos.x));
    
    % Calculate Identified Averages
    identStiffSG.QssAvgOverL = nanmean(identStiffSG.QssVsL(stressGaugeOpts.avgQVsLRange));
    identStiffSG.GxyVsL = identStiffSG.QssVsL;
    identStiffSG.GxyAvgOverL = nanmean(identStiffSG.GxyVsL(stressGaugeOpts.avgQVsLRange));
    
    
elseif strcmp('orthotropicAngle',globalOpts.matModel)
    
    % Set the range over which the average stiffness is identified (currently set to all slices for off-axis composites)
    stressGaugeOpts.avgQVsLRange = round(1:angledSlice1.xMaxInd);
       
    if angledSlice1.xMaxInd > 0      
        fprintf('Linearly fitting the 22 stress and strain components to obtain Q22.\n')
        [identStiffSG.Q22VsL, identStiffSG.Q22LinFitCoeffs, identStiffSG.Q22FitFrameRange] =...
            func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stressSlice1.avg22,strainSlice1.avg22);
        

        fprintf('Linearly fitting the 12 stress and strain components to obtain Q66.\n')
        [identStiffSG.Q66VsL_Slice1, identStiffSG.Q66LinFitCoeffs_Slice1, identStiffSG.Q66FitFrameRange_Slice1] = ...
            func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stressSlice1.avg12,strainSlice1.avg12);
        
    end
    
    if angledSlice2.xMaxInd > 0
        fprintf('Linearly fitting the 11 stress and strain components to obtain Q11.\n')
        [identStiffSG.Q11VsL, identStiffSG.Q11LinFitCoeffs]...
            = func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stressSlice2.avg11,strainSlice2.avg11);
       

        fprintf('Linearly fitting the 12 stress and strain components to obtain Q66.\n')
        [identStiffSG.Q66VsL_Slice2, identStiffSG.Q66LinFitCoeffs_Slice2, identStiffSG.Q66FitFrameRange_Slice2] = ...
            func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stressSlice2.avg12,strainSlice2.avg12);
        
    end
    
    % Identified stiffness vs position
    identStiffSG.Q22AvgOverL = nanmean(identStiffSG.Q22VsL(stressGaugeOpts.avgQVsLRange));
    identStiffSG.Q66AvgOverL_Slice1 = nanmean(identStiffSG.Q66VsL_Slice1(stressGaugeOpts.avgQVsLRange));
    identStiffSG.Q11AvgOverL = nanmean(identStiffSG.Q11VsL(stressGaugeOpts.avgQVsLRange));
    identStiffSG.Q66AvgOverL_Slice2 = nanmean(identStiffSG.Q66VsL_Slice2(stressGaugeOpts.avgQVsLRange));
    
    % e11 correction
    if strcmp('on', switches.Q12CorrectionExpData)                                       %%%%%%Add_section_toimport this
    
    % Q12 calc
    identStiffSG.Q12VsL = (material.nuxy*identStiffSG.Q22VsL)/(1-(material.nuyx*material.nuxy));
    % Translate
    identStiffSG.Q12VsL = -identStiffSG.Q12VsL';
    % Corrected stress 
    stressSlice1.avg22Corr = stressSlice1.avg22 - identStiffSG.Q12VsL.*strainSlice1.avg11;
        if angledSlice1.xMaxInd > 0      
            fprintf('Linearly fitting the corrected 22 stress and strain components to obtain Q22 Corr.\n')
            [identStiffSG.Q22VsLCorr,identStiffSG.Q22LinFitCoeffsCorr] =...
                func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stressSlice1.avg22Corr,strainSlice1.avg22);
            identStiffSG.Q22AvgOverLCorr = nanmean(identStiffSG.Q22VsLCorr(stressGaugeOpts.avgQVsLRange));
            identStiffSG.E22AvgOverLCorr = identStiffSG.Q22AvgOverLCorr*(1-(material.nuxy*material.nuyx));
        end
    end
   
end


%% POST-PROCESSING - Image Sequences of Kinematic Fields
fprintf('\n--------------------------------------------------------------\n')
fprintf('POST PROCESSING: Image Sequences\n')
fprintf('--------------------------------------------------------------\n')

% Set the plot properties
plotProps = func_initPlotPropsStruct(plotParams.formatType);

%--------------------------------------------------------------------------
% Create Video Sequences of Disp, Strain, Accel and Strain Rate
% NOTE: these images are required for the crack location identification
if strcmp(globalOpts.plotImageSeqs,'no')
    fprintf('Image sequence plotting disabled.\n')
else
    fprintf('Plotting image sequence of disp, accel, strain and strain rate.\n')
    fprintf('NOTE: image sequence used to identify crack location for strength identification.\n')
    
    func_plotAllFullFieldVideos(globalOpts,plotParams,imagePath,pos,time,...
        disp,strain,strainRate,accel, material);
end

%% POST-PROCESSING: Diagnostic Figures for Stiffness Identification
fprintf('\n--------------------------------------------------------------\n')
fprintf('POST PROCESSING: Stiffness Identification Diagnostic Figures\n')
fprintf('--------------------------------------------------------------\n')

%--------------------------------------------------------------------------
% Check if save folder exists, if not, create it.
savePath = [imagePath,'ProcessedData_DiagnosticFigs'];
savePaths{1} = savePath;

if strcmp(globalOpts.plotImageSeqs,'prompt')
    % Check if the save path exits, if it does ask the user if it
    plotDiagFigs = func_checkIfPathExistsDlg(savePath,...
        'Folder for diagnostic figures exists, plot again?','Plot Diagnostics?');
elseif strcmp(globalOpts.plotImageSeqs,'yes') || strcmp(globalOpts.plotImageSeqs,'diagnostic')
    plotDiagFigs = true;
    if exist(savePath,'file') ~= 7
        mkdir(savePath);
    end
else
    plotDiagFigs = false; 
end


if plotDiagFigs
    fprintf('Ploting diagnostic figures.\n')
    
    %----------------------------------------------------------------------
    % Plot General Diagnostics: Fields Averages and Loading Pulse
    func_plotGeneralFieldDiagnostics(savePath,plotParams,...
        specimen,time,pos,disp,accel,strain,strainRate,stress)
       
    %% --------------------------------------------------------------------
    % STRESS-GAUGE: Plot stress-strain curves at several locations
    fprintf('\tPlotting stress-strain curves.\n')
    if strcmp('orthotropicReduced',globalOpts.matModel)
        func_plotSSCurvesSGRedOrtho(savePath,plotParams, ssCurvePlotParams,pos,stress,strain,identStiffSG)

    elseif strcmp('orthotropicAngle',globalOpts.matModel)
        func_plotSSCurvesSGOrthoAng(savePaths{1},plotParams,ssCurvePlotParams,material,...
            angledSlice1,angledSlice2,pos,stressSlice1,strainSlice1,stressSlice2,strainSlice2,identStiffSG)
        
    else % Isotropic case as default for stress-gauge
        func_plotSSCurvesSGIso(savePath,ssCurvePlotParams,pos,stress,strain)
    end
    
    %----------------------------------------------------------------------
    % STRESS-GAUGE: Plot identified Stiffness vs Length
    fprintf('\tPlotting identified stiffness over the length of the specimen.\n')
    
    if strcmp('orthotropicReduced',globalOpts.matModel)
        ssCurveIdentPlotOpts.indLRange = 1:length(pos.x);
        func_plotSGIdentStiffVsLOrthoRed(savePath,plotParams,ssCurveIdentPlotOpts,pos,material,identStiffSG)
                
    elseif strcmp('orthotropicAngle',globalOpts.matModel)
        func_plotSGIdentStiffVsLOrthoAng(savePaths{1},plotParams,... 
        ssCurveIdentPlotOpts,pos,material,identStiffSG,...
        angledSlice1,angledSlice2,stressGaugeOpts);
    
    else % Isotropic case as default for stress-gauge
        func_plotSGIdentStiffVsLIso(savePath,ssCurveIdentPlotOpts,pos,material,identStiffSG)
    end

end

%% POST-PROCESSING: Strength Identification

if globalOpts.identifyStrength
    
    fprintf('\n--------------------------------------------------------------\n')
    fprintf('POST PROCESSING: Strength Identification\n')
    fprintf('--------------------------------------------------------------\n')
    
    %--------------------------------------------------------------------------
    % Obtain Fracture Location
    % Check if the fracture file exists, if it does ask to load it
    loadFractFile = false;
    fractFile = 'fracture.mat';
    fractFilePath = imagePath;
    if exist([fractFilePath,fractFile],'file') == 2
        if strOpts.autoLoadFractureFile
            loadFractFile = true;
            load([fractFilePath,fractFile])
        else
            % Load the file and present the parameters:
            load([fractFilePath,fractFile])
            dlgStr = {'Fracture file found with:',...
                ['locX: ',sprintf('%i',fracture.locX)],...
                ['locY: ',sprintf('%i',fracture.locY)],...
                'Load file?'};

            choice = questdlg(dlgStr,'Load fracture file?', 'Yes','No','No');
            switch choice
                case 'Yes'
                    loadFractFile = true;
                case 'No'
                    loadFractFile = false;
                    clear fracture
            end
        end
    end

    % If the fracture location doesn't already exist, let the user input it
    if ~loadFractFile
        % Check if the user wants to manually input this data or not
        inputFractLoc = false;
        choice = questdlg('Manually input fracture location and time?', ...
        'Fracture Location Input', 'Manual','Graphic Pick','Manual');
        switch choice
            case 'Manual'
                inputFractLoc = true;
            case 'Graphic Pick'
                inputFractLoc = false;
        end

        if inputFractLoc
            fprintf('Obtaining fracture location (x,y) from user input by manual input.\n')
            fracture = func_manualInputFractureLoc(pos,time);
        else
            fprintf('Obtaining fracture location (x,y) from user input by picking.\n')
            fracture = func_findFractureLoc(imagePath,pos,time,disp,accel,strain,strainRate);
        end
        % Save the fracture location to file
        save([fractFilePath,fractFile],'fracture')
    end

    %--------------------------------------------------------------------------
    % Iniatialise strength processing options
    strOpts.virtualGauge = func_createVirtualGauge(strOpts.virtualGauge,fracture,strain);
    
    % Depending on material model set the stiffness matrix for stres
    % calculation from strain
    if strcmp('isotropic',globalOpts.matModel)
        if strcmp('SG',strOpts.stiffnessMethod)
            strOpts.stiffnessQxx = identStiffSG.QxxAvgOverL;
            strOpts.stiffnessQxy = material.Qxy;
        elseif strcmp('VF',strOpts.stiffnessMethod)
            strOpts.stiffnessQxx = identStiffVFOpt.QxxAvgOverL;
            strOpts.stiffnessQxy = identStiffVFOpt.QxyAvgOverL; 
        else
            strOpts.stiffnessQxx = nanmean([identStiffSG.QxxAvgOverL,identStiffVFOpt.QxxAvgOverT]);
            strOpts.stiffnessQxy = identStiffVFOpt.QxyAvgOverT;
        end
        strOpts.stiffnessQMat = [strOpts.stiffnessQxx,strOpts.stiffnessQxy,0;...
                                     strOpts.stiffnessQxy,strOpts.stiffnessQxx,0;...
                                     0,0,(strOpts.stiffnessQxx-strOpts.stiffnessQxy)/2]; 
    elseif strcmp('orthotropicReduced',globalOpts.matModel)
        if strcmp('SG',strOpts.stiffnessMethod)
            strOpts.stiffnessQxx = identStiffSG.QxxAvgOverL;
        elseif strcmp('VF',strOpts.stiffnessMethod)
            strOpts.stiffnessQxx = identStiffVFOpt.QxxAvgOverL;
        else
            strOpts.stiffnessQxx = nanmean([identStiffSG.QxxAvgOverL,identStiffVFOpt.QxxAvgOverT]);
        end
        strOpts.stiffnessQMat = [strOpts.stiffnessQxx,0,0;...
                                 0,0,0;...
                                 0,0,0]; 
    elseif strcmp('orthotropic',globalOpts.matModel)
        fprintf('WARNING: orthotropic strength identification not implemented.\n')
    elseif strcmp('orthotropicAngle',globalOpts.matModel)
        fprintf('WARNING: angled orthotropic strength identification not implemented.\n')
    else
        fprintf('WARNING: specifed material model not recognised.\n')
    end
    
    %--------------------------------------------------------------------------
    % Calculate the stress from the strains using stiffness matrix
    fprintf('Calculating stress from strain using the identified stiffness matrix.\n')
    [stress.QRecon.x,~,~] = func_calcStressFromStrain2D(strOpts.stiffnessQMat,...
        strain.x,strain.y,strain.s);
    
    fprintf('Calculating average stress and strain in the virtual gauge area at the fracture location.\n')
    [stress,strain] = func_calcAvgStressStrainInGaugeArea(globalOpts,strOpts,stress,strain);
    
    % Plot the difference between the acceleration and strain stress measures
    fprintf('Plotting the difference between the stress gauge and stress from strain vs time.\n');
    hf = func_plotStressDiffVsTime(plotParams,savePath,time,stress,fracture);
    saveFile = [savePath,'\','StrIdent_StressDiff']; 
    print(hf,saveFile,plotProps.format,plotProps.saveRes)
    saveas(hf,saveFile,'fig')

    %--------------------------------------------------------------------------
    % Manually or Automatically Identify the Strength
    if strcmp(strOpts.method,'Manual')
        % MANUAL Strength Identification
        fprintf('ANALYSE STRESS DIFFERENCE TO DETERMINE FRACTURE POINT.\n');
        fprintf('Press any key to continue.\n');
        pause;

        % Get the Fracture Frame/Time
        fprintf('Obtaining fracture frame from user input by manual input.\n')
        fracture.strengthFrame = func_manualInputFractureFrame(pos,time);

        % Identify strength
        fprintf('Identifying strength from stress gauge and strains using the virtual gauge area.\n')
        [identStrength.stressGauge,identStrength.linStressGauge,identStrength.fromStrain] =...
            func_identifyStrength(time,strOpts.virtualGauge,fracture,stress,strain,strainRate);
    else
        % AUTOMATED Strength Identification
        [identStrength.stressGauge,identStrength.linStressGauge,fracture] = ...
                func_identifyStrengthAutomated(strOpts.virtualGauge,fracture,stress);
        identStrength.fromStrain = nan;
    end

    %--------------------------------------------------------------------------
    % DIAGNOSTIC IMAGE SEQ 1: Width averaged stress measures and fields
    fprintf('Plotting diagnostic image sequence 1: width average stress and fields.\n')
    plotParams.cutPxY = 0;

    strImageSeqSavePath = [imagePath,'StrIdent_ImageSeq_WAvg\'];
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        plotImageSeq = func_checkIfPathExistsDlg(strImageSeqSavePath,...
            'Width averaged strength identification image sequence folder found, plot again?','Plot Image Sequence?');
    elseif strcmp(globalOpts.plotImageSeqs,'yes') || strcmp(globalOpts.plotImageSeqs,'diagnostic')
        plotImageSeq = true;
        if exist(strImageSeqSavePath,'file') ~= 7
            mkdir(strImageSeqSavePath);
        end
    else
        plotImageSeq = false; 
    end

    % Video of width average stress, raw strain, strain, accel, strain rate
    if plotImageSeq
        func_plotIdentStrVid_WAvg(plotParams,strImageSeqSavePath,pos,time,...
            disp,accel,strain,strainRate,stress,fracture);
    end

    %--------------------------------------------------------------------------
    % DIAGNOSTIC IMAGE SEQ 2: Stress fields and local stress measures/curve
    fprintf('Plotting diagnostic image sequence 2: stress fields and stress measures.\n')

    strImageSeqSavePath = [imagePath,'StrIdent_ImageSeq_Field\'];
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        plotImageSeq = func_checkIfPathExistsDlg(strImageSeqSavePath,...
            'Stress field strength identification image sequence folder found, plot again?','Plot Image Sequence?');
    elseif strcmp(globalOpts.plotImageSeqs,'yes') || strcmp(globalOpts.plotImageSeqs,'diagnostic')
        plotImageSeq = true;
        if exist(strImageSeqSavePath,'file') ~= 7
            mkdir(strImageSeqSavePath);
        end
    else
        plotImageSeq = false; 
    end

    % Video of stress fields from linear gauge and strain, local 
    % stress-strain curve and stress difference 
    if plotImageSeq
        if strcmp('isotropic',globalOpts.matModel)
            ssCurveLabels.xStr = '$\overline{\epsilon_{xx} + \nu \epsilon_{yy}}^{A}$ [$mm.m^{-1}$]';
            ssCurveLabels.yStr = '$\overline{\sigma_{xx}}^{A}$ [$MPa$]';
            func_plotIdentStrVid_Field(plotParams,strImageSeqSavePath,pos,time,...
                stress,stress.virtGaugeAvg.x,strain.virtGaugeAvg.xny,ssCurveLabels,fracture,strOpts,false);

        elseif strcmp('orthotropicReduced',globalOpts.matModel)
            ssCurveLabels.xStr = '$\overline{\epsilon_{xx}}^{A}$ [$mm.m^{-1}$]';
            ssCurveLabels.yStr = '$\overline{\sigma_{xx}}^{A}$ [$MPa$]';
            func_plotIdentStrVid_Field(plotParams,strImageSeqSavePath,pos,time,...
                stress,stress.virtGaugeAvg.x,strain.virtGaugeAvg.x,ssCurveLabels,fracture,strOpts,false);

        elseif strcmp('orthotropic',globalOpts.matModel)
            fprintf('WARNING: orthotropic strength identification not implemented.\n')

        elseif strcmp('orthotropicAngle',globalOpts.matModel)
            fprintf('WARNING: angled orthotropic strength identification not implemented.\n')

        else
            fprintf('WARNING: specifed material model not recognised.\n')
        end 
    end
    
    %--------------------------------------------------------------------------
    % Plot the strain rate at the fracture location up to the fracture frame
    fprintf('Calculating maximum strain rate at the fracture location.\n')
    fprintf('Plotting the strain rate at the fracture location.\n')
    strainRate.strStrainRate = func_plotStrainRateAtFracLoc(plotParams,savePath,...
        time,strOpts.virtualGauge,fracture,strainRate);
    
    % Update the maximum strain rate variables
    if smoothingOpts.spatialSmooth == true
        edgePx = grid.pxPerPeriod+round(smoothingOpts.spatialKernal(1)/2)+1;
    else
        edgePx = grid.pxPerPeriod+1;
    end
    xRange = edgePx:size(strainRate.xAvg,1) - edgePx;
    tRange = 1:fracture.strengthFrame;
    % Ignore the edges where the data is corrupted by smoothing effects
    strainRate.xAvgMinPreFract = min(min(strainRate.xAvg(xRange,tRange)));
    strainRate.xAvgMaxPreFract = max(max(strainRate.xAvg(xRange,tRange)));
    identStrainRate.strStrainRate = strainRate.strStrainRate;
    
    %--------------------------------------------------------------------------
    % FINAL DIAG FIG: Plot the stress fields maps and local stress-strain curves
    fprintf('Plotting diagnostic images at fracture time: stress fields and stress measures.\n')
     if strcmp('isotropic',globalOpts.matModel)
        ssCurveLabels.xStr = '$\overline{\epsilon_{xx} + \nu \epsilon_{yy}}^{A}$ [$mm.m^{-1}$]';
        ssCurveLabels.yStr = '$\overline{\sigma_{xx}}^{A}$ [$MPa$]';
        hf = func_plotIdentStrVid_Field(plotParams,savePath,pos,time,...
            stress,stress.virtGaugeAvg.x,strain.virtGaugeAvg.xny,ssCurveLabels,fracture,strOpts,true);
        saveFile = [savePath,'\','StrIdent_Diagnostics'];
        %export_fig(saveFile,'-eps','-pdf','-c[Inf,Inf,Inf,Inf]',hf)
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')

    elseif strcmp('orthotropicReduced',globalOpts.matModel)
        ssCurveLabels.xStr = '$\overline{\epsilon_{xx}}^{A}$ [$mm.m^{-1}$]';
        ssCurveLabels.yStr = '$\overline{\sigma_{xx}}^{A}$ [$MPa$]';
        hf = func_plotIdentStrVid_Field(plotParams,savePath,pos,time,...
            stress,stress.virtGaugeAvg.x,strain.virtGaugeAvg.x,ssCurveLabels,fracture,strOpts,true);
        saveFile = [savePath,'\','StrIdent_Diagnostics'];
        %export_fig(saveFile,'-eps','-pdf','-c[Inf,Inf,Inf,Inf]',hf)
        print(hf,saveFile,plotProps.format,plotProps.saveRes)
        saveas(hf,saveFile,'fig')

    elseif strcmp('orthotropic',globalOpts.matModel)
        fprintf('WARNING: orthotropic strength identification not implemented.\n')

    elseif strcmp('orthotropicAngle',globalOpts.matModel)
        fprintf('WARNING: angled orthotropic strength identification not implemented.\n')

    else
        fprintf('WARNING: specifed material model not recognised.\n')
     end 
    
    %--------------------------------------------------------------------------
    % Get the max stress and its location and time over the whole FOV
    % Useful for specimens that did not actually fail
    identStrength.FOVlims.rangeX = grid.pxPerPeriod+1:size(stress.xLinearGauge,2)-grid.pxPerPeriod-1;
    identStrength.FOVlims.rangeY = grid.pxPerPeriod+1:size(stress.xLinearGauge,1)-grid.pxPerPeriod-1;
    identStrength.FOVlims.rangeT = floor(smoothingOpts.WATemporalKernal(1)/2):...
        size(stress.xLinearGauge,3)-floor(smoothingOpts.WATemporalKernal(1)/2);
    
    % Get the maximum and minimum stress guage values and their locs in
    % space and time over the whole FOV
    temp = stress.xAvg(identStrength.FOVlims.rangeX,identStrength.FOVlims.rangeT);
    [identStrength.FOVlims.SGMax,ind] = max(temp(:));
    [identStrength.FOVlims.SGMaxLocX,identStrength.FOVlims.SGMaxLocT]...
        = ind2sub(size(temp),ind);
    [identStrength.FOVlims.SGMin,ind] = min(temp(:));
    [identStrength.FOVlims.SGMinLocX,identStrength.FOVlims.SGMinLocT]...
        = ind2sub(size(temp),ind);
    clear temp
    
    % Correct for the pixels/frames cut from the edges
    identStrength.FOVlims.SGMaxLocX = identStrength.FOVlims.SGMaxLocX+identStrength.FOVlims.rangeX(1)-1;
    identStrength.FOVlims.SGMaxLocT = identStrength.FOVlims.SGMaxLocT+identStrength.FOVlims.rangeT(1)-1;
    identStrength.FOVlims.SGMinLocX = identStrength.FOVlims.SGMinLocX+identStrength.FOVlims.rangeX(1)-1;
    identStrength.FOVlims.SGMinLocT = identStrength.FOVlims.SGMinLocT+identStrength.FOVlims.rangeT(1)-1;
    
    %--------------------------------------------------------------------------
    % Store vectors of the stress vs time at the fracture location
    identStrength.stressVsTAccel.xxAvgOverY = squeeze(stress.xAvg(fracture.locX,:));
    identStrength.stressVsTAccel.xxAvgOverGauge = stress.virtGaugeAvg.x; 
    identStrength.stressVsTStrain.xxAvgOverY = squeeze(stress.QRecon.x(:,fracture.locX,:));
    identStrength.stressVsTStrain.xxAvgOverGauge = stress.QRecon.virtGaugeAvg.x;
    
    %--------------------------------------------------------------------------
    % Write identified values to the console
    fprintf('\nIdentified strength values:\n')
    Strength_FracLoc = fracture
    Strength_StressGauge = identStrength.stressGauge
    Strength_LinearGauge = identStrength.linStressGauge

    fprintf('Strength Identification COMPLETE.\n')
    close all
else
    identStrength = nan;
    virtualGauge = nan;
    fracture = nan;
end

%% IMAGE SEQUENCES: for powerpoint presentations
if globalOpts.plotPresentationImages 
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        choice = questdlg('Plot images sequences for presentation? (takes a long time)', ...
            'Plot Image Sequence?', 'Yes','No','No');
        switch choice
            case 'Yes'
                plotPresentationImageSeq = true;
            case 'No'
                plotPresentationImageSeq= false;
        end
    elseif strcmp(globalOpts.plotImageSeqs,'yes') || strcmp(globalOpts.plotImageSeqs,'diagnostic')
        plotPresentationImageSeq = true;
    else
        plotPresentationImageSeq = false; 
    end

    if plotPresentationImageSeq == true
        fprintf('\n--------------------------------------------------------------\n')
        fprintf('IMAGE SEQUENCES: powerpoint presentations\n')
        fprintf('--------------------------------------------------------------\n')
        
        % Setup the plot parameters for the presentation figures
        savePath = imagePath;
        
        % Specify the frames over which the data is plotted
        if isnan(fracture)
            endFrame = time.numFrames;
        else
            endFrame = fracture.strengthFrame+5;
        end
        if endFrame > time.numFrames
            endFrame = time.numFrames;
        end
        plotParams.tRange = 1:endFrame;
        
        % Specify the location of the SS curve to be plotted
        if isnan(fracture)
            ssCurve.locX = round(length(pos.x)/2);
        else
            ssCurve.locX = fracture.locX;
        end
        % Plot the image sequences for presentation purposes
        fprintf('Plotting presentation image sequences.\n')
        func_plotPresentationVideos(globalOpts,plotParams,savePath,pos,time,...
            strain,strainRate,accel,stress,identStiffSG,ssCurve)
    end
end
%% ------------------------------------------------------------------------
% Save data
%--------------------------------------------------------------------------
fprintf('\n--------------------------------------------------------------\n')
fprintf('Saving processed data to a .mat file. \n')
fprintf('--------------------------------------------------------------\n')

% Save Path
savePath = imagePath; 

% Save file name
saveFile = 'AllProcessedData.mat';

% Clear switches to cancel cross-code interference
clear switches

if globalOpts.autoSaveProcData
    % Main conditional for saving all or a selection of workspace variables 
    if ~saveAllWorkspaceVariables
        % Medium file size
        if ~globalOpts.identifyStrength
            fprintf('Saving a selection of the workspace variables. \n')
            save([savePath,saveFile],'test','specimen','material','time','pos',...
            'disp','accel','strain','strainRate','stress',...
            'identStiffSG');
        else
            fprintf('Saving a selection of the workspace variables including the strength information. \n')
            save([savePath,saveFile],'test','specimen','material','time','pos',...
            'disp','accel','strain','strainRate','stress',...
            'identStiffSG','identStrength','fracture'); 
        end
    else
        % Large file size. Use to save all angled slice data for UD45 or MD45 samples.
        fprintf('Saving all workspace variables. \n')
        save([savePath,saveFile]);
    end
end


%% Print Identified Properties to Console
fprintf('\n--------------------------------------------------------------\n')
fprintf('IDENTIFIED MATERIAL PROPERTIES\n')
fprintf('--------------------------------------------------------------\n')

% Display the identified stiffnesses
fprintf('Identified stiffness values:\n')
IdentifiedStiffness_SG = identStiffSG

% Strain rate
if globalOpts.identifyStrength
     % Create condensed info on the strain rate with the strength stats included
    identStrainRate.xAvgMinPreFract = strainRate.xAvgMinPreFract;
    identStrainRate.xAvgMaxPreFract = strainRate.xAvgMaxPreFract;
else
    % Create condensed info on the strain rate
    identStrainRate.xAvgMin = strainRate.xAvgMin;
    identStrainRate.xAvgMax = strainRate.xAvgMax;
end
StrainRateMaxValues = identStrainRate;

% Strength
if globalOpts.identifyStrength
    fprintf('Identified strength values:\n')
    Strength_FracLoc = fracture
    Strength_StressGauge = identStrength.stressGauge
    Strength_LinearGauge = identStrength.linStressGauge
end

close all
fprintf('\nCOMPLETE.\n\n') 