function func_plotAllFullFieldVideos(globalOpts,plotParams,imagePath,pos,time,disp,...
    strain,strainRate,accel,material)
% Author: Lloyd Fletcher
% PhotoDyn Group, University of Southampton
% Date: 13/3/2017
%
% Plots 2D frames of data as an image sequence
%--------------------------------------------------------------------------
%
% Modifications
%
% Sam Parry 16/10/2019
% Used set(gca,'YDir','normal'); in func_plot_FullFieldImageSeq.m so that
% the origin 'appears' at the bottom left-hand corner of the sample. Then
% apply 'flipud' to the fields in this function in the plotVars{1} call, so
% that the fields are back to the correct orientation. 
% 
% Sam Parry. 03/06/2020
% Added switches to turn on and off various fields
% Added 'clear plotVars' after each plot to prevent plot error with counter.
%
% Sam Parry. 04/08/2020 
% Split the code into three sections:
% 1) GCSYS 
% 2) MCSYS 
% 3) DIAGNOSTIC
% Within each option, use the switches to turn particular field plots on or off.
%
%--------------------------------------------------------------------------
% Switches
switches.plotGCSYS = false;
switches.plotMCSYS = true;
switches.plotDIAGNOSTIC = false;

% For both GCSYS and MCSYS
switches.plotDisp = true;
switches.plotAccel = true;
switches.plotStrainAndStrainRate = true;
switches.plotStrainAndRawStrainALL = false;

%--------------------------------------------------------------------------
% Label strings for all image sequence axis
labelStrs.x = 'X [$mm$]';
labelStrs.y = 'Y [$mm$]';

%\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
% 1) GCSYS
%\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
if switches.plotGCSYS

%--------------------------------------------------------------------------
% Image Sequence - Disp - ALL - GCSYS

if switches.plotDisp

    % Save path
    imageSeqSavePath = [imagePath,'ImageSeq_DispALL_GCSYS\'];

    % Prompt and create the directory
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        plotImageSeq = func_checkIfPathExistsDlg(imageSeqSavePath,...
            'Disp - ALL - GCSYS image sequence folder found, plot again?','Plot Image Sequence?'); 
    elseif strcmp(globalOpts.plotImageSeqs,'yes') || strcmp(globalOpts.plotImageSeqs,'diagnostic')
        plotImageSeq = true;
        if exist(imageSeqSavePath,'file') ~= 7
            mkdir(imageSeqSavePath);
        end
    else
        plotImageSeq = false; 
    end

    if plotImageSeq
        % Create title strings and plot variables for the video
        labelStrs.t{1} = 'Displacement $u_{x}$, [$mm$]';
        labelStrs.t{2} = 'Displacement $u_{y}$, [$mm$]';

        % Remove the rigid body displacement
        [deformDisp.x,~] = func_calcDispRemoveRigidBody(disp.x);
        [deformDisp.y,~] = func_calcDispRemoveRigidBody(disp.y);
        
        % Assign the different variables to be plotted
        plotVars{1} = flipud(deformDisp.x)*10^3;
        plotVars{2} = flipud(deformDisp.y)*10^3;

        % Specify the parameters of the plot
        plotParams.Rows = 1;
        plotParams.Cols = 2;
        plotParams.titleFrameNum = true;
        plotParams.cRange{1} = plotParams.cAxisDisp;
        plotParams.cRange{2} = plotParams.cAxisDisp;

        % Plot and save the image sequence to file
        func_plotFullFieldImageSeq(imageSeqSavePath,plotParams,labelStrs,pos,time,plotVars)
    end
    % Clear the plot variables for the next plot
    clear plotVars
end
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Image Sequence - Accel - ALL - GCSYS

if switches.plotAccel

    % Cut the edge pixels or not
    plotParams.cutEdgePx = false;

    % Create the save path for the heat maps
    imageSeqSavePath = [imagePath,'ImageSeq_AccelALL_GCSYS\'];

    % Prompt and create the directory
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        plotImageSeq = func_checkIfPathExistsDlg(imageSeqSavePath,...
            'Accel - ALL - GCSYS image sequence folder found, plot again?','Plot Image Sequence?');   
    elseif strcmp(globalOpts.plotImageSeqs,'yes')
        plotImageSeq = true;
        if exist(imageSeqSavePath,'file') ~= 7
            mkdir(imageSeqSavePath);
        end
    else
        plotImageSeq = false; 
    end

    if plotImageSeq   
        % Create title strings and plot variables for the video
        labelStrs.t{1} = 'Acceleration $a_{x}$, [$m.s^{-2}$]';
        labelStrs.t{2} = 'Acceleration $a_{y}$, [$m.s^{-2}$]';

        % Assign the different variables to be plotted
        plotVars{1} = flipud(accel.x);
        plotVars{2} = flipud(accel.y);

        % Specify the parameters of the plot
        plotParams.Rows = 1;
        plotParams.Cols = 2;
        plotParams.titleFrameNum = true;
        plotParams.cRange{1} = plotParams.cAxisAccel;
        plotParams.cRange{2} = plotParams.cAxisAccel;

        % Plot and save the image sequence to file
        func_plotFullFieldImageSeq(imageSeqSavePath,plotParams,labelStrs,pos,time,plotVars)
    end
    % Clear the plot variables for the next plot
    clear plotVars
end
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Image Sequence - Strain and Strain Rate - ALL - GCSYS

if switches.plotStrainAndStrainRate

    % Cut edge
    plotParams.cutEdgePx = true;

    % Create the save path for the heat maps
    imageSeqSavePath = [imagePath,'ImageSeq_StrainStrainRateALL_GCSYS\'];

    % Create folder if prompted
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        plotImageSeq = func_checkIfPathExistsDlg(imageSeqSavePath,...
            'Strain and Strain Rate - ALL - GCSYS image sequence folder found, plot again?','Plot Image Sequence?');   
    elseif strcmp(globalOpts.plotImageSeqs,'yes')
        plotImageSeq = true;
        if exist(imageSeqSavePath,'file') ~= 7
            mkdir(imageSeqSavePath);
        end
    else
        plotImageSeq = false; 
    end

    if plotImageSeq   
        % Create title strings and plot variables for the video
        labelStrs.t{1} = 'Strain $\epsilon_{xx}$, [$mm.m^{-1}$]';
        labelStrs.t{2} = 'Strain $\epsilon_{yy}$, [$mm.m^{-1}$]';
        labelStrs.t{3} = 'Strain $\epsilon_{xy}$, [$mm.m^{-1}$]';
        labelStrs.t{4} = 'Strain Rate $d\epsilon_{xx}/dt$, [$s^{-1}$]';
        labelStrs.t{5} = 'Strain Rate $d\epsilon_{yy}/dt$, [$s^{-1}$]';
        labelStrs.t{6} = 'Strain Rate $d\epsilon_{xy}/dt$, [$s^{-1}$]';
        
        % Assign the different variables to be plotted
        plotVars{1} = flipud(strain.x)*10^3;
        plotVars{2} = flipud(strain.y)*10^3;
        plotVars{3} = flipud(strain.s)*10^3;
        plotVars{4} = flipud(strainRate.x);
        plotVars{5} = flipud(strainRate.y);
        plotVars{6} = flipud(strainRate.s);
        
        % Specify the parameters of the plot
        plotParams.Rows = 2;
        plotParams.Cols = 3;
        plotParams.titleFrameNum = true;
        plotParams.cRange{1} = plotParams.cAxisStrain;
        plotParams.cRange{2} = plotParams.cAxisStrain;
        plotParams.cRange{3} = plotParams.cAxisStrain;
        plotParams.cRange{4} = plotParams.cAxisStrainRate;
        plotParams.cRange{5} = plotParams.cAxisStrainRate;
        plotParams.cRange{6} = plotParams.cAxisStrainRate;
        
        % Plot and save the image sequence to file
        func_plotFullFieldImageSeq(imageSeqSavePath,plotParams,labelStrs,pos,time,plotVars)
    end
    % Clear the plot variables for the next plot
    clear plotVars
end
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Image Sequence - Strain and Raw Strain - ALL - GCSYS

if switches.plotStrainAndRawStrainALL

    % Cut edge 
    plotParams.cutEdgePx = false;

    % Create the save path for the heat maps
    imageSeqSavePath = [imagePath,'ImageSeq_StrainRawALL_GCSYS\'];

    % Create folder if needed
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        plotImageSeq = func_checkIfPathExistsDlg(imageSeqSavePath,...
            'Strain and Raw Strain - ALL - GCSYS image sequence folder found, plot again?','Plot Image Sequence?');
    elseif strcmp(globalOpts.plotImageSeqs,'yes')
        plotImageSeq = true;
        if exist(imageSeqSavePath,'file') ~= 7
            mkdir(imageSeqSavePath);
        end
    else
        plotImageSeq = false; 
    end

    if plotImageSeq

        % Calculate the RAW strains from the displacements 
        rawStrain = func_calcStrainFromDisp(disp,pos.xStep,pos.yStep);

        % Create title strings and plot variables for the video
        labelStrs.t{1} = 'Strain $\epsilon_{xx}$, [$mm.m^{-1}$]';
        labelStrs.t{2} = 'Strain $\epsilon_{yy}$, [$mm.m^{-1}$]';
        labelStrs.t{3} = 'Strain $\epsilon_{xy}$, [$mm.m^{-1}$]';
        labelStrs.t{4} = 'Raw Strain $\epsilon_{xx}$, [$mm.m^{-1}$]';
        labelStrs.t{5} = 'Raw Strain $\epsilon_{yy}$, [$mm.m^{-1}$]';
        labelStrs.t{6} = 'Raw Strain $\epsilon_{xy}$, [$mm.m^{-1}$]';

        % Assign the different variables to be plotted
        plotVars{1} = flipud(strain.x)*10^3;
        plotVars{2} = flipud(strain.y)*10^3;
        plotVars{3} = flipud(strain.s)*10^3;
        plotVars{4} = flipud(rawStrain.x)*10^3;
        plotVars{5} = flipud(rawStrain.y)*10^3;
        plotVars{6} = flipud(rawStrain.s)*10^3;

        % Specify the parameters of the plot
        plotParams.Rows = 2;
        plotParams.Cols = 3;
        plotParams.titleFrameNum = true;
        plotParams.cRange{1} = plotParams.cAxisStrain;
        plotParams.cRange{2} = plotParams.cAxisStrain;
        plotParams.cRange{3} = plotParams.cAxisStrain;
        plotParams.cRange{4} = plotParams.cAxisRawStrain;
        plotParams.cRange{5} = plotParams.cAxisRawStrain;
        plotParams.cRange{6} = plotParams.cAxisRawStrain;

        % Plot and save the image sequence to file
        func_plotFullFieldImageSeq(imageSeqSavePath,plotParams,labelStrs,pos,time,plotVars)
    end
    % Clear the plot variables for the next plot
    clear plotVars
end
%--------------------------------------------------------------------------

end
%% \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
% 2) MCSYS
%\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
if switches.plotMCSYS

%--------------------------------------------------------------------------
% Image Sequence - Disp - ALL - MCSYS

if switches.plotDisp

    % Save path
    imageSeqSavePath = [imagePath,'ImageSeq_DispALL_MCSYS\'];

    % Prompt and create the directory
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        plotImageSeq = func_checkIfPathExistsDlg(imageSeqSavePath,...
            'Disp - ALL - MCSYS image sequence folder found, plot again?','Plot Image Sequence?'); 
    elseif strcmp(globalOpts.plotImageSeqs,'yes') || strcmp(globalOpts.plotImageSeqs,'diagnostic')
        plotImageSeq = true;
        if exist(imageSeqSavePath,'file') ~= 7
            mkdir(imageSeqSavePath);
        end
    else
        plotImageSeq = false; 
    end

    if plotImageSeq
           
        % Create title strings and plot variables for the video
        labelStrs.t{1} = 'Displacement $u_{1}$, [$mm$]';
        labelStrs.t{2} = 'Displacement $u_{2}$, [$mm$]';

        % Remove the rigid body displacement
        [deformDisp.mat11,~] = func_calcDispRemoveRigidBody(disp.mat11);
        [deformDisp.mat22,~] = func_calcDispRemoveRigidBody(disp.mat22);
        
        % Assign the different variables to be plotted
        plotVars{1} = flipud(deformDisp.mat11)*10^3;
        plotVars{2} = flipud(deformDisp.mat22)*10^3;

        % Specify the parameters of the plot
        plotParams.Rows = 1;
        plotParams.Cols = 2;
        plotParams.titleFrameNum = true;
        plotParams.cRange{1} = plotParams.cAxisDisp;
        plotParams.cRange{2} = plotParams.cAxisDisp;

        % Plot and save the image sequence to file
        func_plotFullFieldImageSeq(imageSeqSavePath,plotParams,labelStrs,pos,time,plotVars)

    end
    % Clear the plot variables for the next plot
    clear plotVars
end
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Image Sequence - Accel - ALL - MCSYS

if switches.plotAccel

    % Cut the edge pixels or not
    plotParams.cutEdgePx = false;

    % Create the save path for the heat maps
    imageSeqSavePath = [imagePath,'ImageSeq_AccelALL_MCSYS\'];

    % Prompt and create the directory
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        plotImageSeq = func_checkIfPathExistsDlg(imageSeqSavePath,...
            'Accel - ALL - MCSYS image sequence folder found, plot again?','Plot Image Sequence?');   
    elseif strcmp(globalOpts.plotImageSeqs,'yes')
        plotImageSeq = true;
        if exist(imageSeqSavePath,'file') ~= 7
            mkdir(imageSeqSavePath);
        end
    else
        plotImageSeq = false; 
    end

    if plotImageSeq   
        % Create title strings and plot variables for the video
        labelStrs.t{1} = 'Acceleration $a_{1}$, [$m.s^{-2}$]';
        labelStrs.t{2} = 'Acceleration $a_{2}$, [$m.s^{-2}$]';

        % Assign the different variables to be plotted
        plotVars{1} = flipud(accel.mat11);
        plotVars{2} = flipud(accel.mat22);

        % Specify the parameters of the plot
        plotParams.Rows = 1;
        plotParams.Cols = 2;
        plotParams.titleFrameNum = true;
        plotParams.cRange{1} = plotParams.cAxisAccel;
        plotParams.cRange{2} = plotParams.cAxisAccel;

        % Plot and save the image sequence to file
        func_plotFullFieldImageSeq(imageSeqSavePath,plotParams,labelStrs,pos,time,plotVars)
    end
    % Clear the plot variables for the next plot
    clear plotVars
end
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Image Sequence - Strain and Strain Rate - ALL - MCSYS

if switches.plotStrainAndStrainRate

    % Cut edge
    plotParams.cutEdgePx = true;

    % Save path
    imageSeqSavePath = [imagePath,'ImageSeq_StrainStrainRateALL_MCSYS\'];

    % Create folder if prompted
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        plotImageSeq = func_checkIfPathExistsDlg(imageSeqSavePath,...
            'Strain and Strain Rate - ALL - MCSYS image sequence folder found, plot again?','Plot Image Sequence?');   
    elseif strcmp(globalOpts.plotImageSeqs,'yes')
        plotImageSeq = true;
        if exist(imageSeqSavePath,'file') ~= 7
            mkdir(imageSeqSavePath);
        end
    else
        plotImageSeq = false; 
    end

    if plotImageSeq   
        % Create title strings and plot variables for the video
        labelStrs.t{1} = 'Strain $\epsilon_{11}$, [$mm.m^{-1}$]';
        labelStrs.t{2} = 'Strain $\epsilon_{22}$, [$mm.m^{-1}$]';
        labelStrs.t{3} = 'Strain $\gamma_{12}$, [$mm.m^{-1}$]';
        labelStrs.t{4} = 'Strain Rate $d\epsilon_{11}/dt$, [$s^{-1}$]';
        labelStrs.t{5} = 'Strain Rate $d\epsilon_{22}/dt$, [$s^{-1}$]';
        labelStrs.t{6} = 'Strain Rate $d\gamma_{12}/dt$, [$s^{-1}$]';
        
        % Assign the different variables to be plotted
        plotVars{1} = flipud(strain.mat11)*10^3;
        plotVars{2} = flipud(strain.mat22)*10^3;
        plotVars{3} = flipud(strain.mat12)*10^3;
        plotVars{4} = flipud(strainRate.mat11);
        plotVars{5} = flipud(strainRate.mat22);
        plotVars{6} = flipud(strainRate.mat12);
        
        % Specify the parameters of the plot
        plotParams.Rows = 2;
        plotParams.Cols = 3;
        plotParams.titleFrameNum = true;
        plotParams.cRange{1} = plotParams.cAxisStrain;
        plotParams.cRange{2} = plotParams.cAxisStrain;
        plotParams.cRange{3} = plotParams.cAxisStrain;
        plotParams.cRange{4} = plotParams.cAxisStrainRate;
        plotParams.cRange{5} = plotParams.cAxisStrainRate;
        plotParams.cRange{6} = plotParams.cAxisStrainRate;
        % Plot and save the image sequence to file
        func_plotFullFieldImageSeq(imageSeqSavePath,plotParams,labelStrs,pos,time,plotVars)
    end
    % Clear the plot variables for the next plot
    clear plotVars
end
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Image Sequence - Strain and Raw Strain - ALL - MCSYS

if switches.plotStrainAndRawStrainALL

    % Cut edge
    plotParams.cutEdgePx = false;

    % Save path
    imageSeqSavePath = [imagePath,'ImageSeq_StrainRawALL_MCSYS\'];

    % Create folder if prompted
    if strcmp(globalOpts.plotImageSeqs,'prompt')
        plotImageSeq = func_checkIfPathExistsDlg(imageSeqSavePath,...
            'Strain and Raw Strain - ALL - MCSYS image sequence folder found, plot again?','Plot Image Sequence?');
    elseif strcmp(globalOpts.plotImageSeqs,'yes')
        plotImageSeq = true;
        if exist(imageSeqSavePath,'file') ~= 7
            mkdir(imageSeqSavePath);
        end
    else
        plotImageSeq = false; 
    end

    if plotImageSeq

        % Calculate the RAW strains from the displacements 
        rawStrain = func_calcStrainFromDisp(disp,pos.xStep,pos.yStep);

        % Rotate the RAW strains into material coordinates
        [rawStrain.mat11,rawStrain.mat22,rawStrain.mat12] = func_rotateStrain2D(...
            rawStrain.x,rawStrain.y,rawStrain.s,material.rotAngle);

        % Create title strings and plot variables for the video
        labelStrs.t{1} = 'Strain $\epsilon_{11}$, [$mm.m^{-1}$]';
        labelStrs.t{2} = 'Strain $\epsilon_{22}$, [$mm.m^{-1}$]';
        labelStrs.t{3} = 'Strain $\gamma_{12}$, [$mm.m^{-1}$]';
        labelStrs.t{4} = 'Raw Strain $\epsilon_{11}$, [$mm.m^{-1}$]';
        labelStrs.t{5} = 'Raw Strain $\epsilon_{22}$, [$mm.m^{-1}$]';
        labelStrs.t{6} = 'Raw Strain $\gamma_{12}$, [$mm.m^{-1}$]';
        
        % Assign the different variables to be plotted
        plotVars{1} = flipud(strain.mat11)*10^3;
        plotVars{2} = flipud(strain.mat22)*10^3;
        plotVars{3} = flipud(strain.mat12)*10^3;
        plotVars{4} = flipud(rawStrain.mat11)*10^3;
        plotVars{5} = flipud(rawStrain.mat22)*10^3;
        plotVars{6} = flipud(rawStrain.mat12)*10^3;
        
        % Specify the parameters of the plot
        plotParams.Rows = 2;
        plotParams.Cols = 3;
        plotParams.titleFrameNum = true;
        plotParams.cRange{1} = plotParams.cAxisStrain;
        plotParams.cRange{2} = plotParams.cAxisStrain;
        plotParams.cRange{3} = plotParams.cAxisStrain;
        plotParams.cRange{4} = plotParams.cAxisRawStrain;
        plotParams.cRange{5} = plotParams.cAxisRawStrain;
        plotParams.cRange{6} = plotParams.cAxisRawStrain;
        % Plot and save the image sequence to file
        func_plotFullFieldImageSeq(imageSeqSavePath,plotParams,labelStrs,pos,time,plotVars)
    end
    % Clear the plot variables for the next plot
    clear plotVars
end
%--------------------------------------------------------------------------

end
%% \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
% 3) DIAGNOSTIC
%\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
if switches.plotDIAGNOSTIC

%--------------------------------------------------------------------------
% Image Sequence - Disp, Accel, Strain and Strain Rate X - GCSYS

    
% Save Path
imageSeqSavePath = [imagePath,'ImageSeq_ALLX_GCSYS_DIAG\'];

% Prompt and create the directory
if strcmp(globalOpts.plotImageSeqs,'prompt')
    plotImageSeq = func_checkIfPathExistsDlg(imageSeqSavePath,...
        'X field image sequence folder found, plot again?','Plot Image Sequence?'); 
elseif strcmp(globalOpts.plotImageSeqs,'yes') || strcmp(globalOpts.plotImageSeqs,'diagnostic')
    plotImageSeq = true;
    if exist(imageSeqSavePath,'file') ~= 7
        mkdir(imageSeqSavePath);
    end
else
    plotImageSeq = false; 
end

if plotImageSeq
    % Create title strings and plot variables for the video
    labelStrs.t{1} = 'Displacement $\delta_{x}$, [$mm$]';
    labelStrs.t{2} = 'Accel $a_{x}$, [$m.s^{-2}$]';
    labelStrs.t{3} = 'Strain $\epsilon_{xx}$, [$mm.m^{-1}$]';
    labelStrs.t{4} = 'Strain Rate $d\epsilon_{xx}/dt$, [$s^{-1}$]';
    
    % Assign the different variables to be plotted
    [deformDisp.x,~] = func_calcDispRemoveRigidBody(disp.x);
    plotVars{1} = flipud(deformDisp.x)*10^3;
    plotVars{2} = flipud(accel.x);
    plotVars{3} = flipud(strain.x*10^3);
    plotVars{4} = flipud(strainRate.x);
    
    % Specify the parameters of the plot
    plotParams.Rows = 2;
    plotParams.Cols = 2;
    plotParams.titleFrameNum = true;
    plotParams.cRange{1} = plotParams.cAxisDisp;
    plotParams.cRange{2} = plotParams.cAxisAccel;
    plotParams.cRange{3} = plotParams.cAxisStrain;
    plotParams.cRange{4} = plotParams.cAxisStrainRate;
    % Plot and save the image sequence to file
    func_plotFullFieldImageSeq(imageSeqSavePath,plotParams,labelStrs,pos,time,plotVars)
end
% Clear the plot variables for the next plot
clear plotVars

%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Image Sequence of Disp and Accel - ALL COMPONENTS - GCSYS

% Cut edge or not
plotParams.cutEdgePx = false;

% Save path
imageSeqSavePath = [imagePath,'ImageSeq_DispAccelALL_GCSYS_DIAG\'];

% Prompt and create the directory
if strcmp(globalOpts.plotImageSeqs,'prompt')
    plotImageSeq = func_checkIfPathExistsDlg(imageSeqSavePath,...
        'All components of disp/accel image sequence folder found, plot again?','Plot Image Sequence?'); 
elseif strcmp(globalOpts.plotImageSeqs,'yes')
    plotImageSeq = true;
    if exist(imageSeqSavePath,'file') ~= 7
        mkdir(imageSeqSavePath);
    end
else
    plotImageSeq = false; 
end

if plotImageSeq
    
% Remove rigid body displacement
%[rawStrain.x,~,~] = gradient(disp.x,pos.xStep,pos.yStep,time.step);    %delete?
[deformDisp.x,~] = func_calcDispRemoveRigidBody(disp.x);
[deformDisp.y,~] = func_calcDispRemoveRigidBody(disp.y);

% Create title strings and plot variables for the video
labelStrs.t{1} = 'Displacement $\delta_{x}$, [$mm$]';
labelStrs.t{2} = 'Displacement $\delta_{y}$, [$mm$]';
labelStrs.t{3} = 'Acceleration $a_{x}$, [$m.s^{-2}$]';
labelStrs.t{4} = 'Acceleration $a_{y}$, [$m.s^{-2}$]';

% Assign the different variables to be plotted
plotVars{1} = flipud(deformDisp.x)*10^3;
plotVars{2} = flipud(deformDisp.y)*10^3;
plotVars{3} = flipud(accel.x);
plotVars{4} = flipud(accel.y);

% Specify the parameters of the plot
plotParams.Rows = 2;
plotParams.Cols = 2;
plotParams.titleFrameNum = true;

% Plot ranges
plotParams.cRange{1} = plotParams.cAxisDisp;
plotParams.cRange{2} = plotParams.cAxisDisp;
plotParams.cRange{3} = plotParams.cAxisAccel;
plotParams.cRange{4} = plotParams.cAxisAccel;

% Plot and save the image sequence to file
func_plotFullFieldImageSeq(imageSeqSavePath,plotParams,labelStrs,pos,time,plotVars)
end
% Clear the plot variables for the next plot
clear plotVars
%--------------------------------------------------------------------------

end
%\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
end
