%% IMAGE BASED INERTIAL IMPACT (IBII) TEST - Data Processing - v3.0ir
% Authors: Lloyd Fletcher, Jared Van-Blitterswyk
% PhotoDyn Group, University of Southampton
% http://photodyn.org/
% Date Created: 12th Dec. 2017 - v1.0ir
% Date Edited: 11th Mar. 2019 - v3.0ir
%
%--------------------------------------------------------------------------
% Modifications and name change
% Sam Parry - 21/04/2020
%
% This code has three options: 
%
% 1) Import FE displacement fields and process with the VFM
%    switches.FEValidMode = 1;
% 
% 2) Import FE displacement fields and interpolate them to the geometry of a specific test sample.
%    switches.FEValidMode = 0;
%    This can be used in a Q12 correction of E22 in an IBII test, as the experimental e11 strains can be noisy.
%
% 3) Import DIC displacement fields and process with the VFM
%    switches.processDICData = 'on';
%
% For each of the three options above, kinematic fields, stress-strain curves
% and modulus vs. position plots can be made for UD90 and off-axis cases.
%
%--------------------------------------------------------------------------

clc
clear all
close all

%--------------------------------------------------------------------------
% Switches
%--------------------------------------------------------------------------

switches.FEValidMode = 0; % Set to true to view the FE fields only. Set to false to calculate avg11 interpolated to each sample, for e11 correction of E22.
switches.ChangeFEModelMCSrotAngleToCWPositive = 'on'; % Conversion between +CW (Abaqus) and +CCW (This code, due to pos.y being downward positive)
switches.processDICData = 'off'; % Turn 'on' to import DIC displacement fields after processing with MatchID

% Plot options
switches.plotImageSeqs = 'no'; % Code specific option to turn-off plotting fields

% Save options
switches.saveStiffnessData = 'off'; % Turn 'on' to save a stiffness data .mat file for multi-stiffness vs. position plots.
switches.saveAllData = 'off'; % Save all the data (but, gives large file sizes)
switches.saveAvg11_FE = 'on'; % Save a file containing the average fibre strains (using the specific sample geometry) for later e11 correction of E22.

%--------------------------------------------------------------------------
% Functions path
%--------------------------------------------------------------------------

% Add the path for the Grid Method code and other useful functions:
funcPath = 'E:\IBIIProcessingCode_v3.0ir\ProcessingFunctions_2019\';

% If the default path is not found we should find it
if exist(funcPath,'file') ~= 7
    hWarn = warndlg('Folder for processing functions not found.','Function folder not found');
    waitfor(hWarn);
    fprintf('Adding paths for processing functions.\n')
    funcPath = uigetdir(pwd,'Locate Processing Function Folder');
end

addpath(funcPath);
addpath([funcPath,'GridMethodToolbox\']);
addpath([funcPath,'Export_Fig\']);
addpath([funcPath,'linspecer\']);

%% ------------------------------------------------------------------------
% Create a folder for the processed FE data
%--------------------------------------------------------------------------
hardCodePath = false; % Careful not to mix UD90 and UD45 data

% Select the UD90 or UD45 folder
if ~hardCodePath
    basePath = uigetdir(pwd,'Select the folder where you want to save the processed FE fields.');
else
    %basePath = 'E:\experimentalData\ModulusPaper\FE_UD90';
    basePath = 'E:\experimentalData\ModulusPaper\FE_UD45';
end

% If it does not exist, create a directory for the processed FE data
fprintf('If it does not exist, create a directory for the processed FE data. \n')
dataPath = ([basePath,'\']);
    if ~exist(dataPath,'dir')
        mkdir(dataPath);
    end
addpath(dataPath);

%% ------------------------------------------------------------------------
% FE init file 
%--------------------------------------------------------------------------

% Point to the init file path or use hardcoded path
if ~hardCodePath
    [initFEFile,initFEPath] = uigetfile({'*.*','All Files'},'Select the FE simulation processing parameters file');
else
    initFEPath = 'E:\experimentalData\ModulusPaper\FE_UD90\AbaqusToMatlab\'; 
    %initFEPath = 'E:\experimentalData\ModulusPaper\FE_UD45\AbaqusToMatlab\'; 
    initFEFile = 'FESimProcessingParameters_v1.mat';   
end

% Load the processing parameters from file
fprintf('Loading FE initialisation file.\n')
load([initFEPath,initFEFile]);

%% ------------------------------------------------------------------------
% FE displacement field file path 
%--------------------------------------------------------------------------

% Load the FE displacement file from the common directory
if ~hardCodePath
    [FEDataFile,FEDataPath] = uigetfile({'*.*','All Files'},'Select the FE simulation displacement data file');
else
    FEDataFile = 'FESimKinematicFields_2D_CFRPPulse_UD90.mat';
    %FEDataFile = 'FESimKinematicFields_2D_CFRPPulse_UD45.mat';
    FEDataPath = 'E:\experimentalData\ModulusPaper\FE_UD90\AbaqusToMatlab\';     
    %FEDataPath = 'E:\experimentalData\ModulusPaper\FE_UD45\AbaqusToMatlab\';     
end

% Load the FE disp fields into the Workspace
fprintf('Loading FE data from selected .mat file.\n')
load([FEDataPath,FEDataFile]);

% % Flip the fields because pos.y is upside down                           %%%%%%%%%%%%%
% dispFE.xFE = flipud(dispFE.xFE);
% dispFE.yFE = flipud(dispFE.yFE);

%% ------------------------------------------------------------------------
% Load the experimental processing parameters file
%--------------------------------------------------------------------------
% Required for the fitting and plotting options

% Load the experimental test initialisation file
if ~hardCodePath
    [initFile,initPath] = uigetfile({'*.*','All Files'},'Select the experimental processing parameters file');
else
    initFile = 'processingParameters_v3_0ir.mat';
    initPath = 'E:\experimentalData\ModulusPaper\UD45-S1\';
end

% Load the experimental processing parameters file
fprintf('Loading experimental processing parameters file.\n')
load([initPath,initFile]);

% Turn smoothing off for this case
globalOpts.smoothingOn = false;   

%% ------------------------------------------------------------------------
% Grid Method
%--------------------------------------------------------------------------

if ~switches.FEValidMode
    
    % Process the raw tiff images using the grid method code developed by Grediac et al.
    fprintf('\n--------------------------------------------------------------\n')
    fprintf('GRID METHOD PROCESSING\n')
    fprintf('--------------------------------------------------------------\n')

    %--------------------------------------------------------------------------
    % GRID IMAGE PROCESSING

    % Check if the images have already been processed otherwise process the data
    gridDataSavePath = initPath;
    gridDataFile = 'GridMethod_ProcessedData.mat';

    fprintf('Checking for existing processed data file.\n')
    processSliceData = true;
    if exist([gridDataSavePath,gridDataFile],'file') == 2
        fprintf('Grid method data file found.\n')
        if gridMethodOpts.autoLoadProccessedDataFile
            processSliceData = false;
        else   
            choice = questdlg('Processed grid data file found, process again?', ...
            'Process Grid Images', 'Yes','No','No');
            switch choice
                case 'Yes'
                    fprintf('.\n')
                    processSliceData = true;
                case 'No'
                    processSliceData = false;
            end
        end
    end

    if processSliceData
        fprintf('Processing images using the grid method toolbox.\n')
        % Process the image squence with the grid method toolbox
        [grid,pos,disp] = func_gridMethodImageProcessing(imagePath,imageFile,...
            grid,gridMethodOpts,imageNoise);

        % Save the data to file so a reprocess is not necessary
        fprintf('Saving processed grid data to: %s.\n',gridDataFile)
        save([gridDataSavePath,gridDataFile],'grid','pos','disp')

        fprintf('Grid Method Processing Complete.\n')
    else
        fprintf('Loading processed grid data from: %s.\n',gridDataFile)
            load([gridDataSavePath,gridDataFile]);
    end

    %--------------------------------------------------------------------------
    % Update Geometry and Number of Frames Based on Displacement Matrix Size
    fprintf('Updating geometry variables based on the size of displacement field matrices.\n')
    [specimen,grid] = func_updateSpecGeom(specimen,grid,disp);

    % Currently the rotations are unused so remove them to save RAM
    disp = rmfield(disp,'rot');
    
else
    fprintf('No Grid Method data loaded \n')
end


%% ------------------------------------------------------------------------
% Structure variable name assignment
%--------------------------------------------------------------------------

if ~switches.FEValidMode
    
    % Interpolate FE data to experimental grid (Requires pos struct from specific sample)
    fprintf('Interpolating FE disp to experimental geometry.\n')
    
    for tt = 1:time.numFrames
        
        % Grid the FE displacement vectors
        [FE.pos.xGrid, FE.pos.yGrid] = ndgrid(FE.pos.y, FE.pos.x);
        
        % F = griddedInterpolant(X1,X2,V) X1 and X2 define the grids to sample the field value V. Use ndgrid not meshgrid             
        Fx = griddedInterpolant(FE.pos.xGrid, FE.pos.yGrid, dispFE.xFE(:,:,tt));
        Fy = griddedInterpolant(FE.pos.xGrid, FE.pos.yGrid, dispFE.yFE(:,:,tt));

        % Create nd grids from the pos struct
        [pos.xNDGrid, pos.yNDGrid] = ndgrid(pos.y, pos.x);
        
        % Interpolate FE disp fields to the image grid [vq = F(xq,yq)];
        disp.x(:,:,tt) = Fx(pos.xNDGrid, pos.yNDGrid);
        disp.y(:,:,tt) = Fy(pos.xNDGrid, pos.yNDGrid);

        % Clear interpolants each time step
        clear 'Fx' 'Fy'

    end
 
elseif strcmp('on',switches.processDICData)
    
    % If we are processing DIC fields, define pos struct here.
    fprintf('Defining pos struct for processing DIC data.\n')
    
    rowPx = size(disp.x,1);
    colPx = size(disp.x,2);

    pos.xStep = grid.mPerPx; 
    pos.yStep = grid.mPerPx;
    pos.x = pos.xStep/2 : pos.xStep : colPx*grid.mPerPx;
    pos.y = pos.yStep/2 : pos.yStep : rowPx*grid.mPerPx;
    [pos.xGrid,pos.yGrid] = meshgrid(pos.x,pos.y);

else
    
    % If not interpolating to an experimental specimen dimension (i.e. processing FE data only), 
    % set the below structs to that defined in the FE init file.
    fprintf('Use the FE data to define the disp, pos, specimen and grid structs.\n')
    
    % Disp
    disp.x = dispFE.xFE;
    disp.y = dispFE.yFE;
    
    % Pos
    pos.x = FE.pos.x;
    pos.y = FE.pos.y;
    pos.xStep = FE.pos.xStep;
    pos.yStep = FE.pos.yStep;
    [pos.xGrid,pos.yGrid] = meshgrid(pos.x,pos.y); 
    
    % Specimen
    specimen.length = FE.specLength;
    specimen.height = FE.specHeight;
    specimen.volume = specimen.length*specimen.height*specimen.thickness;
    specimen.mass = specimen.volume*material.rho;
    
    % Grid
    grid.length = specimen.length;
    grid.height= specimen.height;
    
end

%% POST-PROCESSING: Smoothing and Kinematic Field Derivation
% Smooth the displacement data and then calculate acceleration and strains
% Extrapolate the data to account for the missing pitch on the edges

fprintf('\n--------------------------------------------------------------\n')
fprintf('POST PROCESSING: Smoothing and Kinematic Field Calculation\n')
fprintf('--------------------------------------------------------------\n')
   
%--------------------------------------------------------------------------
% Smooth and Calculate Strain
fprintf('Calculating strain from the displacement fields.\n')
[disp,strain,strainRate] = func_smoothCalcStrain(globalOpts,pos,time,grid,disp,smoothingOpts,extrapOpts);

%--------------------------------------------------------------------------
% Smooth and Calculate Acceleration
fprintf('Calculating acceleration from the displacement fields.\n')
[disp,vel,accel] = func_smoothCalcAccel(pos,time,grid,disp,smoothingOpts,extrapOpts,diffOpts);

%--------------------------------------------------------------------------
% Set up the slice structs
if strcmp('orthotropicAngle',globalOpts.matModel)
    % Create both angled slice structs for tranverse and parallel fibre cuts.
    angledSlice1 = func_initAngledSliceStruct(specimen,pos,angledSlice1);
    angledSlice2 = func_initAngledSliceStruct(specimen,pos,angledSlice2);  
end

%% ------------------------------------------------------------------------
% FE simulation data angle convention switch
% FE simulation assumes +CCW, with the y-axis upwards positive. 
% This matlab code assumes CW+, because the pos.y struct is downward positive.

if  strcmp('orthotropicAngle',globalOpts.matModel) && strcmp('on', switches.ChangeFEModelMCSrotAngleToCWPositive)
    material.rotAngle = 45;
    angledSlice1.angle = 45;
    angledSlice2.angle = 135;
end

%% --------------------------------------------------------------------------
% Rotate Kinematic Fields to Material Co-ords
% NOTE: only required for orthotropicAngle material model
if strcmp('orthotropicAngle',globalOpts.matModel)
    fprintf('Rotating kinematic fields into the material co-ords, angle = %0.2f.\n',material.rotAngle)
    
    % Note disp.mat11 is rotated relative to the x-axis.
    fprintf('Rotating the displacement fields.\n')
    [disp.mat11,disp.mat22] = func_rotateVector2D(disp.x,disp.y,material.rotAngle);
    
    fprintf('Rotating the acceleration fields.\n')
    [accel.mat11,accel.mat22] = func_rotateVector2D(accel.x,accel.y,material.rotAngle);
    
    fprintf('Rotating strain fields to the material co-ord system.\n')
    [strain.mat11,strain.mat22,strain.mat12] = func_rotateStrain2D(...
        strain.x,strain.y,strain.s,material.rotAngle);
    
    fprintf('Rotating strain rate fields to the material co-ord system.\n')
    [strainRate.mat11,strainRate.mat22,strainRate.mat12] = func_rotateStrain2D(...
        strainRate.x,strainRate.y,strainRate.s,material.rotAngle);
end

%% --------------------------------------------------------------------------
% Create material coordinate fields for the reduced orthotropic case.
% Note: Cannot use rotation equations, as they give a sign error on the wavefront

if strcmp('orthotropicReduced',globalOpts.matModel)
    fprintf('Creating the UD90 case material coordinate fields.\n')
    disp.mat11 = -disp.y; % pos.y is upside down
    disp.mat22 = disp.x;
    accel.mat11 = -accel.y; % calculated from disp.y
    accel.mat22 = accel.x;
    strain.mat11 = -strain.y; % calculated from disp.y
    strain.mat22 = strain.x;
    strain.mat12 = -strain.s; % calculated from disp.y
    strainRate.mat11 = -strainRate.y;
    strainRate.mat22 = strainRate.x;
    strainRate.mat12 = -strainRate.s; % calculated from disp.y
end


%% --------------------------------------------------------------------------
% STANDARD STRESS GAUGE: CALCULATION

fprintf('Calculating average stress with the stress-gauge equation.\n')

% Standard Stress Gauge equation calculation - xAvg
[stress.xAvg,~] = func_stressGaugeProcess(material, time, pos, accel.x, accel.xAvg);

% Standard Stress Gauge equation calculation - xyAvg
[stress.sAvg,~] = func_stressGaugeProcess(material, time, pos, accel.y, accel.yAvg);

%% --------------------------------------------------------------------------
% Angled slice interpolation - Strain calcs along slices

if strcmp('orthotropicAngle',globalOpts.matModel)
    fprintf('Calculating strain averages along both angled slices.\n')
    %//////////////////////////////////////////////////////////////////
    % Slice 1, along the fibres: strain calculation
    if angledSlice1.xMaxInd > 0 
        fprintf('Interpolating average strain over each angled slice for slice 1.\n')
        
        [strainSlice1.avg22, strainSlice1.field22] = ...
            func_calcRotatedFieldAvgAlongSlice(angledSlice1, pos, strain.mat22);
        
        [strainSlice1.avg12, strainSlice1.field12] = ...
            func_calcRotatedFieldAvgAlongSlice(angledSlice1, pos, strain.mat12);
        
        [strainSlice1.avg11, strainSlice1.field11] = ...
            func_calcRotatedFieldAvgAlongSlice(angledSlice1, pos, strain.mat11);
    else
        fprintf('Slice 1 is too long for the sample length. Cancelling calculation.\n')
        strainSlice1 = nan;
    end

    %//////////////////////////////////////////////////////////////////
    % Slice 2, transverse to the fibres: strain calculation
    if angledSlice2.xMaxInd > 0 
        fprintf('Interpolating average strain over each angled slice for slice 2.\n')
        
        [strainSlice2.avg11, strainSlice2.field11] = ...
            func_calcRotatedFieldAvgAlongSlice(angledSlice2, pos, strain.mat11);
        
        [strainSlice2.avg12, strainSlice2.field12] = ...
            func_calcRotatedFieldAvgAlongSlice(angledSlice2, pos, strain.mat12);
        
        [strainSlice2.avg22, strainSlice2.field22] = ...
            func_calcRotatedFieldAvgAlongSlice(angledSlice2, pos, strain.mat22);
    else
        fprintf('Slice 2 is too long for the sample length. Cancelling calculation.\n')
        strainSlice2 = nan;
    end
end

%% --------------------------------------------------------------------------
% Angled slice interpolation - Accel calcs along slices

if strcmp('orthotropicAngle',globalOpts.matModel)
    
    %//////////////////////////////////////////////////////////////////
    % Slice 1, along the fibres: accel calculation
    if angledSlice1.xMaxInd > 0 
        fprintf('Calculating average stress along slice 1 with the angled stress-gauge at an angle of: %.2f.\n',angledSlice1.angle)
        
        [accelSlice1.surfAvg11, angledSlice1.surfArea] = func_calcRotatedFieldAreaAvgAlongSlice(...
            angledSlice1, specimen, time, pos, accel.mat11);
        
        [accelSlice1.surfAvg22,~] = func_calcRotatedFieldAreaAvgAlongSlice(...
            angledSlice1, specimen, time, pos, accel.mat22);
        
        % Modified 04/09/2019 - to account for off-axis composites with fibres pointing
        % downwards at the impact edge. See FBD derivations for explanation. 
        if angledSlice1.angle > 0
                % Calculate the standard angled stress-gauge
                stressSlice1.avg22 = -1*(material.rho*angledSlice1.surfArea./...
                    angledSlice1.length.*accelSlice1.surfAvg22);

                stressSlice1.avg12 = -1*(material.rho*angledSlice1.surfArea./...
                    angledSlice1.length.*accelSlice1.surfAvg11);
        else
                % Calculate the standard angled stress-gauge
                stressSlice1.avg22 = (material.rho*angledSlice1.surfArea./...
                    angledSlice1.length.*accelSlice1.surfAvg22);

                stressSlice1.avg12 = (material.rho*angledSlice1.surfArea./...
                    angledSlice1.length.*accelSlice1.surfAvg11);
        end
        
    else
        fprintf('Slice 1 is too long for the sample length. Cancelling calculation.\n')
        accelSlice1 = nan;
        stressSlice1 = nan;
    end
    
    % //////////////////////////////////////////////////////////////////
    % Slice 2, transverse to the fibres: accel calculation
    if angledSlice2.xMaxInd > 0
        fprintf('Calculating average stress along slice 2 with the angled stress-gauge at an angle of: %.2f.\n',angledSlice2.angle)
        
        [accelSlice2.surfAvg11, angledSlice2.surfArea] = func_calcRotatedFieldAreaAvgAlongSlice(...
            angledSlice2, specimen, time, pos, accel.mat11);
        
        [accelSlice2.surfAvg22,~] = func_calcRotatedFieldAreaAvgAlongSlice(...
            angledSlice2, specimen, time, pos, accel.mat22);

        % Calculate the standard angled stress-gauge
        stressSlice2.avg11 = material.rho*angledSlice2.surfArea./...
            angledSlice2.length.*accelSlice2.surfAvg11;
        
        stressSlice2.avg12 = material.rho*angledSlice2.surfArea./...
            angledSlice2.length.*accelSlice2.surfAvg22;
    else
        fprintf('Slice 2 is too long for the sample length. Cancelling calculation.\n')
        accelSlice2 = nan;
        stressSlice2 = nan;
    end
    
end

%% --------------------------------------------------------------------------
% STANDARD STRESS GAUGE: find stiffness(es) by fitting the stress-strain curves
% --------------------------------------------------------------------------
fprintf('Identifying stiffness(es) by fitting the stress-strain curves.\n')

if strcmp('orthotropicReduced',globalOpts.matModel)
    fprintf('Linearly fitting the stress and strain (e_xx) components to obtain Qxx.\n')
    
    % Fit the stress strain curves to obtain Qxx/Exx
    % identStiffSG = func_identQxxFromStressStrainCurve(stressGaugeOpts,stress,strain);
    [identStiffSG.QxxVsL, identStiffSG.QxxLinFitCoeffs, identStiffSG.QxxFitFrameRange]...
        = func_identStiffLinFitStressStrainCurve(stressGaugeOpts, stress.xAvg, strain.xAvg);
    
    % Set the range over which the average stiffness is identified
    stressGaugeOpts.avgQVsLRange = round(stressGaugeOpts.avgQVsLRangePc(1)*length(pos.x))...
        :round(stressGaugeOpts.avgQVsLRangePc(2)*length(pos.x));
    
    % Calculate Identified Averages
    identStiffSG.QxxAvgOverL = nanmean(identStiffSG.QxxVsL(stressGaugeOpts.avgQVsLRange));
    identStiffSG.ExxVsL = identStiffSG.QxxVsL;
    identStiffSG.ExxAvgOverL = nanmean(identStiffSG.ExxVsL(stressGaugeOpts.avgQVsLRange));
    
    fprintf('Linearly fitting the stress and strain (e_xy) components to obtain Qss.\n')
    % Fit the stress strain curves to obtain Qss/Gxy
    [identStiffSG.QssVsL, identStiffSG.QssLinFitCoeffs, identStiffSG.QssFitFrameRange]...
        = func_identStiffLinFitStressStrainCurve(stressGaugeOpts, stress.sAvg, strain.sAvg);
    
    % Set the range over which the average stiffness is identified
    stressGaugeOpts.avgQVsLRange = round(stressGaugeOpts.avgQVsLRangePc(1)*length(pos.x))...
        :round(stressGaugeOpts.avgQVsLRangePc(2)*length(pos.x));
    
    % Calculate Identified Averages
    identStiffSG.QssAvgOverL = nanmean(identStiffSG.QssVsL(stressGaugeOpts.avgQVsLRange));
    identStiffSG.GxyVsL = identStiffSG.QssVsL;
    identStiffSG.GxyAvgOverL = nanmean(identStiffSG.GxyVsL(stressGaugeOpts.avgQVsLRange));
    
    
elseif strcmp('orthotropicAngle',globalOpts.matModel)
    
    % Set the range over which the average stiffness is identified (currently set to all slices for off-axis composites)
    stressGaugeOpts.avgQVsLRange = round(1:angledSlice1.xMaxInd);
    
    if angledSlice1.xMaxInd > 0      
        
        fprintf('Linearly fitting the 22 stress and strain components to obtain Q22.\n')
        [identStiffSG.Q22VsL, identStiffSG.Q22LinFitCoeffs, identStiffSG.Q22FitFrameRange] =...
            func_identStiffLinFitStressStrainCurve(stressGaugeOpts, stressSlice1.avg22, strainSlice1.avg22);
   
        fprintf('Linearly fitting the 12 stress and strain components to obtain Q66.\n')
        [identStiffSG.Q66VsL_Slice1, identStiffSG.Q66LinFitCoeffs_Slice1, identStiffSG.Q66FitFrameRange_Slice1] = ...
            func_identStiffLinFitStressStrainCurve(stressGaugeOpts, stressSlice1.avg12, strainSlice1.avg12);
      
    end
    
    if angledSlice2.xMaxInd > 0
        
         % Set the range over which the average stiffness is identified (currently set to all slices for off-axis composites)
        stressGaugeOpts.avgQVsLRange = round(1:angledSlice2.xMaxInd);
        
        fprintf('Linearly fitting the 22 stress and strain components to obtain Q11.\n')
        [identStiffSG.Q11VsL, identStiffSG.Q11LinFitCoeffs]...
            = func_identStiffLinFitStressStrainCurve(stressGaugeOpts, stressSlice2.avg11, strainSlice2.avg11);
   
        fprintf('Linearly fitting the 12 stress and strain components to obtain Q66.\n')
        [identStiffSG.Q66VsL_Slice2, identStiffSG.Q66LinFitCoeffs_Slice2, identStiffSG.Q66FitFrameRange_Slice2] = ...
            func_identStiffLinFitStressStrainCurve(stressGaugeOpts, stressSlice2.avg12, strainSlice2.avg12);
       
    end
    
    % Calculate identified stiffness averages (full range for angled slices)
    identStiffSG.Q22AvgOverL = nanmean(identStiffSG.Q22VsL(stressGaugeOpts.avgQVsLRange));
    identStiffSG.Q66AvgOverL_Slice1 = nanmean(identStiffSG.Q66VsL_Slice1(stressGaugeOpts.avgQVsLRange));
    identStiffSG.Q11AvgOverL = nanmean(identStiffSG.Q11VsL(stressGaugeOpts.avgQVsLRange));
    identStiffSG.Q66AvgOverL_Slice2 = nanmean(identStiffSG.Q66VsL_Slice2(stressGaugeOpts.avgQVsLRange));
    
end

% Display the identified stiffnesses
fprintf('Identified stiffness values:\n')
IdentifiedStiffness_SG = identStiffSG

%% -----------------------------------------------------------------------%
% Plotting
%-------------------------------------------------------------------------%

% Set the plot properties
plotProps = func_initPlotPropsStruct(plotParams.formatType);

%% -----------------------------------------------------------------------%
% Kinematic fields

if strcmp('yes',switches.plotImageSeqs)
    
    fprintf('Plotting kinematic fields.\n')
    
    % Base path to pass to the function to plot the fields
    savePath = ([dataPath,'\']);
    
    % Plot the fields
    func_plotAllFullFieldVideos(globalOpts,plotParams,savePath,pos,time,...
        disp,strain,strainRate,accel,material);
end
 
%% -----------------------------------------------------------------------%
% Field averages and loading pulse

if strcmp(globalOpts.plotImageSeqs,'yes')
       
    % Save path for FE Processed Images
    savePath = ([dataPath,'\','ProcessedFEData_DiagnosticFigs']);
    
    if ~exist(savePath,'dir')
        mkdir(savePath);
    end
    addpath(savePath);

    fprintf('Ploting diagnostic figures.\n')
    
    % Plot field averages and loading pulse
    func_plotGeneralFieldDiagnostics(savePath,plotParams,...
        specimen,time,pos,disp,accel,strain,strainRate,stress)

%-------------------------------------------------------------------------%
% ss-curves

    fprintf('Plotting stress-strain curves.\n')

    if strcmp('orthotropicReduced', globalOpts.matModel)
        func_plotSSCurvesSGRedOrtho(savePath, plotParams, ssCurvePlotParams, pos, stress, strain, identStiffSG)

    elseif strcmp('orthotropicAngle',globalOpts.matModel)
        func_plotSSCurvesSGOrthoAng(savePath, plotParams, ssCurvePlotParams, material,...
            angledSlice1, angledSlice2, pos, stressSlice1, strainSlice1, stressSlice2, strainSlice2, identStiffSG)

    else % Isotropic case as default for stress-gauge
        func_plotSSCurvesSGIso(savePath,ssCurvePlotParams,pos,stress,strain)
    end

%-------------------------------------------------------------------------%
% Modulus vs position

    fprintf('Plotting identified stiffness over the length of the specimen.\n')

    if strcmp('orthotropicReduced',globalOpts.matModel)
        ssCurveIdentPlotOpts.indLRange = 1:length(pos.x);
        func_plotSGIdentStiffVsLOrthoRed(savePath, plotParams, ssCurveIdentPlotOpts, pos, material, identStiffSG)

    elseif strcmp('orthotropicAngle',globalOpts.matModel)
        func_plotSGIdentStiffVsLOrthoAng(savePath, plotParams, ssCurveIdentPlotOpts, pos, material, identStiffSG,...
           angledSlice1,angledSlice2, stressGaugeOpts) 
      
    else % Isotropic case as default for stress-gauge
        func_plotSGIdentStiffVsLIso(savePath, ssCurveIdentPlotOpts, pos, material, identStiffSG)
    end
    
end

%% -----------------------------------------------------------------------%
% Saving
%-------------------------------------------------------------------------%

% Save path for all types of data files
savePath = ([dataPath,'\']);

% Save the stiffness data for plotting multiple E vs. xo figures
if strcmp('on',switches.saveStiffnessData)

    fprintf('Saving stiffness data...\n')
    
    % Save filename    
    saveFile = 'identStiffSG.mat'; % Save file name

    if strcmp('orthotropicReduced',globalOpts.matModel)
        fprintf('Saving stiffness data for multi-test comparison.\n');
        save([savePath,saveFile], 'identStiffSG','pos','stressGaugeOpts','globalOpts','specimen');

    elseif strcmp('orthotropicAngle',globalOpts.matModel)
        fprintf('Saving stiffness data for multi-test comparison.\n');
        save([savePath,saveFile], 'identStiffSG','pos','stressGaugeOpts','globalOpts','angledSlice1','angledSlice2','specimen');
    end

end

% Save all the data
if strcmp('on',switches.saveAllData)

    fprintf('Saving all processed data...\n')
    
    % Clear switches to cancel cross-code interference
    clear switches
    
    % Save filename
    saveFile = 'AllProcessed_FEData.mat';
    
    save([savePath,saveFile]);
end

% Save the average fibre strains (for e11 correction of E22) 
if strcmp('on',switches.saveAvg11_FE)
    
    fprintf('Saving the average fibre strains for the e11 correction.\n')
    
    % Clear switches to cancel cross-code interference
    clear switches
    
    % Save filename
    saveFile = 'avg11_FE.mat'; 
    
    if strcmp('orthotropicReduced',globalOpts.matModel)
        FEStrainyAvg = strain.yAvg;
        save([initPath,saveFile], 'FEStrainyAvg')

    elseif strcmp('orthotropicAngle',globalOpts.matModel)
        FEStrainSlice1Avg11 = strainSlice1.avg11;
        save([initPath,saveFile], 'FEStrainSlice1Avg11')
    end
    
end
