% Description of function:
% Subplot of full and fitted over threshold stress strain curves
%
% Date - Version - Additions
% 22/07/2019 - First version
% 05/02/2020 - Single label for figure
% 21/04/2020 - Plot Q12 (e11) corrected stress (FE e11 and Exp e11 values)
%
% Notes:
%
% 1) To turn on single axis label, set switches.individualAxisLabels
%
%-------------------------------------------------------------------------%

function [] = func_plotFullandFittedStressStrainCurveSubplotOffAxisWithFE(strain, stress, FE, ...
    slice, pos, time, dataComponent, imageSavePath, switches)

    % switches
    switches.singleAxisLabel = 0;
    switches.use_export_fig = true;

    % color generator
    [color] = func_color;

    % Setup the rows and columns for the subplot figure
    Rows = 1;
    Cols = 3;
    hInd=1; % For single axis label 2x3 subplot use: hInd=4, wInd=6. (not working)
    wInd=3; %

    % Subplot over fitted stress strain curve
    hf = figure;
    for xx = 1:length(slice.locXInd)

        % Subplot handle
        ax(xx) = subplot(Rows, Cols, xx);

        % Full curve
        hold on
        plot(strain.avg22(slice.locXInd(xx), time.tRange)*10^3,...   
        stress.avg22(slice.locXInd(xx),time.tRange)*10^-6,'-','Color', color.blue);

        % Q12 correction using Experimental data
        % Full curve
        plot(strain.avg22(slice.locXInd(xx), time.tRange)*10^3,...   
        stress.avg22Corr(slice.locXInd(xx),time.tRange)*10^-6,'--','Color', color.green);

        % Q12 correction using FE data
        % Full curve
        plot(strain.avg22(slice.locXInd(xx), time.tRange)*10^3,...   
        FE.stressSlice1.avg22Corr(slice.locXInd(xx),time.tRange)*10^-6,'-','Color', color.red);

        hold off

        % Font Sizing
        globalFontSize = 15;

        % Grid
        grid on

        % Subplot Title
        func_titleSubplot(pos, slice, xx, globalFontSize);

        % Axis limits and aspect ratio
        func_axisLimits(dataComponent, globalFontSize);

        if ~switches.singleAxisLabel
            % Axis label strings
            func_labelStringGenerator(dataComponent, globalFontSize);
        end

    end

    if switches.singleAxisLabel
        % Determine common axis position with handles (adjust with factors) 
        % [YLabel_x XLabel_y XLabel_y YLabel_y]
        height = ax(hInd).Position(2) + ax(hInd).Position(4) -  ax(hInd).Position(2);
        width = ax(wInd).Position(1) - ax(hInd).Position(1) + ax(hInd).Position(3);
        ax = axes('position',[ax(hInd).Position(1)*0.85 ax(hInd).Position(2)*0.85 width height],'visible','off');    
        ax.XLabel.Visible = 'on';
        ax.YLabel.Visible = 'on';

        % Generate label strings *note: Cant pass strings assigned to structs through functions
        func_labelStringGenerator(dataComponent, globalFontSize);
    end

    % Set the figure background colour to white
    set(gcf,'color','w');

    % Figure Sizing
    figureType = 'ssCurveSubplot1x3'; % define based on subplot rows x cols set at top
    [hf] = func_figureSize(hf, figureType, switches);

    % Create a new directory to save the results
    savePath = [imageSavePath,'\e11Correction'];
        if ~exist(savePath,'dir')
            mkdir(savePath);
        end
    addpath(savePath);

    % Save file name
    saveFile = [savePath,'\',dataComponent];

    if switches.use_export_fig
        % Make images using export_fig
        export_fig(saveFile,'-png','-r400','-c[inf,inf,inf,inf]',hf)
    else
        % Make images using the print command
        print(hf,saveFile,'-dpng','-r500'); 
    end

    % Save figure as a mat file
     saveas(hf,saveFile,'fig')

end