% Description of function:
% Subplot of full and fitted ss-curves over the thresholded range, as various
% slices.
%
% Date - Version - Additions
% 22/07/2019 - Ver1 
% 06/02/2020 - Added individual Title, Axis Limits, Axis Labels and Figure
% Sizing functions. 
% 
% 12/02/2020 - Changed locXInd to slice.locXInd however, angledSlice1 is passed in multiple times
% to keep the plot functions more compact within the function.
%
% Notes:
%
% Change angledSlice1.ssFitVal_s22 to 'angledSlice1.ssFitOverLVal_E22' in the main code
% Change fittedStress{xx} to fittedStress{slice.locXInd(xx)} here.
%
%-------------------------------------------------------------------------%

function [] = func_plotFullandFittedStressStrainCurveSubplotVer2(Strain, Stress, fitFrameRange,...
    slice, fittedStress, pos, time, dataComponent, imageSavePath, material)

    % switches
    switches.singleAxisLabel = 0;
    switches.use_export_fig = true;

    % Crop frames for modulus report.
    crpFrames = 16;

    % Setup the rows and columns for the subplot figure
    Rows = 1;
    Cols = 3;
    hInd=1; %4 used for single axis label
    wInd=3; %6

    % Subplot over fitted stress strain curve
    hf = figure;
    for xx = 1:length(slice.locXInd)
        
        % Subplot handle
        ax(xx) = subplot(Rows, Cols, xx); %insert subplots in plot for i=1:8
        
        hold on
        
        % Full curve
        plot(Strain(slice.locXInd(xx), time.tRange(1:end-crpFrames))*10^3,...   
        Stress(slice.locXInd(xx),time.tRange(1:end-crpFrames))*10^-6,...
        '-+b');
                
        % Thresholded curve
        plot(Strain(slice.locXInd(xx), fitFrameRange{slice.locXInd(xx)})*10^3,...
            Stress(slice.locXInd(xx), fitFrameRange{slice.locXInd(xx)})*10^-6,'-+r');
        
        % Fit to Thresholded curve
         plot(Strain(slice.locXInd(xx),fitFrameRange{slice.locXInd(xx)})*10^3,...
             fittedStress{slice.locXInd(xx)}*10^-6,'k-'); % Plot the fitted y-values
        
        hold off

        % Grid on
        grid on
        
        % Font Sizing
        globalFontSize = 15;
        
        % Subplopt Title
        func_titleSubplot(pos, slice, xx, globalFontSize);

        % Axis Limits
        func_axisLimits(dataComponent, globalFontSize, Strain, Stress, material);

        if ~switches.singleAxisLabel
            % Axis label strings
            func_labelStringGenerator(dataComponent, globalFontSize);
        end
    
    end

    if switches.singleAxisLabel
        % Determine common axis position with handles (adjust with factors) 
        % [YLabel_x XLabel_y XLabel_y YLabel_y]
        height = ax(hInd).Position(2) + ax(hInd).Position(4) -  ax(hInd).Position(2);
        width = ax(wInd).Position(1) - ax(hInd).Position(1) + ax(hInd).Position(3);
        ax = axes('position',[ax(hInd).Position(1)*0.85 ax(hInd).Position(2)*0.85 width height],'visible','off');    
        ax.XLabel.Visible = 'on';
        ax.YLabel.Visible = 'on';

        % Generate label strings *note: Cant pass strings assigned to structs through functions
        func_labelStringGenerator(dataComponent, globalFontSize);
    end

    % Create a new directory to save the results
    savePath = [imageSavePath,'\ssCurves'];
        if ~exist(savePath,'dir')
            mkdir(savePath);
        end
    addpath(savePath);

    % Save file name
    saveFile = [savePath,'\',dataComponent];

    % Figure Sizing
    figureType = 'ssCurveSubplot1x3';  % define based on subplot rows x cols set at top
    func_figureSize(hf, figureType, switches);

    % Set the figure background colour to white
    set(gcf,'color','w');

    if switches.use_export_fig
        % Make images using export_fig
        export_fig(saveFile,'-png','-r400','-c[inf,inf,inf,inf]',hf) %[top,right,bottom,left]
    else
        % Make images using the print command
        print(hf,saveFile,'-dpng','-r500'); 
    end

    % Save figure as a mat file
     saveas(hf,saveFile,'fig')

end