% Description of function:
% Subplot of shear ss-curves and the progressive chord modulus
%
% Date - Version - Additions
% 22/07/2019 - Ver1
%
% Notes:
%
%-------------------------------------------------------------------------%

function [] = func_plotSSCurveAndProgressiveChordModulusSubplot(Strain, Stress, slice, time, fitFrameRange,...
    progressiveModulus, meanProgressiveModulus, fitFrameRange_avgChordModulus, fitRangeLinearRegion, linRangeEndInd, dataComponent, imageSavePath)

    % switches
    switches.singleAxisLabel = 0;
    switches.use_export_fig = true;

    %----------------------------------------------------------------------
    % Crop frames for modulus report.
    crpFrames = 16;
    
    % Font Sizing
    globalFontSize = 15;

    % Setup the rows and columns for the subplot figure
    Rows = 3;
    Cols = 3;

    hf = figure;
    
    % Subplot over fitted stress strain curve
    for xx = 1:length(slice.locXInd)
        
        % Subplot handle
        ax(xx) = subplot(Rows, Cols, xx);
        
        hold on
        
        % Full curve
        plot(Strain(slice.locXInd(xx), time.tRange(1:end-crpFrames))*10^3,...   
            Stress(slice.locXInd(xx),time.tRange(1:end-crpFrames))*10^-6,...
        '-+b');
    
        % Thresholded curve
        plot(Strain(slice.locXInd(xx), fitFrameRange{slice.locXInd(xx)})*10^3,...
            Stress(slice.locXInd(xx), fitFrameRange{slice.locXInd(xx)})*10^-6,'-+r');      
        
        % Axis labels
        xLabelStr = ('Strain, $\overline{\gamma_{12}}^{L}$ (mm.m$^{-1}$)');
        yLabelStr = ('Stress, $\overline{\sigma_{12}}^{L}$ (MPa)');
        func_axisLabels(xLabelStr, yLabelStr, globalFontSize)
        
        % Title
        titleStrings{1} = (['$x_{0} = \, $',sprintf('%0.1f', slice.sliceRangeX(slice.locXInd(xx))*10^3),' mm']);
        plotProperties.titleFontSize = globalFontSize;
        func_title(titleStrings, plotProperties)  
        
        hold off

        % Grid on
        grid on
        
    end
    
    % Chord modulus vs. strain (On the second row)
    for xx = 1:length(slice.locXInd)
        
        % Subplot handle
        ax(xx+Cols) = subplot(Rows, Cols, xx+Cols);
        
        hold on
             
        % Plot the progressive chord modulus value vs. strain
        plot(Strain(slice.locXInd(xx),fitFrameRange{slice.locXInd(xx)})*10^3, ...
            progressiveModulus(slice.locXInd(xx),1:length(fitFrameRange{slice.locXInd(xx)}))*10^-9,'xb-');     

        % Plot the mean       
        plot([Strain(slice.locXInd(xx),fitFrameRange_avgChordModulus{slice.locXInd(xx)}(1))*10^3, Strain(slice.locXInd(xx),fitFrameRange_avgChordModulus{slice.locXInd(xx)}(end))*10^3],...
            [meanProgressiveModulus(slice.locXInd(xx))*10^-9, meanProgressiveModulus(slice.locXInd(xx))*10^-9],'r--');   
        
        % plot the index at the predicted end of the linear range (linRangeEndInd)
        plot([Strain(slice.locXInd(xx),linRangeEndInd(slice.locXInd(xx)))*10^3 , Strain(slice.locXInd(xx),linRangeEndInd(slice.locXInd(xx)))*10^3],...
            [0, meanProgressiveModulus(slice.locXInd(xx))*10^-9],'k--');
        
        % Axis limits
        ylim([3 9]);

        % Axis labels
        xLabelStr = ('Strain, $\overline{\gamma_{12}}^{L}$ (mm.m$^{-1}$)');
        yLabelStr = ('Chord Modulus, (GPa)');
        func_axisLabels(xLabelStr, yLabelStr, globalFontSize)
               
        % Title
        titleStrings{1} = (['$x_{0} = \, $',sprintf('%0.1f', slice.sliceRangeX(slice.locXInd(xx))*10^3),' mm']);
        plotProperties.titleFontSize = globalFontSize;
        func_title(titleStrings, plotProperties)  
        
        hold off

        % Grid on
        grid on
        
    end
      
    % Subplot of ss curves with fitRangeLinearRegion (chord fit)
    for xx = 1:length(slice.locXInd)
        
        % Subplot handle
        ax(xx) = subplot(Rows, Cols, xx+Cols*2);
        
        hold on
        
        % Full curve
        plot(Strain(slice.locXInd(xx), time.tRange(1:end-crpFrames))*10^3,...   
            Stress(slice.locXInd(xx),time.tRange(1:end-crpFrames))*10^-6,...
        '-+b');
    
        % Thresholded curve determined by chord fit
        plot(Strain(slice.locXInd(xx), fitRangeLinearRegion{slice.locXInd(xx)})*10^3,...
            Stress(slice.locXInd(xx), fitRangeLinearRegion{slice.locXInd(xx)})*10^-6,'-+r');      
        
        % Axis labels
        xLabelStr = ('Strain, $\overline{\gamma_{12}}^{L}$ (mm.m$^{-1}$)');
        yLabelStr = ('Stress, $\overline{\sigma_{12}}^{L}$ (MPa)');
        func_axisLabels(xLabelStr, yLabelStr, globalFontSize)
        
        % Title
        titleStrings{1} = (['$x_{0} = \, $',sprintf('%0.1f', slice.sliceRangeX(slice.locXInd(xx))*10^3),' mm']);
        plotProperties.titleFontSize = globalFontSize;
        func_title(titleStrings, plotProperties)  
        
        hold off

        % Grid on
        grid on
        
    end    
    
    %----------------------------------------------------------------------
    % Create a new directory to save the results
    savePath = [imageSavePath,'\chordFit'];
        if ~exist(savePath,'dir')
            mkdir(savePath);
        end
    addpath(savePath);

    % Save file name
    saveFile = [savePath,'\',dataComponent,'AndSSCurveSubplot'];

    % Figure Sizing
    figureType = 'ssCurveSubplot3x3';  % define based on subplot rows x cols set at top
    func_figureSize(hf, figureType, switches);

    % Set the figure background colour to white
    set(gcf,'color','w');

    if switches.use_export_fig
        % Make images using export_fig
        export_fig(saveFile,'-png','-r400','-c[inf,inf,inf,inf]',hf) %[top,right,bottom,left]
    else
        % Make images using the print command
        print(hf,saveFile,'-dpng','-r500'); 
    end

    % Save figure as a mat file
     saveas(hf,saveFile,'fig')

end