% Description of function:
% Plots E22 modulus vs length (from thresholded stress-strain curves) for all slices
%
% Date - Version - Additions
% 22/07/2019 - Ver1
% 13/02/2020 - Replaced func_plotStiffParameters with individual functions
%              axisLimits, labelStringGenerator, figureSize.
% 21/04/2020 - Plot Q12 corrected E22 values
%
% Notes:
%
%-------------------------------------------------------------------------%

function [identStiffAvg, identStiffAvgCorr, identStiffAvgCorrFE] = func_plotStiffnessVsLQ12Corr(slice, e11CorrSlice1, pos,...
    dataComponent, imageSavePath, material, globalOpts, plotProperties)

    % switches
    switches.use_export_fig = true;

    % Variable names for stiffness component vs x_0
    if strcmp('orthotropicAngle',globalOpts.matModel)
        identStiffThreshOverL = slice.identStiffThreshOverL_E22;
        identStiffThreshOverLCorr = e11CorrSlice1.identStiffThreshOverL_E22Corr;
        identStiffThreshOverLCorrFE = e11CorrSlice1.identStiffThreshOverL_E22CorrFE;
        % QS modulus value
        QS_modulus = material.Eyy;
        
    elseif strcmp('orthotropicReduced',globalOpts.matModel)
        identStiffThreshOverL = slice.identStiffThreshOverL_Exx;
        identStiffThreshOverLCorr = e11CorrSlice1.identStiffThreshOverL_ExxCorr;
        identStiffThreshOverLCorrFE = e11CorrSlice1.identStiffThreshOverL_ExxCorrFE;
        % QS modulus value
        QS_modulus = material.Exx;
    end
    
    % Average Stiffness over threshold range for plotting
    identStiffAvg = mean(identStiffThreshOverL(slice.stiffPlotRangeInd));
    identStiffAvgCorr = mean(identStiffThreshOverLCorr(slice.stiffPlotRangeInd));
    identStiffAvgCorrFE = mean(identStiffThreshOverLCorrFE(slice.stiffPlotRangeInd));
    
    % color generator
    [color] = func_color;
         
    % Plot: Stiffness
    hf = figure;
    
    hold on
          
    if strcmp('ExxCorr',dataComponent)
        
        % E22 data
        plot(pos.x*10^3,identStiffThreshOverL*10^-9, '-','Color', color.blue);
                
        % E22 data (Corrected, Experimental e11 values)
        plot(pos.x*10^3,identStiffThreshOverLCorr*10^-9, '-','Color', color.green);
        
        % E22 data (Corrected, FE e11 values)
        plot(pos.x*10^3,identStiffThreshOverLCorrFE*10^-9, '-','Color', color.red);
        
        % QS reference data
        plot([pos.x(1),pos.x(end)*10^3], [QS_modulus*10^-9, QS_modulus*10^-9],'k--');   % QS ref.
        plot([pos.x(1),pos.x(end)*10^3], [1.1*QS_modulus*10^-9, 1.1*QS_modulus*10^-9],'k-');   % QS ref. +10% 
        plot([pos.x(1),pos.x(end)*10^3], [0.9*QS_modulus*10^-9, 0.9*QS_modulus*10^-9],'k-');   % QS ref. -10%
        
    elseif strcmp('E22Corr',dataComponent)
        
        % E22 data
        plot(pos.x(slice.stiffPlotRangeInd)*10^3,identStiffThreshOverL*10^-9, '-','Color', color.blue);
        
        % E22 data (Corrected, Experimental e11 values)
        plot(pos.x(slice.stiffPlotRangeInd)*10^3,identStiffThreshOverLCorr*10^-9, '-','Color', color.green);
        
        % E22 data (Corrected, FE e11 values)
        plot(pos.x(slice.stiffPlotRangeInd)*10^3,identStiffThreshOverLCorrFE*10^-9, '-','Color', color.red);
        
        % QS ref
        plot([pos.x(1),pos.x(end)*10^3], [QS_modulus*10^-9, QS_modulus*10^-9],'k--');   % QS ref.
        plot([pos.x(1),pos.x(end)*10^3], [1.1*QS_modulus*10^-9, 1.1*QS_modulus*10^-9],'k-');   % QS ref. +10% 
        plot([pos.x(1),pos.x(end)*10^3], [0.9*QS_modulus*10^-9, 0.9*QS_modulus*10^-9],'k-');   % QS ref. -10%
               
    end
    hold off
       
    % Global Font Size
    globalFontSize = 12;
    
    % Grid 
    grid on
    
    % Legend
    legendStrings = {...
        'Ident.',...
        '$Q_{12}$ corr. Exp. $\epsilon_{11}$',...
        '$Q_{12}$ corr. FE $\epsilon_{11}$',...
        'QS Ref.',...
        'QS Ref. $\pm$10$\%$'};
    func_legend(legendStrings, plotProperties);
    
    % Axis limits
    func_axisLimits(dataComponent, globalFontSize);
    
    % Axis labels and properties
    func_labelStringGenerator(dataComponent, globalFontSize);
    
    % Set the figure background colour to white
    set(gcf,'color','w');
    
    % Figure size
    figureType = 'modulusVsL';
    func_figureSize(hf, figureType, switches);
    
    % Create a new directory to save the results
    savePath = [imageSavePath,'\e11Correction'];
        if ~exist(savePath,'dir')
            mkdir(savePath);
        end
    addpath(savePath);

    % Save file name
    saveFile = [savePath,'\',dataComponent];

    if switches.use_export_fig
        % Make images using export_fig
        export_fig(saveFile,'-png','-r400','-c[inf,inf,inf,inf]',hf)
    else
        % Make images using the print command
        print(hf,saveFile,'-dpng','-r500'); 
    end

    % Save figure as a mat file
     saveas(hf,saveFile,'fig')
end
