% func_progressiveChordModulusFit_ver4.m
%
% Calculates the linear limit and linear fitting range for the shear 
% stress-strain curves based on progressive chord modulus fits.
%
% By: Sam Parry
%-------------------------------------------------------------------------%
% Summary
%
% (1) Determines the fitting range over the shear loading, based on the
%     maximum stress or strain.
% 
% (2) Defines the progressively increasing fit range for each chord modulus
%     calculation, over the loading portion of the shear ss-curve.
%
% (3) Progressively fits chord modulus values (for each slice).
%
% (4) Calculates the mean chord modulus.
%
% (5) Determines the index where the chord modulus hits the Mean, then gives
%     the linear fitting region. 
%
% Date - Version - Additions
% 29/08/2019 - Ver1
% 
% 16/09/2019 - ver2 - Changed chord fit method to have start and end value.
%                   - Option for normal modulus (in preference to chord) fits.
%
% 11/10/2019 - ver3 - More plotting functions
% 
% 06/07/2020 - ver4 - Code streamlining and removal normal modulus calculation,
%                   - as chord method was more reliable.
%-------------------------------------------------------------------------%
% Notes: 
% (1) Use processMode = 0 to check the fits. Clear vars when debugging.
%-------------------------------------------------------------------------%

function [fitFrameRange, progressiveModulus, avgChordModulus, fitRangeLinearRegion, linRangeEndInd] =...
    func_progressiveChordModulusFit_ver4(strainVar, stressVar, slice,...
    time, chordFitParams, dataComponent, imageSavePath)

%-------------------------------------------------------------------------%
% Set to true for regular processing without debugging. Set to false for debugging mode. 
processMode = 0;

% If we are in debug mode, clear these vars to prevent false values between runs.
if ~processMode
    clear fitFrameRange progressiveModulus meanProgressiveModulus fitRangeLinearRegion linRangeEndInd
end  

% ------------------------------------------------------------------------%
% (1) Determine the FitFrameRange over the shear loading 
% ------------------------------------------------------------------------%
    
% Max number of slices on the sample
NumberOfSlicesVar = slice.xMaxInd;

% Fit to the strain or stress fields (this section only i.e. not to chord fits)
fitField = stressVar; % Alternative: strainVar

% Start index
for xx = 1:NumberOfSlicesVar
    startInd(xx) = time.tRange(1);  %Define as a vector, not a matrix, starting at the first time range value.
    if chordFitParams.minStrainThreshold_12 ~= 0
        for tt = time.tRange(1):time.tRange(end)    % For all tRange (not total frames)
            if abs(fitField(xx,tt)) > abs(chordFitParams.minStrainThreshold_12)
            startInd(xx) = tt;
            break;
            end
        end
    else
    startInd(xx) = time.tRange(1);
    end
end

% (2) End index
for xx = 1:NumberOfSlicesVar                      
    [maxVal(xx), endInd(xx)] = max(abs(fitField(xx,:)));
    fitFrameRange{xx} = startInd(xx):endInd(xx);
end

% ------------------------------------------------------------------------%
% (2) Define the progressively increasing fit range over the shear loading
% ------------------------------------------------------------------------%

for xx = 1:NumberOfSlicesVar
    for ii = 1:length(fitFrameRange{xx})  % Over the loading portion of the curve

        % Increment the end index
        progEndInd(xx,ii) = startInd(xx) + (ii-1);  

        % Progressively fill the fit range (takes some time to load)
        fitFrameRangeProg{xx,ii} = startInd(xx):progEndInd(xx,ii);

    end
end
            
% ------------------------------------------------------------------------%
% (3) Progressively fit the chord modulus.
% ------------------------------------------------------------------------%

if strcmp('Slice1_chordModulus',dataComponent) || strcmp('Slice2_chordModulus',dataComponent) || strcmp('chordModulus',dataComponent)

    for xx = 1:NumberOfSlicesVar
        progressiveModulus(xx,1) = 0;
        for ii = 2:length(fitFrameRange{xx})    % Start from 2 so that no zero modulus value is computed for ii = 1. 

            % Progressively identify the chord modulus (dely/delx)
            progressiveModulus(xx,ii) = (stressVar(xx,fitFrameRangeProg{xx,ii}(end)) - stressVar(xx,fitFrameRangeProg{xx,ii}(1)) )/ ...
            (strainVar(xx,fitFrameRangeProg{xx,ii}(end)) - strainVar(xx,fitFrameRangeProg{xx,ii}(1)) );
        end
    end
end

% ------------------------------------------------------------------------%
% (4) Average of the chord modulus.
% ------------------------------------------------------------------------%

% Start the average chord modulus calculation at the index that coincides
% with the strain noise floor.

% noiseFloorAdjusted value in the init file.

% Determine the index where the strain hits the noiseFloorAdjusted level.

% Start index for the average chord modulus calculation
for xx = 1:NumberOfSlicesVar
    for tt = 1:length(fitFrameRange{xx})
        if abs(strainVar(xx,tt)) > abs(chordFitParams.noiseFloorAdjusted)
        startInd_AvgChordModulus(xx) = tt;
        break;
        end
    end
    
    % Frames to skip at the end of the mean chord modulus calculation (from init file).
        
    % End index for the progressive chord modulus
    endInd_AvgChordModulus(xx) = fitFrameRange{xx}(end) - chordFitParams.skipFramesEnd_AvgChordModulus; 
    
    % Fit frame range for the mean chord modulus calculation.
    fitFrameRange_avgChordModulus{xx} = startInd_AvgChordModulus(xx): endInd_AvgChordModulus(xx); 

    % Calculate the average chord modulus over the average chord modulus fitting range determined above
    avgChordModulus(xx) = mean(progressiveModulus(xx, fitFrameRange_avgChordModulus{xx}));
    
end

% ------------------------------------------------------------------------%
% (5) Determine when the chord modulus falls below the average chord modulus
% ------------------------------------------------------------------------%

% If three (3) progressive chord modulus values fall below the average, the first one
% that does is the index for the linear limit strain

for xx = 1:NumberOfSlicesVar
    for ii = startInd_AvgChordModulus(xx):length(fitFrameRange{xx}-1) % Start at 2, because of zero modulus at first calculation. ii range is -3 due to looking forward for (3) condition
        if abs(progressiveModulus(xx,ii-1:ii+1)) < abs(avgChordModulus(xx)) 
        linRangeEndInd(xx) = ii;
        break;
        end
    end
end

% If the progressive chord modulus fits don't fall below the average, or fall too early due to noise, 
% set the end index to the length of the (fitFrameRange for each slice) multiplied by a factor (see init file)
for xx = 1:NumberOfSlicesVar
    if linRangeEndInd(xx) < ceil(length(fitFrameRange{xx})*chordFitParams.PcFitFrameRangeMidPoint)
        linRangeEndInd(xx) = ceil(length(fitFrameRange{xx})*chordFitParams.PcFitFrameRangeMidPoint);
    end
end

% Finally, give the range over which the shear modulus is calculated over. Separate loop better for debugging.
for xx = 1:NumberOfSlicesVar
    fitRangeLinearRegion{xx} = startInd(xx): linRangeEndInd(xx);
end

%--------------------------------------------------------------------------
    
% Debug Plot
if ~processMode
    func_plotSSCurveAndProgressiveChordModulusSubplot(strainVar, stressVar, slice, time, fitFrameRange,...
        progressiveModulus, avgChordModulus, fitFrameRange_avgChordModulus, fitRangeLinearRegion, linRangeEndInd, dataComponent, imageSavePath);

end  
    
end