% Description of function:
% Finds the min and max strain thresholds for each slice, over each frame
% using the approach of Castres et al, 2017. This is done by:
%
% 1) Assume an initial shear modulus.
% 2) Calculate the theoretical shear strain.
% 3) Calculate the normalised difference in exp/theory shear strain, d2.
% 4) When this is greater than 5%, take that as the upper limit strain.
% 5) Place the timestep-1 when d2 > 5% as the upper frameRange for that slice.
%
% Date - Version - Additions
% 22/08/2019 - Ver1 
%
% Notes: 
%-------------------------------------------------------------------------%

function [fitFrameRange] = func_strainThresholdOverLinearPortion(strainVar, minStrainThreshold,...
    maxStrainThreshold, numberOfSlicesVar, tRange, G12Theory, stressVar, variationInStrain)

    
    % Find the start index
    for xx = 1:numberOfSlicesVar
        startInd(xx) = tRange(1); %Define as a vector, not a matrix, starting at the first time range value.
        if minStrainThreshold ~= 0
            for tt = tRange(1):tRange(end)    % for all tRange (not total frames)
                if abs(strainVar(xx,tt)) > abs(minStrainThreshold)
                startInd(xx) = tt;
                break;
                end
            end
        end
    end

    % Find the end index
    for xx = 1:numberOfSlicesVar
        endInd(xx) = tRange(end); 
        for tt = tRange(29):tRange(end)     % Ignore the inital part
            e12Theory = abs(stressVar./G12Theory);  % Theoretical strain value
            d2 = (abs(strainVar - e12Theory))./strainVar;  % Calculate the normalised difference in shear strain
            d2(isinf(d2)) = 0;  % Replace inf with zeros
            if abs(d2(xx,tt)) > abs(variationInStrain)
                endInd(xx) = tt-1;
                break
            end
        end
        fitFrameRange{xx} = startInd(xx):endInd(xx);  % Make a 1x384 cell with start and end frames for each slice. 
    end

end