% main_OffAxisPostProcessor_v2.m
%
% Analyse material properties of off-axis composites evaluated in IBII 
% tests AFTER running the main IBII processing code.
%
%-------------------------------------------------------------------------%
% This code has three main features: 
%
% 1) Chord modulus fit (UD45 case only)
%    switches.chordModulusFitting = 'on';    
%    Determine the linear limit of the shear stress-strain curves with
%    progressive chord modulus fits. 
%
% 2) Fibre strain correction for the E22 modulus identification
%    switches.Q12Correction = 'on';
%    Correct the E22 modulus with the FE average fibre strains. This requires
%    'main_IBIIProcessing_v3_0ir_ImportFEFields.m' to be executed first.
%
% 3) Normalised strain rate
%    switches.normalisedStrainRate = 'on';
%    Calculate the strain-normalised strain rate.
%-------------------------------------------------------------------------%
% Version:
%
% v1 - By Sam Parry, 23/04/2020
% v2 - By Sam Parry, 13/08/2020, updates and code streamlining 
%-------------------------------------------------------------------------%
% Notes:
% 
% 1) Before using this code, you need to have processed an IBII test with
%    the code 'main_IBIIProcessing_V3_0ir.m' and saved the output from that
%    code with the name 'AllProcessedData.mat'.
%-------------------------------------------------------------------------%

close all
clear all
clc

%-------------------------------------------------------------------------%
% Switches
%-------------------------------------------------------------------------%
hardCodePath = false;

% Code features
switches.chordModulusFitting = 'off'; % Progressively fit chord moduli to the shear ss-curves to determine shear linear limit.
switches.Q12Correction = 'on'; % Correct the E22 modulus with the FE average fibre strains.
switches.normalisedStrainRate = 'on'; % Determine normalised strain rates.

% Plot options
switches.ssCurvesToFracFrame = 1; % Plot the ss curves to the frac frame only. 

% Saving
switches.saveAllData = 'on'; 
switches.saveStiffnessData = 'on';

%% -----------------------------------------------------------------------%
% Functions path

fprintf('Adding paths for processing functions.\n')

if ~hardCodePath
    funcPath = ([uigetdir('E:\experimentalData\ModulusPaper\ProcessingScripts\code3_IBII_TestPostProcessor\','Select path to processing functions'), '\']);
else
    % Add function path for this code
    funcPath = 'E:\experimentalData\ModulusPaper\ProcessingScripts\code3_IBII_TestPostProcessor\';

    % If the default path is not found we should find it
    if exist(funcPath,'file') ~= 7
        hWarn = warndlg('Folder for processing functions not found.','Function folder not found');
        waitfor(hWarn);
        funcPath = uigetdir(pwd,'Locate Processing Function Folder');
    end
end

% Add paths to Matlab
addpath(funcPath);
addpath([funcPath,'GridMethodToolbox\']);
addpath([funcPath,'Export_Fig\']);
addpath([funcPath,'linspecer\']);

%% -----------------------------------------------------------------------%
% Post-processing init file

if ~hardCodePath
    [initFile,initPath] = uigetfile({'*.*','All Files'},'Select the post-processing init file');
else
    % Note: not an image file, but var name kept for conformity to other codes.
    initFile = 'post-ProcessingParameters_v1.mat';
    initPath = 'E:\experimentalData\ModulusPaper\UD45-S1\';
end

% Load the processing parameters from file
load([initPath,initFile])

%% -----------------------------------------------------------------------%
% % Load the FE average fibre strain data for the Q12 correction 

if strcmp('on', switches.Q12Correction)
    
    fprintf('Load the FE averege fibre strains, interpolated to this test geometry.\n') 
    if ~hardCodePath
        [FEDataFile,FEDataPath] = uigetfile({'*.*','All Files'},'Select the FE averege fibre strain file');
    else   
        
        if strcmp('orthotropicReduced', globalOpts.matModel)
            FEDataPath = 'E:\experimentalData\ModulusPaper\FE_UD90\matlabProcessing\';  
        elseif strcmp('orthotropicAngle', globalOpts.matModel)
            FEDataPath = 'E:\experimentalData\ModulusPaper\FE_UD45\matlabProcessing\';
        end
        FEDataFile = 'avg11_FE.mat';
        
    end
    load([FEDataPath,FEDataFile])
end

%% -----------------------------------------------------------------------%
% Load the experimental processed data file

if ~hardCodePath
    fprintf('Loading the experimental processed data file.\n')
    [imageFile,imagePath] = uigetfile({'*.*','All Files'},'Select the experimental processed data file');
else
    % Note: not an image file, but var name kept for conformity to other codes.
    imageFile = 'AllProcessedData.mat';
    imagePath = 'E:\experimentalData\ModulusPaper\UD90-S1\';
    %imagePath = 'E:\experimentalData\ModulusPaper\UD45-S1\';
end

% Add path to Matlab
addpath(imagePath)

% Load the post-processed data file
fprintf('Loading the post-processed data file.\n')
load([imagePath,imageFile])


%% \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
% Preamble for this file 
%-------------------------------------------------------------------------%
% Specify all the parameters for the chord fitting, e11 correction and the
% normalised strain rate calculations. 

if strcmp('orthotropicAngle',globalOpts.matModel)

    %-------------------------------------------------------------------------%
    % Thresholds for manual fitting % In strain units (m/m)

    % Transverse
    manualFitParams.minThreshold_22 = 0; 
    manualFitParams.maxThreshold_22 = NaN;

    % Shear
    manualFitParams.minThreshold_12 = 0;
    manualFitParams.maxThreshold_12 = 10e-3;

    %-------------------------------------------------------------------------%
    % Slice and Time Indexes (These rely on xMaxInd, imported from AllProcessedData.m)

    % Vector containing the percentage of total slices, used for plotting ss-curves
    angledSlice1.locXPcVec = [0.25,0.5,0.75];
    angledSlice2.locXPcVec = [0.25,0.5,0.75];
    
    % Index of the percentage of total slices
    angledSlice1.locXInd = round(angledSlice1.xMaxInd*angledSlice1.locXPcVec);
    angledSlice2.locXInd = round(angledSlice2.xMaxInd*angledSlice2.locXPcVec);
    
    % x-position over the range of slices (m)
    angledSlice1.sliceRangeX = 0:pos.xStep:pos.xStep*(angledSlice1.xMaxInd-1);
    angledSlice2.sliceRangeX = 0:pos.xStep:pos.xStep*(angledSlice2.xMaxInd-1);
    angledSlice1.sliceRangeX = pos.xStep/2 : pos.xStep : pos.xStep*(angledSlice1.xMaxInd-1) + pos.xStep/2; 
    angledSlice2.sliceRangeX = pos.xStep/2 : pos.xStep : pos.xStep*(angledSlice2.xMaxInd-1) + pos.xStep/2;
    
    % x0 distance on the sample to plot ss-curves. Note: different to chord fit version below 
    angledSlice1.slicex0Dist = 0.006; % (m)
    angledSlice2.slicex0Dist = 0.006; % (m)

    % xIndex of the slice distance above
    angledSlice1.xInd = find(abs(angledSlice1.sliceRangeX - angledSlice1.slicex0Dist) < 0.0001);
    angledSlice2.xInd = find(abs(angledSlice2.sliceRangeX - angledSlice2.slicex0Dist) < 0.0001);
    
    % Percentage of max number of slices for stiffness vs. length plots
    angledSlice1.stiffPlotRangeVec = [1 1];
    angledSlice2.stiffPlotRangeVec = [1 1]; 

    % Index range for identified stiffness vs. length plots
    angledSlice1.stiffPlotRangeInd = 1 : angledSlice1.stiffPlotRangeVec(2)*angledSlice1.xMaxInd; % Use all slices for off-axis specimens.
    angledSlice2.stiffPlotRangeInd = 1 : angledSlice2.stiffPlotRangeVec(2)*angledSlice2.xMaxInd; % Use all slices for off-axis specimens.

    %-------------------------------------------------------------------------%
    % Chord fitting properties for both slices

    % Used in the chord modulus vs. shear strain (single plot) to check linear limit strain
    chordFitSlice1.slicex0Dist = 0.016; % (m)
    chordFitSlice2.slicex0Dist = 0.006; % (m)
    
    % xIndex of the slice distance above
    chordFitSlice1.xInd = find(abs(angledSlice1.sliceRangeX - chordFitSlice1.slicex0Dist) < 0.0001);
    chordFitSlice2.xInd = find(abs(angledSlice2.sliceRangeX - chordFitSlice2.slicex0Dist) < 0.0001);

elseif strcmp('orthotropicReduced',globalOpts.matModel)
    
    % Transverse and shear fit limits
    verticalSlice.minStrainThreshold_xx = 0;
    verticalSlice.minStrainThreshold_ss = 0;
    
    % Vector containing the percentage of total slices (used for plotting ss-curves)
    verticalSlice.locXPcVec = [0.25,0.5,0.75];

    % Max number of slices
    verticalSlice.xMaxInd = length(ssCurveIdentPlotOpts.indLRange);
    
    % Index of the percentage of total slices (used for plotting ss-curves)
    verticalSlice.locXInd = round(verticalSlice.xMaxInd *verticalSlice.locXPcVec);
    
    % x-position over slices
    verticalSlice.sliceRangeX = pos.xStep/2 : pos.xStep : pos.xStep*(length(ssCurveIdentPlotOpts.indLRange) - 1) + pos.xStep/2;   

    % Index range for the identified stiffness vs. length plots (mid 50% for the UD90 case - e.g. 93:280)
    verticalSlice.stiffPlotRangeInd = stressGaugeOpts.avgQVsLRange; 

    % x-position of the range for the identified stiffness vs. length plots (mid 50% for the UD90 case)
    verticalSlice.stiffDistanceRangeX = verticalSlice.sliceRangeX(verticalSlice.stiffPlotRangeInd); 

end

% For both cases
time.tRange = 1:length(time.vec);

%-------------------------------------------------------------------------%
% Normalised strain rate

% Approximate fracture time for normalised strain rate calculation. 
clear fracture
fracture.locFrame = 90;

% Calculate the normalised strain rate up to the estimated fracture frame -Tk/2.
strainRate.tMaxInd = fracture.locFrame - floor(smoothingOpts.FFTemporalKernal(1)/2);

%-------------------------------------------------------------------------%
% Save path for all post-processed images and variables

imageSavePath = [imagePath,'PostProcessedImages'];                           
if ~exist(imageSavePath,'dir')
    mkdir(imageSavePath); %
end

% End of preamble
%\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

%% -----------------------------------------------------------------------%
% Plot full stress-strain curves
% ------------------------------------------------------------------------%
% Before the shear ss-curve chord fitting, plot the ss-curves. 

if strcmp('orthotropicAngle',globalOpts.matModel)
    
    % Single
    % Transverse stress vs. strain (22)
    func_plotStressStrainCurveSingle(strainSlice1.avg22, stressSlice1.avg22,...
        'Slice1_SSCurve_s22', imageSavePath, angledSlice1);

    % Shear stress vs. strain (12)
    func_plotStressStrainCurveSingle(strainSlice1.avg12, stressSlice1.avg12,...
        'Slice1_SSCurve_s12', imageSavePath, angledSlice1);
    func_plotStressStrainCurveSingle(strainSlice2.avg12, stressSlice2.avg12,...
        'Slice2_SSCurve_s12', imageSavePath, angledSlice2);

    % Subplots
    % Transverse stress vs. strain (22)
    func_plotStressStrainCurveSubplot(strainSlice1.avg22, stressSlice1.avg22,...
        'Slice1_SSCurve_full_subplot_s22',imageSavePath,angledSlice1,time,pos, material);

    % Shear stress vs. strain (12)
    func_plotStressStrainCurveSubplot(strainSlice1.avg12, stressSlice1.avg12,...
        'Slice1_SSCurve_full_subplot_s12',imageSavePath,angledSlice1,time,pos,material);
    func_plotStressStrainCurveSubplot(strainSlice2.avg12, stressSlice2.avg12,...
        'Slice2_SSCurve_full_subplot_s12',imageSavePath,angledSlice2,time,pos,material);

end

%% -----------------------------------------------------------------------%
% Strain Thresholding - Manual
%-------------------------------------------------------------------------%
% Based on manual input of min and max strain thresholds.    

% ------------------------------------------------------------------------%
% Determine the fitFrameRange 

if strcmp('orthotropicAngle',globalOpts.matModel)

    % epsilon_22 (Note: end index based on stress, as strains sometimes cause issue due to excessive noise)
    [angledSlice1.fitFrameRange_e22] = func_fitFrameRangeManual(strainSlice1.avg22, stressSlice1.avg22, ...
        manualFitParams.minThreshold_22, manualFitParams.maxThreshold_22, angledSlice1, time);

    % epsilon_12 Slice 1 - (Note: Both start and end indexes based on strain)
    [angledSlice1.fitFrameRange_e12] =  func_fitFrameRangeManual(strainSlice1.avg12, strainSlice1.avg12,...
        manualFitParams.minThreshold_12, manualFitParams.maxThreshold_12, angledSlice1, time);

    % epsilon_12 Slice 2 
    [angledSlice2.fitFrameRange_e12] =  func_fitFrameRangeManual(strainSlice2.avg12, strainSlice2.avg12,...
        manualFitParams.minThreshold_12, manualFitParams.maxThreshold_12, angledSlice2, time);
    
elseif strcmp('orthotropicReduced',globalOpts.matModel)


    %% ------------------------------------------------------------------------%
    % Fit frame range
    % Find the min and max frames for the strain thresholds, for each verticalSlice.
    % Manual min selection, comp only
    % ------------------------------------------------------------------------%

    % e22
    [verticalSlice.fitFrameRange_exx] = func_strainCompOnly(strain.xAvg, stress.xAvg,...
        verticalSlice.minStrainThreshold_xx, verticalSlice, time);

    % e12
    [verticalSlice.fitFrameRange_ess] = func_strainCompOnly(strain.sAvg, stress.sAvg,...
        verticalSlice.minStrainThreshold_ss, verticalSlice, time);  
    
end

% ------------------------------------------------------------------------%
% Fit the stress-strain curves

if strcmp('orthotropicAngle',globalOpts.matModel)

    % Stiffness Identification - Transverse
    [angledSlice1.ssFitOverLVal_E22, angledSlice1.ssFitOverLCoeffs_E22, angledSlice1.identStiffThreshOverL_E22] = func_fitStressStrainOverSpecLength(...
        strainSlice1.avg22, stressSlice1.avg22, angledSlice1.fitFrameRange_e22, angledSlice1.xMaxInd);

    % Stiffness Identification - Shear (to 10 me) - Slice 1
    [angledSlice1.ssFitOverLVal_G12, angledSlice1.ssFitOverLCoeffs_G12, angledSlice1.identStiffThreshOverL_G12] = func_fitStressStrainOverSpecLength(...
        strainSlice1.avg12, stressSlice1.avg12, angledSlice1.fitFrameRange_e12, angledSlice1.xMaxInd);

    % Stiffness Identification - Shear (to 10 me) - Slice 2
    [angledSlice2.ssFitOverLVal_G12, angledSlice2.ssFitOverLCoeffs_G12, angledSlice2.identStiffThreshOverL_G12] = func_fitStressStrainOverSpecLength(...
        strainSlice2.avg12, stressSlice2.avg12, angledSlice2.fitFrameRange_e12, angledSlice2.xMaxInd);

elseif strcmp('orthotropicReduced',globalOpts.matModel)

    [verticalSlice.ssFitOverLVal_Exx, verticalSlice.ssFitOverLCoeffs_Exx, verticalSlice.identStiffThreshOverL_Exx] = func_fitStressStrainOverSpecLength(...
    strain.xAvg, stress.xAvg, verticalSlice.fitFrameRange_exx, verticalSlice.xMaxInd);

    [verticalSlice.ssFitOverLVal_Gxy, verticalSlice.ssFitOverLCoeffs_Gxy, verticalSlice.identStiffThreshOverL_Gxy] = func_fitStressStrainOverSpecLength(...
    strain.sAvg, stress.sAvg, verticalSlice.fitFrameRange_ess, verticalSlice.xMaxInd);

end

%-------------------------------------------------------------------------%
% Plot - SS curves with fit

if strcmp('orthotropicAngle',globalOpts.matModel)
    
    % Transverse stress vs. strain (22) All slices
    func_plotFullandFittedStressStrainCurveSubplotVer2(strainSlice1.avg22, stressSlice1.avg22, angledSlice1.fitFrameRange_e22,...
        angledSlice1, angledSlice1.ssFitOverLVal_E22, pos, time, 'Slice1_SSCurve_fullandFitted_subplot_s22', imageSavePath, material);

    % Shear stress vs. strain (12) (to 10 me) - Slice 1
    func_plotFullandFittedStressStrainCurveSubplotVer2(strainSlice1.avg12, stressSlice1.avg12, angledSlice1.fitFrameRange_e12,...
        angledSlice1, angledSlice1.ssFitOverLVal_G12, pos, time,'Slice1_SSCurve_fullandFitted_to10me_subplot_s12', imageSavePath, material);

    % Shear stress vs. strain (12) (to 10 me) - Slice 2
    func_plotFullandFittedStressStrainCurveSubplotVer2(strainSlice2.avg12, stressSlice2.avg12, angledSlice2.fitFrameRange_e12,...
        angledSlice2, angledSlice2.ssFitOverLVal_G12, pos, time,'Slice2_SSCurve_fullandFitted_to10me_subplot_s12', imageSavePath, material);

elseif strcmp('orthotropicReduced',globalOpts.matModel)

    func_plotFullandFittedStressStrainCurveSubplotVer2(strain.xAvg, stress.xAvg, verticalSlice.fitFrameRange_exx,...
        verticalSlice, verticalSlice.ssFitOverLVal_Exx, pos, time, 'SSCurve_fullandFitted_subplot_sxx', imageSavePath, material);
    
    func_plotFullandFittedStressStrainCurveSubplotVer2(strain.sAvg, stress.sAvg, verticalSlice.fitFrameRange_ess,...
        verticalSlice, verticalSlice.ssFitOverLVal_Gxy, pos, time, 'SSCurve_fullandFitted_subplot_sxy', imageSavePath, material);
    
end

%% ------------------------------------------------------------------------%
% (1) Strain Thresholding - Chord
% ------------------------------------------------------------------------%
% Fit the shear stress-strain curves over the elastic portion of the curve
% using progressive Chord Modulus fits to determine the elastic limit strain. 
% For a progressive linear fit, use dataComponent = 'Slice1_progressiveModulus';

if strcmp('on',switches.chordModulusFitting)

% Fit the chord modulus progressively

dataComponent = 'Slice1_chordModulus';
[chordFitSlice1.fitFrameRange, chordFitSlice1.progressiveModulus, chordFitSlice1.meanProgressiveModulus,...
    chordFitSlice1.fitRangeLinearRegion, chordFitSlice1.linRangeEndInd] = func_progressiveChordModulusFit_ver4...
    (strainSlice1.avg12, stressSlice1.avg12, angledSlice1, time, chordFitParams, dataComponent, imageSavePath);

dataComponent = 'Slice2_chordModulus'; 
[chordFitSlice2.fitFrameRange, chordFitSlice2.progressiveModulus, chordFitSlice2.meanProgressiveModulus,...
    chordFitSlice2.fitRangeLinearRegion, chordFitSlice2.linRangeEndInd] = func_progressiveChordModulusFit_ver4...
    (strainSlice2.avg12, stressSlice2.avg12, angledSlice2, time, chordFitParams, dataComponent, imageSavePath);

% -------------------------------------------------------------------------
% Plot
% Chord Modulus vs time - Slice 1
func_plotProgressiveChordModulus(chordFitSlice1.xInd, chordFitSlice1.fitFrameRange, chordFitSlice1.progressiveModulus,...
    chordFitSlice1.meanProgressiveModulus, strainSlice1, imageSavePath, dataComponent, angledSlice1,...
    chordFitSlice1.linRangeEndInd, time);

% Chord Modulus vs time - Slice 2
func_plotProgressiveChordModulus(chordFitSlice2.xInd, chordFitSlice2.fitFrameRange, chordFitSlice2.progressiveModulus,...
    chordFitSlice2.meanProgressiveModulus, strainSlice2, imageSavePath, dataComponent, angledSlice2,...
    chordFitSlice2.linRangeEndInd, time);

% Diagnostic Plot
% Single - full ss-curve with linear region in red.
% Shear stress vs. strain (12) - Slice 1
func_plotSSCurveFullandLinearRegion(strainSlice1.avg12, stressSlice1.avg12,...
    'Slice1_SSCurveFullandLinearRegion_s12', imageSavePath, chordFitSlice1);

% Shear stress vs. strain (12) - Slice 2
func_plotSSCurveFullandLinearRegion(strainSlice2.avg12, stressSlice2.avg12,...
    'Slice2_SSCurveFullandLinearRegion_s12', imageSavePath, chordFitSlice2);

% -------------------------------------------------------------------------
% Fit (Chord)- all slices.
[chordFitSlice1.ssChordFitOverLVal_G12, chordFitSlice1.ssChordFitOverLCoeffs_G12, chordFitSlice1.identChordStiffThreshOverL_G12] =...
    func_fitStressStrainOverSpecLength(strainSlice1.avg12, stressSlice1.avg12, chordFitSlice1.fitRangeLinearRegion, angledSlice1.xMaxInd);

[chordFitSlice2.ssChordFitOverLVal_G12, chordFitSlice2.ssChordFitOverLCoeffs_G12, chordFitSlice2.identChordStiffThreshOverL_G12] =...
    func_fitStressStrainOverSpecLength(strainSlice2.avg12, stressSlice2.avg12, chordFitSlice2.fitRangeLinearRegion, angledSlice2.xMaxInd);

% Plot 
% Subplot - Full & Thresholded (from Chord Modulus) at 3-6 slices.
func_plotFullandFittedStressStrainCurveSubplotVer2(strainSlice1.avg12, stressSlice1.avg12, chordFitSlice1.fitRangeLinearRegion,...
    angledSlice1, chordFitSlice1.ssChordFitOverLVal_G12, pos, time,'Slice1_SSCurve_fullandFittedLinearRegion_subplot_s12', imageSavePath, material);

func_plotFullandFittedStressStrainCurveSubplotVer2(strainSlice2.avg12, stressSlice2.avg12, chordFitSlice2.fitRangeLinearRegion,...
    angledSlice2, chordFitSlice2.ssChordFitOverLVal_G12, pos, time,'Slice2_SSCurve_fullandFittedLinearRegion_subplot_s12', imageSavePath, material);

else
    fprintf('Chord modulus fitting not completed. \n')
end

%% -----------------------------------------------------------------------%
% Plot Modulus against length for all slices, over threshold.
% ------------------------------------------------------------------------%

if strcmp('orthotropicAngle',globalOpts.matModel)

    % Transverse E22 (Manual thresholding) - Over compression portion only
    dataComponent = 'Slice1_E22';
    [angledSlice1.identStiffAvg_E22] = func_plotStiffnessVsL(angledSlice1.identStiffThreshOverL_E22,...
        angledSlice1, pos, dataComponent, imageSavePath, material.Eyy, plotProperties);

    % Stiffness values identified from ss-curves with linear limit obtained from progressive chord fitting function
    dataComponent = 'Slice1_G12';
    [angledSlice1.identStiffAvg_G12] = func_plotStiffnessVsL(chordFitSlice1.identChordStiffThreshOverL_G12,...
        angledSlice1, pos, dataComponent, imageSavePath, material.Gxy, plotProperties);

    dataComponent = 'Slice2_G12';
    [angledSlice2.identStiffAvg_G12] = func_plotStiffnessVsL(chordFitSlice2.identChordStiffThreshOverL_G12,...
        angledSlice2, pos, dataComponent, imageSavePath, material.Gxy, plotProperties);

    % Shear G12 (manual limit strain fit)
    % Stiffness values identified from ss curves with linear limit determined by chord fit
    dataComponent = 'Slice1_G12_to10me';
    [angledSlice1.identStiffAvg_G12_to10me] = func_plotStiffnessVsL(angledSlice1.identStiffThreshOverL_G12,...
        angledSlice1, pos, dataComponent, imageSavePath, material.Gxy, plotProperties);

    dataComponent = 'Slice2_G12_to10me';
    [angledSlice2.identStiffAvg_G12_to10me] = func_plotStiffnessVsL(angledSlice2.identStiffThreshOverL_G12,...
        angledSlice2, pos, dataComponent, imageSavePath, material.Gxy, plotProperties);

elseif strcmp('orthotropicReduced',globalOpts.matModel)
    
    % Transverse E22 (Manual thresholding) - Over compressive loading
    dataComponent = 'Exx';
    [verticalSlice.identStiffAvg_Exx] = func_plotStiffnessVsL(verticalSlice.identStiffThreshOverL_Exx,...
        verticalSlice, pos, dataComponent, imageSavePath, material.Eyy, plotProperties);

    % Shear Gxy -  Over compression portion only
    dataComponent = 'Gxy';
    [verticalSlice.identStiffAvg_Gxy] = func_plotStiffnessVsL(verticalSlice.identStiffThreshOverL_Gxy,...
        verticalSlice, pos, dataComponent, imageSavePath, material.Gxy, plotProperties);
    
end

%% -----------------------------------------------------------------------%
% (2) Q12 (or e11) correction to E22 identification
% ------------------------------------------------------------------------%

if strcmp('on',switches.Q12Correction)

    % UD45 case
    if strcmp('orthotropicAngle',globalOpts.matModel)
        
        % Calculate Q12 - from Q12 = v12*E22/1-v12v21
        % Note: nu12 = nuyx due to (x,y) = (2,1) for UD90 case
        angledSlice1.identStiffThreshOverL_Q12 = (material.nuxy*angledSlice1.identStiffThreshOverL_E22)/...
            (1-material.nuyx*material.nuxy);

        % Translate
        angledSlice1.identStiffThreshOverL_Q12 = angledSlice1.identStiffThreshOverL_Q12';

        % Corrected stress (Experimental e11 data)
        stressSlice1.avg22Corr = stressSlice1.avg22 - angledSlice1.identStiffThreshOverL_Q12.*strainSlice1.avg11;

        % Corrected stress (FE e11 data)  *Note: both should be more negative because they are in compression 
        FE.stressSlice1.avg22Corr = stressSlice1.avg22 - angledSlice1.identStiffThreshOverL_Q12.*FEStrainSlice1Avg11;

        % Fit (Experimental data)
        [angledSlice1.ssFitOverLVal_Q22Corr, angledSlice1.ssFitOverLCoeffs_Q22Corr, angledSlice1.identStiffThreshOverL_Q22Corr] =...
            func_fitStressStrainOverSpecLength(strainSlice1.avg22, stressSlice1.avg22Corr, angledSlice1.fitFrameRange_e22, angledSlice1.xMaxInd);

        % Fit (FE data)
        [angledSlice1.ssFitOverLVal_Q22CorrFE, angledSlice1.ssFitOverLCoeffs_Q22CorrFE, angledSlice1.identStiffThreshOverL_Q22CorrFE] =...
            func_fitStressStrainOverSpecLength(strainSlice1.avg22, FE.stressSlice1.avg22Corr, angledSlice1.fitFrameRange_e22, angledSlice1.xMaxInd);

        % Convert stiffness components to Youngs Modulus components (E22 from Q22)
        angledSlice1.identStiffThreshOverL_E22Corr = angledSlice1.identStiffThreshOverL_Q22Corr*(1-(material.nuxy*material.nuyx));
        angledSlice1.identStiffThreshOverL_E22CorrFE = angledSlice1.identStiffThreshOverL_Q22CorrFE*(1-(material.nuxy*material.nuyx));

        %--------------------------------------------------------------------------
        % Plot ss-curves
        func_plotFullandFittedStressStrainCurveSubplotOffAxisWithFE(strainSlice1, stressSlice1, FE, ...
            angledSlice1, pos, time,'SSCurve_fullandFitted_subplot_sxx_Q12Corr', imageSavePath, switches);

        % Plot stiffness
        dataComponent = 'E22Corr';
        [~, angledSlice1.identStiffAvg_E22Corr, angledSlice1.identStiffAvg_E22CorrFE] =...
            func_plotStiffnessVsLQ12Corr(angledSlice1, angledSlice1, pos, dataComponent, imageSavePath, material, globalOpts, plotProperties);
        %--------------------------------------------------------------------------
    
    % UD90 case
    elseif strcmp('orthotropicReduced',globalOpts.matModel)
        
        % Calculate Qxy - from Q12 = v12*E22/1-v12v21. Note for the UD90 case: nu12 = nuyx due to (x,y) = (2,1).
        verticalSlice.identStiffThreshOverL_Qxy = (material.nuyx*verticalSlice.identStiffThreshOverL_Exx)/...
            (1-material.nuyx*material.nuxy);

        % Translate
        verticalSlice.identStiffThreshOverL_Qxy = verticalSlice.identStiffThreshOverL_Qxy';

        % Corrected stress (Experimental data)  *Note: both more negative because in compression
        stress.xAvgCorr = stress.xAvg - verticalSlice.identStiffThreshOverL_Qxy.*strain.yAvg;

        % Corrected stress (FE data)
        FE.stress.xAvgCorr = stress.xAvg - verticalSlice.identStiffThreshOverL_Qxy.*FEStrainyAvg;

        % Fit (Experimental data)
        [verticalSlice.ssFitOverLVal_QxxCorr, verticalSlice.ssFitOverLCoeffs_QxxCorr, verticalSlice.identStiffThreshOverL_QxxCorr] =...
            func_fitStressStrainOverSpecLength(strain.xAvg, stress.xAvgCorr, verticalSlice.fitFrameRange_exx, verticalSlice.xMaxInd);

        % Fit (FE data)
        [verticalSlice.ssFitOverLVal_QxxCorrFE, verticalSlice.ssFitOverLCoeffs_QxxCorrFE, verticalSlice.identStiffThreshOverL_QxxCorrFE] =...
            func_fitStressStrainOverSpecLength(strain.xAvg, FE.stress.xAvgCorr, verticalSlice.fitFrameRange_exx, verticalSlice.xMaxInd);

        % Convert stiffness components to Youngs Modulus components (E22 from Q22)
        verticalSlice.identStiffThreshOverL_ExxCorr = verticalSlice.identStiffThreshOverL_QxxCorr*(1-(material.nuxy*material.nuyx));
        verticalSlice.identStiffThreshOverL_ExxCorrFE = verticalSlice.identStiffThreshOverL_QxxCorrFE*(1-(material.nuxy*material.nuyx));

        %--------------------------------------------------------------------------

        % Plot ss-curves
        func_plotFullandFittedStressStrainCurveSubplotTransverseWithFE(strain, stress, FE, ...
            verticalSlice, pos, time,'SSCurve_fullandFitted_subplot_sxx_Q12Corr', imageSavePath,...
            fracture, switches, plotProperties);

        % Plot
        dataComponent = 'ExxCorr';
        [~, verticalSlice.identStiffAvg_ExxCorr, verticalSlice.identStiffAvg_ExxCorrFE] = ...
            func_plotStiffnessVsLQ12Corr(verticalSlice, verticalSlice, pos, dataComponent, imageSavePath, material, globalOpts, plotProperties);
        
        %--------------------------------------------------------------------------
        
    else
        fprintf('Q12 correction of E22 not completed. \n')
    
    end
    
end

%% -----------------------------------------------------------------------%
% (3) Strain-normalised strain rate
% ------------------------------------------------------------------------%

% Interpolate the rotated strain rate fields to the slices for each timestep.
if strcmp('orthotropicAngle',globalOpts.matModel)
    
    % Slice 1, along the fibres: interpolate strain rate to the slice --------------
    fprintf('Interpolate strain rate values to slice 1 \n')
        [strainRateSlice1.avg11, strainRateSlice1.avg22, strainRateSlice1.avg12,...
            strainRateSlice1.field11, strainRateSlice1.field22, strainRateSlice1.field12] = func_interpolateFieldsToSlice...
            (time, pos, strainRate, angledSlice1);

    % Slice 2, transverse to the fibres: interpolate strain rate to the slice ------
    fprintf('Interpolate strain rate values to slice 2 \n')
        [strainRateSlice2.avg11, strainRateSlice2.avg22, strainRateSlice2.avg12,...
            strainRateSlice2.field11, strainRateSlice2.field22, strainRateSlice2.field12] = func_interpolateFieldsToSlice...
            (time, pos, strainRate, angledSlice2);
end

% Strain rate assessments (see headings for descriptions)
if strcmp('on',switches.normalisedStrainRate)

    if strcmp('orthotropicAngle',globalOpts.matModel)
        
        % Minimum average strain rate (maximum absolute) over the range of slices, over time. 
        dataComponent = 'SR_MinAvgVsTime_22';
        [strainRateSlice1.avg22PeakToFracFrame] = func_plotMinAvgStrainRateOverTime(strainRateSlice1.avg22, time, strainRate.tMaxInd,...
            dataComponent, imageSavePath);
        dataComponent = 'SR_MinAvgVsTime_Slice1_12';
        [strainRateSlice1.avg12AvgPeakToFracFrame] = func_plotMinAvgStrainRateOverTime(strainRateSlice1.avg12, time, strainRate.tMaxInd,...
            dataComponent, imageSavePath);
        dataComponent = 'SR_MinAvgVsTime_Slice2_12';
        [strainRateSlice2.avg12AvgPeakToFracFrame] = func_plotMinAvgStrainRateOverTime(strainRateSlice2.avg12, time, strainRate.tMaxInd,...
            dataComponent, imageSavePath);

        % Minimum average strain rate over the fit frame range, per verticalSlice.
        dataComponent = 'SR_MinAvgVsL_22';
        [strainRateSlice1.avg22PeakMeanOverFitRange] = func_plotMinAvgStrainRateVsL(strainRateSlice1.avg22, angledSlice1.sliceRangeX, ...
            angledSlice1.fitFrameRange_e22, dataComponent, imageSavePath);
        dataComponent = 'SR_MinAvgVsL_Slice1_12';
        [strainRateSlice1.avg12PeakMeanOverFitRange] = func_plotMinAvgStrainRateVsL(strainRateSlice1.avg12, angledSlice1.sliceRangeX, ...
            chordFitSlice1.fitRangeLinearRegion, dataComponent, imageSavePath);
        dataComponent = 'SR_MinAvgVsL_Slice2_12';
        [strainRateSlice2.avg12PeakMeanOverFitRange] = func_plotMinAvgStrainRateVsL(strainRateSlice2.avg12, angledSlice2.sliceRangeX, ...
            chordFitSlice2.fitRangeLinearRegion, dataComponent, imageSavePath);

        % ------------------------------------------------------------------------%
        % Crop fields
        % strains - in material coordinates over the whole field (i.e. not along slices)
        [strain.mat22Cropped] = func_cropField(strain.mat22, grid, smoothingOpts, strainRate.tMaxInd);
        [strain.mat12Cropped] = func_cropField(strain.mat12, grid, smoothingOpts, strainRate.tMaxInd);

        % strainRates
        [strainRate.mat22Cropped] = func_cropField(strainRate.mat22, grid, smoothingOpts, strainRate.tMaxInd);
        [strainRate.mat12Cropped] = func_cropField(strainRate.mat12, grid, smoothingOpts, strainRate.tMaxInd);

        % Normalised strain rate (all points)  (Output in modulus paper)
        [strainRate.mat22StrainNormalised] = func_normalisedStrainRate(strain.mat22Cropped, strainRate.mat22Cropped);
        [strainRate.mat12StrainNormalised] = func_normalisedStrainRate(strain.mat12Cropped, strainRate.mat12Cropped);

        %--------------------------------------------------------------------------
        % Calculate the strain-normalised strain rate (at each slice) over the modulus fit range
        [strainRateSlice1.avg22StrainNormalisedPerSlice] = func_normalisedStrainRateVsL(angledSlice1.fitFrameRange_e22,...
            angledSlice1.sliceRangeX, strainSlice1.field22, strainRateSlice1.field22);
        [strainRateSlice1.avg12StrainNormalisedPerSlice] = func_normalisedStrainRateVsL(chordFitSlice1.fitRangeLinearRegion,...
            angledSlice1.sliceRangeX, strainSlice1.field12, strainRateSlice1.field12);
        [strainRateSlice2.avg12StrainNormalisedPerSlice] = func_normalisedStrainRateVsL(chordFitSlice2.fitRangeLinearRegion,...
            angledSlice2.sliceRangeX, strainSlice2.field12, strainRateSlice2.field12);

        % Plot the strain-normalised strain rate vs. L 
        dataComponent = 'SR_SNorm_VsL_Slice1_22';
        [strainRateSlice1.avg22StrainNormalisedAvg, ~] = func_plotStrainRateVsL(strainRateSlice1.avg22StrainNormalisedPerSlice,...
            angledSlice1, dataComponent, imageSavePath, strainRate.mat22StrainNormalised, globalOpts);
        dataComponent = 'SR_SNorm_VsL_Slice1_12';
        [strainRateSlice1.avg12StrainNormalisedAvg, ~] = func_plotStrainRateVsL(strainRateSlice1.avg12StrainNormalisedPerSlice,...
            angledSlice1, dataComponent, imageSavePath, strainRate.mat12StrainNormalised, globalOpts);
        dataComponent = 'SR_SNorm_VsL_Slice2_12';
        [strainRateSlice2.avg12StrainNormalisedAvg, ~] = func_plotStrainRateVsL(strainRateSlice2.avg12StrainNormalisedPerSlice,...
            angledSlice2, dataComponent, imageSavePath, strainRate.mat12StrainNormalised, globalOpts);
        
        
    elseif strcmp('orthotropicReduced',globalOpts.matModel)
        
        % Minimum average strain rate (maximum absolute) over the range of slices, over time. 
        dataComponent = 'SR_MinAvgVsTime_22';
        [strainRateVerticalSlice.xAvgPeakToFracFrame] = func_plotMinAvgStrainRateOverTime(strainRate.xAvg, time, strainRate.tMaxInd,...
            dataComponent, imageSavePath);
        dataComponent = 'SR_MinAvgVsTime_12';
        [strainRateVerticalSlice.sAvgPeakToFracFrame] = func_plotMinAvgStrainRateOverTime(strainRate.sAvg, time, strainRate.tMaxInd,...
            dataComponent, imageSavePath);
              
        % Minimum average strain rate over the fit frame range, per verticalSlice
        dataComponent = 'SR_MinAvgVsL_22';
        [strainRateVerticalSlice.xAvgPeakMeanOverFitRange] = func_plotMinAvgStrainRateVsL(strainRate.xAvg, ssCurveIdentPlotOpts.indLRange*pos.xStep, ...
            identStiffSG.QxxFitFrameRange, dataComponent, imageSavePath);
        dataComponent = 'SR_MinAvgVsL_12';
        [strainRateVerticalSlice.sAvgPeakMeanOverFitRange] = func_plotMinAvgStrainRateVsL(strainRate.sAvg, ssCurveIdentPlotOpts.indLRange*pos.xStep, ...
            identStiffSG.QssFitFrameRange, dataComponent, imageSavePath);   
        
        % ------------------------------------------------------------------------%
        % Crop fields
        % strains - in material coordinates over the whole field (i.e. not along slices)
        [strain.xCropped] = func_cropField(strain.x, grid, smoothingOpts, strainRate.tMaxInd);
        [strain.sCropped] = func_cropField(strain.s, grid, smoothingOpts, strainRate.tMaxInd);

        % strainRates
        [strainRate.xCropped] = func_cropField(strainRate.x, grid, smoothingOpts, strainRate.tMaxInd);
        [strainRate.sCropped] = func_cropField(strainRate.s, grid, smoothingOpts, strainRate.tMaxInd);

        % Normalised strain rate (all points)
        [strainRate.xStrainNormalised] = func_normalisedStrainRate(strain.xCropped, strainRate.xCropped);
        [strainRate.sStrainNormalised] = func_normalisedStrainRate(strain.sCropped, strainRate.sCropped);

        %--------------------------------------------------------------------------
        % Calculate the strain-normalised strain rate (at each slice) over the modulus fit range
        [strainRateVerticalSlice.xAvgStrainNormalisedPerSlice] = func_normalisedStrainRateVsL(identStiffSG.QxxFitFrameRange,...
            ssCurveIdentPlotOpts.indLRange*pos.xStep, strain.x, strainRate.x);
        [strainRateVerticalSlice.sAvgStrainNormalisedPerSlice] = func_normalisedStrainRateVsL(identStiffSG.QssFitFrameRange,...
            ssCurveIdentPlotOpts.indLRange*pos.xStep, strain.s, strainRate.s);

        % Plot the strain-normalised strain rate vs. L (Note output mid50% average for UD90 case)
        dataComponent = 'SR_SNorm_VsL_Slice1_22';
        [~, strainRateVerticalSlice.xStrainNormalisedAvg] = func_plotStrainRateVsL(strainRateVerticalSlice.xAvgStrainNormalisedPerSlice, ...
            verticalSlice, dataComponent, imageSavePath, strainRate.xStrainNormalised, globalOpts);
        dataComponent = 'SR_SNorm_VsL_Slice1_12';
        [~, strainRateVerticalSlice.sStrainNormalisedAvg] = func_plotStrainRateVsL(strainRateVerticalSlice.sAvgStrainNormalisedPerSlice, ...
            verticalSlice, dataComponent, imageSavePath, strainRate.sStrainNormalised, globalOpts);
        
    else
            fprintf('Normalised strain rate assessment not completed. \n');
    end
            
end

%% -----------------------------------------------------------------------%
% Saving
% ------------------------------------------------------------------------%

% Save all the data
if strcmp('on',switches.saveAllData)

    fprintf('Saving all processed data...\n')
    
    % Save filename
    saveFile = 'AllPost_ProcessedData.mat';
    
    %Save
    save([savePath,saveFile]);

end

% Save the modulus vs. position data
if strcmp('on',switches.saveStiffnessData)
    fprintf('Saving modulus data for multiple modulus assessment. \n')
    
    % Specify where to save the Modulus vs. length data for this test, for later use in plot all tests 
    savePathModulusVsL = imagePath;
       
    if strcmp('orthotropicAngle',globalOpts.matModel)
               
        % Save
        save([savePathModulusVsL,test.description,'.mat'], 'angledSlice1', 'angledSlice2', 'globalOpts',...
            'chordFitSlice1', 'chordFitSlice2','pos');
    
    elseif strcmp('orthotropicReduced',globalOpts.matModel)

        % Save
        save([savePathModulusVsL,test.description,'.mat'], 'verticalSlice', 'pos'); 
    
    end
end

% ------------------------------------------------------------------------%
% END
% ------------------------------------------------------------------------%