function [identQVFOpt1,identQVFOpt2] = func_VFDynOptimisedIsotropic(virtualField,pos,strain,accel,material)
% choose the number of elements in the x and y direction
m = virtualField.meshX; % number of nodes in the X direction
n = virtualField.meshY; % number of nodes in the Y direction
L = max(pos.x) + min(pos.x);
w = max(pos.y) + min(pos.y);

[rows,cols,frames] = size(strain.x); % size of the array of data points

n_nodes = (m+1)*(n+1); % number of nodes
n_points = rows*cols; % number of measurement points

L_el = L/m; % length of elements
w_el = w/n; % width of elements

%reshape coordinate and strain matrices into vectors
X1 = reshape(pos.xGrid, n_points,1);
X2 = reshape(flipud(pos.yGrid), n_points,1);

%% Build Up Virtual Fields
% assign an element to each measurement point
iii = floor(X1*m/L)+1; % elements in the x1 direction
jjj = floor(X2*n/w)+1; % elements in the x2 direction

%define parametric coordinates
xsi1 = 2*X1/L_el - 2*iii +1; % along x1 direction
xsi2 = 2*X2/w_el - 2*jjj +1; % along x2 direction

% calculate virtual displacements
u0 = zeros(n_points, 1);
u1elem = 0.25*[(1-xsi1).*(1-xsi2) u0 (1+xsi1).*(1-xsi2) u0 (1+xsi1).*(1+xsi2) u0 (1-xsi1).*(1+xsi2) u0]; % first row of N matrix (see pg. 51)
u2elem = 0.25*[u0 (1-xsi1).*(1-xsi2) u0 (1+xsi1).*(1-xsi2) u0 (1+xsi1).*(1+xsi2) u0 (1-xsi1).*(1+xsi2)]; % second row of N matrix (see pg. 51)
    
% calculate virtual strains
v0 = zeros(n_points, 1);
Eps1elem = [-(1-xsi2) v0 (1-xsi2) v0 (1+xsi2) v0 -(1+xsi2) v0]*1/2/L_el;
Eps2elem = [v0 -(1-xsi1) v0 -(1+xsi1) v0 (1+xsi1) v0 (1-xsi1)]*1/2/w_el;
Eps6elem = [-(1-xsi1)/w_el -(1-xsi2)/L_el -(1+xsi1)/w_el (1-xsi2)/L_el (1+xsi1)/w_el (1+xsi2)/L_el (1-xsi1)/w_el -(1+xsi2)/L_el]*1/2;

%% Consruct the matrices in the form of the final optimization matrix 
% Bij will be used for the speciality condition
% Hij will be used to compute the Hessian matrix
B11 = zeros(1,2*n_nodes);
B12 = B11;
H11 = zeros(2*n_nodes, 2*n_nodes);
H22 = H11;
H12 = H11;
H66 = H11;

% Define the nodes
n1 = (iii-1)*(n+1) + jjj; % first node
n2 = iii*(n+1) + jjj; % second node
n3 = iii*(n+1) + jjj +1; % third node
n4 = (iii-1)*(n+1) + jjj +1; % fourth node

% matrix containing the degrees of freedom affected by each data point
assemble = [2*n1-1 2*n1 2*n2-1 2*n2 2*n3-1 2*n3 2*n4-1 2*n4];
%% Define Virtual Boundary Conditions
Aconst = zeros((n+1), 2*n_nodes); %there are 2(n+1) boundary conditions and 2 degrees of freedom per node
for i = 1:(n+1) % u1(x1 = L) = 0 (n+1) conditions
    Aconst(i,2*n_nodes-2*(n+1)+2*i-1) = 1;
end

%% Construct Z vector of zeros except for conditions of speciality
% Za is the vector used to find the virtual field to identify Q11
% Zb is the vector used to find the virtual field to identify Q12
Za = zeros(1, 2*n_nodes + size(Aconst,1)+1); % 32 columns plus 16 for describing the boundary conditions
Zb = Za;

Za(2*n_nodes + size(Aconst,1)+1 : 2*n_nodes + size(Aconst,1)+2) = [1 0]; % special VF condition for Q11
Zb(2*n_nodes + size(Aconst,1)+1 : 2*n_nodes + size(Aconst,1)+2) = [0 1]; % special VF condition for Q12

%% Compute stiffness for each frame using VFs with minimal noise sensitivity
% allocate memory for vector holding stiffness values as a function of time

for j = 1:frames
    warning('off','all')
    % reshape matrices for each frame
    Eps1 = reshape(squeeze(strain.x(:,:,j)),n_points,1);
    Eps2 = reshape(squeeze(strain.y(:,:,j)),n_points,1);
    Eps6 = reshape(squeeze(strain.s(:,:,j)),n_points,1);
    
    A1 = reshape(squeeze(accel.x(:,:,j)),n_points,1);
    A2 = reshape(squeeze(accel.y(:,:,j)),n_points,1);

    for k = 1:n_points
        assemble1 = assemble(k,:);
        B11(assemble1) = B11(assemble1) + (Eps1(k)*Eps1elem(k,:) + Eps2(k)*Eps2elem(k,:) + 0.5*Eps6(k)*Eps6elem(k,:))*(L*w/n_points); % multiply actual strains by virtual strains
        B12(assemble1) = B12(assemble1) + (Eps1(k)*Eps2elem(k,:) + Eps2(k)*Eps1elem(k,:) - 0.5*Eps6(k)*Eps6elem(k,:))*(L*w/n_points);

        % assemble Hessian matrix (minimization of sensitivity to noise)
        H11(assemble1,assemble1) = H11(assemble1,assemble1) + Eps1elem(k,:)'*Eps1elem(k,:);
        H22(assemble1,assemble1) = H22(assemble1,assemble1) + Eps2elem(k,:)'*Eps2elem(k,:);
        H12(assemble1,assemble1) = H12(assemble1,assemble1) + Eps1elem(k,:)'*Eps2elem(k,:);
        H66(assemble1,assemble1) = H66(assemble1,assemble1) + Eps6elem(k,:)'*Eps6elem(k,:);
    end
    
    % build up A matrix with all constraints
    A = [Aconst; B11; B12];
    % speciality conditions
    B = zeros(size(A,1));

    %% Solving the optimization problem - same as for polynomials
    Q = [2 1]; % initial guesses for stiffness values - required since H relies on Qij

    n_iter = 20; % maximum iterations
    delta_lim = 0.001; % tolerance on optimization

    delta = 10; % starting with a tolerance larger than delta_lim
    i = 1;
    Qold = Q; % stiffnesses from previous iteration required to compute error

    while i<n_iter && delta> delta_lim
        % Hessian matrix
        H = (L*w/n_points)^2*((Q(1)^2 + Q(2)^2)*H11 + (Q(1)^2 + Q(2)^2)*H22 + (1/4*(Q(1)^2) + 1/4*(Q(2)^2) - 1/2*(Q(1)*Q(2)))*H66 + 4*Q(1)*Q(2)*H12);
   
        % NOTE: to avoid numerical "Warning: Matrix is close to singular or
        % badly scaled" matrix Opt can be scaled with the parameter corr.
        % It does not change the results of the optimization.
        % To avoid using, put corr = 1

        corr = max(max(A))/max(max(H)); % normalization coefficient
        %     corr = 1;
        OptM = [H*corr, A'*corr; A,B]; % matrix for virtual fields optimization

        %vector containing the polynomial coefficients for Q11 and Q12 the
        %Lagrange multipliers
        Ya = OptM\Za';
        Yb = OptM\Zb';

        %remove the lagrangian multipliers from the Y vectors because they are
        %of no interest
        Ya(2*n_nodes + 1: size(Ya)) = [];
        Yb(2*n_nodes + 1: size(Yb)) = [];

        for k = 1:n_points
            % virtual displacement field
            u1vv = zeros(1,2*n_nodes);
            u2vv = zeros(1,2*n_nodes);
            assemble1 = assemble(k,:);
            u1vv(assemble1) = u1elem(k,:);
            u2vv(assemble1) = u2elem(k,:);
            u1va(k) = u1vv*Ya; % for Q11
            u2va(k) = u2vv*Ya; % for Q11
            u1vb(k) = u1vv*Yb; % for Q12
            u2vb(k) = u2vv*Yb; % for Q12
        end
        %calculating Q11 from the first optimized virtual field
        Q(1) = -material.rho*(L*w/n_points)*(sum(u1va'.*A1) + sum(u2va'.*A2));
        % calculating Q11 from the first optimized virtual field
        Q(2) = -material.rho*(L*w/n_points)*(sum(u1vb'.*A1) + sum(u2vb'.*A2));

        % compute difference between the current and previous identified values
        delta = sum((Qold-Q).^2./Q.^2);

        Qold = Q; % store current parameters as old before the next iteration
        i = i+1; % increment the step
    end

    % Final results
    identQVFOpt1(1,j) = Q(1);
    identQVFOpt2(1,j) = Q(2);
    
    H = (L*w/n_points)^2*((Q(1)^2 + Q(2)^2)*H11 + (Q(1)^2 + Q(2)^2)*H22 + (1/4*(Q(1)^2) + 1/4*(Q(2)^2) - 1/2*(Q(1)*Q(2)))*H66 + 4*Q(1)*Q(2)*H12);
    
    B11 = zeros(1,2*n_nodes);
    B12 = zeros(1,2*n_nodes);
    u1va = zeros(1,n_points);
    u2va = zeros(1,n_points);
    u1vb = zeros(1,n_points);
    u2vb = zeros(1,n_points);
end
