function [ grid,pos,disp ] = func_gridMethodImageProcessing_ImageDef(refImagePath,...
refImageFile,grid,gridMethodOpts,noise)
% Author: Lloyd Fletcher
% PhotoDyn Group, University of Southampton
% Date: 10/8/2017
%
% Processes a sequence of grid images to obtain displacement fields
% This code uses the grid processing tool box that can be found at:
% www.thegridMethodOpts.net, developed by Grediac

%--------------------------------------------------------------------------
% 1) Load the Reference Image and Find All Images in Folder
% Get the repeated part of the string assuming underscore termination
[~,~,imageExt] = fileparts(refImageFile);
usLoc = find(refImageFile == '_');
if ~isempty(usLoc)
    startStr = refImageFile(1:usLoc(end));
else
    startStr = 'DefGrid_';
end

% Ask for the filename starter and new filename
resp = inputdlg({'Enter the repeating part of the image file string:'},...
             'Input repeating file string', 1, {startStr} );        
fstart = resp{1};

% Get the names of all image files with the same ext in the directory
refImageFileStruct = dir([refImagePath,fstart,'*',imageExt]);
refImageFileCell = {refImageFileStruct.name}';

% Sort the data files in numerical order
sortedFiles{1} = refImageFile;
for i = 2:length(refImageFileCell)
    checkStr1 = [fstart,num2str(i),imageExt];   
    
    nn = length(num2str(i));
    if nn==1
        num = ['00' num2str(i)];
    elseif nn==2
        num = ['0' num2str(i)];
    else
        num = num2str(i);
    end
    checkStr2 = [fstart,num,imageExt];
    
    for j = 2:length(refImageFileCell)       
        if strcmp(checkStr1,refImageFileCell{j}) || strcmp(checkStr2,refImageFileCell{j})
            sortedFiles{i} = refImageFileCell{j};
            break
        end
    end
end
refImageFileCell = sortedFiles;

% Load the reference image 
refImage = imread([refImagePath,refImageFile]);
refImage = double(refImage);

% Add noise to the image if required
if noise.addNoise
	refImage = func_addNoiseToImages(noise.pcNoise,refImage,...
		noise.bits,noise.convToUInt16);
end
    
%--------------------------------------------------------------------------
% 2) Mask Images
maskImages = inputdlg('Select region of interest: y/n', 'ROI selection', 1, {'y'});

if strcmp(maskImages{1},'y')
    hf = figure; 
    imshow(refImage, []);
    title('Select window to keep after masking')
    [~,~,tempImage,Rect] = imcrop(hf);
    specLocBottomLeft = ceil([Rect(1),Rect(2)]);
    specLocTopRight = floor([Rect(1)+Rect(3),Rect(2)+Rect(4)]);   
elseif strcmp(maskImages{1},'c1')
    % Specimen is 1.8mm from the bottom left hand corner
    specLocBottomLeft = [12,1];
    specLocTopRight = [398,250];
elseif strcmp(maskImages{1},'c2')
    % Specimen is 2mm from the bottom left hand corner
    specLocBottomLeft = [12,1];
    specLocTopRight = [400,250];
else
    specLocBottomLeft = [1,1];
    specLocTopRight = [size(refImage,2),size(refImage,1)];   
end
close all
range.x = (specLocBottomLeft(1):specLocTopRight(1));
range.y = (specLocBottomLeft(2):specLocTopRight(2));

%--------------------------------------------------------------------------
% 3) Input Grid Parameters
fprintf('Input grid parameters.\n')
gridData = inputdlg({'Number of pixels per period:','Grid pitch: (m)'}, ...
             'Grid analysis parameters', 1, {num2str(grid.pxPerPeriod), num2str(grid.pitch)} );
grid.pxPerPeriod = str2double(gridData{1}); 
grid.pitch = str2double(gridData{2});
grid.mPerPx = grid.pitch/grid.pxPerPeriod;

%--------------------------------------------------------------------------
% 4) Build the Analysis Window
%Build Window
analysisWindow = build_window(gridMethodOpts.windowFlag,...
    gridMethodOpts.windowWidth*grid.pxPerPeriod); % See documentation
% 0 = Gaussian window, default
% 1 = Bi-triangular window, localised phenom, low noise

%--------------------------------------------------------------------------
% 5) Process Images
% Pre-alloc vars for speed
[sy,sx] = size(refImage);
st = length(refImageFileCell );
disp.x = zeros([sy,sx,st]); 
disp.y = zeros([sy,sx,st]);
disp.rot = zeros([sy,sx,st]);
phi.x = zeros([sy,sx,st]);
phi.y = zeros([sy,sx,st]);

% Spatial Unwrapping
fprintf('Spatial unwrapping.\n')
for i = 1:length(refImageFileCell)
    % Load the current image file
    currImage = imread([refImagePath,refImageFileCell{i}]);
    currImage = double(currImage);
	
	% Add noise to the image if required
	if noise.addNoise
		currImage = func_addNoiseToImages(noise.pcNoise,currImage,...
            noise.bits,noise.convToUInt16);
	end
    
    % Calculation of the phase and spatial phase unwrapping
    [phi.x(:,:,i),phi.y(:,:,i),phaseMod.x(:,:,i),phaseMod.y(:,:,i)] = ...
        LSA(currImage, analysisWindow, grid.pxPerPeriod);
    phi.x(:,:,i) = unwrap2D(single(phi.x(:,:,i)));
    phi.y(:,:,i) = unwrap2D(single(phi.y(:,:,i)));
end

% Temporal Unwrapping
% CAUTION: this only works for a sequence of images with small increments in
% the phase, uses average phase over the FOV to calc rigid body motion
if gridMethodOpts.temporalUnwrap
    range.x = (specLocBottomLeft(1):specLocTopRight(1));
    range.y = (specLocBottomLeft(2):specLocTopRight(2));
    phi.x_nuw = phi.x;
    phi.y_nuw = phi.y;
    
    fprintf('Temporal unwrappping.\n')
    threshold = pi;
    phase = func_temporalUnwrap(phi,threshold,'field',range);

    for i = 1:size(phi.x,3)
        phi.x(:,:,i) = phi.x(:,:,i) + 2*pi*phase.x(i); 
        phi.y(:,:,i) = phi.y(:,:,i) + 2*pi*phase.y(i);
    end
end

if gridMethodOpts.debug
    figure;
    hold on
    plot(squeeze(mean(mean(phi.x(range.y,range.x,:)))),'-+b')
    plot(squeeze(mean(mean(phi.x_nuw(range.y,range.x,:)))),'-xr')
    xlabel('Frame')
    ylabel('Mean Phase Over ROI')
    legend('Temp Unwrapped','No Unwrap')
    hold off    
end

% Calculate Displacements and Strains
fprintf('Calculating displacement and strain components.\n')
disp.x = zeros(size(phi.x));
disp.y = zeros(size(phi.x));
disp.rot = zeros(size(phi.x));

% Calculate a window over which the specimen exists + an extra pitch to
% avoid nans - currently uses the range variable
% X Range
if (min(range.x)-grid.pxPerPeriod) < 1
    startInd = 1;
else
    startInd = min(range.x)-grid.pxPerPeriod;
end
if (max(range.x)+grid.pxPerPeriod) > size(phi.x,2)
    endInd = size(phi.x,2);
else
    endInd = max(range.x)+grid.pxPerPeriod;
end
dispRangeX = startInd:endInd;

% Y Range
if (min(range.y)-grid.pxPerPeriod) < 1
    startInd = 1;
else
    startInd = min(range.y)-grid.pxPerPeriod;
end
if (max(range.y)+grid.pxPerPeriod) > size(phi.y,1)
    endInd = size(phi.y,1);
else
    endInd = max(range.y)+grid.pxPerPeriod;
end
dispRangeY = startInd:endInd;

for i = 1:length(refImageFileCell) 
    [disp.x(dispRangeY,dispRangeX,i), disp.y(dispRangeY,dispRangeX,i),~,~,~, disp.rot(dispRangeY,dispRangeX,i)]...
        = calculate_U_EPS(grid.pxPerPeriod,squeeze(phi.x(dispRangeY,dispRangeX,1)),squeeze(phi.y(dispRangeY,dispRangeX,1)),...
        squeeze(phi.x(dispRangeY,dispRangeX,i)),squeeze(phi.y(dispRangeY,dispRangeX,i)),gridMethodOpts.dispCalcMethod,100);  
end

%--------------------------------------------------------------------------
% 6) Convert Displacement and Crop to ROI 
% Convert the displacement to mm
fprintf('Cropping to ROI and updating position/size vectors.\n')
disp.x = disp.x*(grid.pitch/grid.pxPerPeriod); 
disp.y = disp.y*(grid.pitch/grid.pxPerPeriod);

% Crop the image to the ROI
disp.x = disp.x(range.y,range.x,:);
disp.y = disp.y(range.y,range.x,:);

% Create the position mesh grid
pos.x = grid.mPerPx/2:grid.mPerPx:(grid.mPerPx*size(disp.x,2));
pos.y = grid.mPerPx/2:grid.mPerPx:(grid.mPerPx*size(disp.x,1));
[pos.xGrid,pos.yGrid] = meshgrid(pos.x,pos.y);
pos.xStep = grid.mPerPx;
pos.yStep = grid.mPerPx;

% Update grid size parameters
grid.length = size(disp.x,2)*grid.pitch/grid.pxPerPeriod;
grid.height = size(disp.y,1)*grid.pitch/grid.pxPerPeriod;

%--------------------------------------------------------------------------
% Ask user about grid rotation and correct if needed
fprintf('Correcting for any grid rotation.\n')
[grid,disp] = func_rotateDisp(grid,disp);

end

