%% IMAGE-BASED INERTIAL IMPACT TEST - TWO CAMERA PROCESSING
% PhotoDyn Research Group
% Author: Jared Van Blitterswyk
% Date: 13 Sept. 2018

% This code loads in processed grid method data from the master and slave
% camera, computes the common ROI and then re-computes kinematic fields.
% The fields are then averaged and used to compute the stiffness of the
% material using the stress-strain curve and optimised VFM approaches.

%% Initialize
clc
clear all
close all

fprintf('--------------------------------------------------------------\n')
fprintf('IMAGE-BASED INERTIAL IMPACT (IBII): Back-to-back Image Processing\n')
fprintf('--------------------------------------------------------------\n')
% add data path for image processing functions
funcPath = 'E:\PhD Southampton\MATLAB\IBIITest_ProcessingToolBox_v2_3\GlobalProcessingFunctions_2018\';
%funcPath = [pwd,'\GlobalProcessingFunctions_2018\'];

% If the default path is not found we should find it
if exist(funcPath,'file') ~= 7
    hWarn = warndlg('Folder for global processing functions not found.','Function folder not found');
    waitfor(hWarn);
    funcPath = uigetdir(pwd,'Locate Global Processing Function Folder');
end
addpath(funcPath);
addpath([funcPath,'GridMethodToolbox\']);
addpath([funcPath,'Export_Fig\'])

gridDataFile = 'Vars_For_BTB_Processing.mat';
initFile = 'processingParameters.mat';

cameraBit = 12;
cameraDynRange = 2^cameraBit;

debugOpts.showDiagnosticFigures = true;
selectROIs = false;
numCameras = 2;
if numCameras == 2 
    filename_fields = 'MasterAndSlaveFields_CommonFOV_S41S71T11.mat';
else
    filename_fields = 'ProcessedData_S41S71T11.mat';
end
    
%% Select Folders Containing Images and Processed Data
if numCameras == 2
    fprintf('Loading reference image from master camera.\n')
    [imageFileM,imagePathM] = uigetfile({'*.*','All Files'},'Select the first image in the sequence (Master Camera)');
    fprintf('Loading reference image from slave camera.\n')
    [imageFileS,imagePathS] = uigetfile({'*.*','All Files'},'Select the first image in the sequence (Slave Camera)');
else
    fprintf('Loading reference image from master camera.\n')
    [imageFileM,imagePathM] = uigetfile({'*.*','All Files'},'Select the first image in the sequence (Master Camera)');
end

%% INITIALISE - Load Processing Parameter Data Structures
initPath = imagePathM;
if exist([initPath,initFile],'file') ~= 2
    hWarn = warndlg('Processing parameter file does not exist.','Processing parameters not found');
    waitfor(hWarn);
    [initFile,initPath,~] = uigetfile('*.mat','Locate processing parameter file');
end
% Load the processing parameters from file
fprintf('Loading initialisation file.\n');
load([initPath,initFile])

plotParams.tempLimit = 100; % update to variable fracture - need to load original data file
framesToFit = 70;
plotParams.folderFieldsVFM = strcat('E:\PhD Southampton\Experimental Data\CFRP_TT_B2B\Diagnostic Figures\SPC 7\Fields_VFM\');
plotParams.saveFigFieldsSS = true;
plotParams.saveFigFieldsStressStrain = true;
plotParams.saveFigLineStressStrain = true;
plotParams.saveFigFieldsVFM = false;
plotParams.folderFieldsSS = 'E:\PhD Southampton\Experimental Data\CFRP_TT_B2B\Diagnostic Figures\SPC 7\Fields_SS\';
plotParams.folderFieldsStressDiff = 'E:\PhD Southampton\Experimental Data\CFRP_TT_B2B\Diagnostic Figures\SPC 7\Fields_StressStrain\';
plotParams.folderLineStressDiff = 'E:\PhD Southampton\Experimental Data\CFRP_TT_B2B\Diagnostic Figures\SPC 7\Line_StressStrain\';
plotParams.folderFieldsDiff = 'E:\PhD Southampton\Experimental Data\CFRP_TT_B2B\Diagnostic Figures\SPC 7\Fields_Diff\';
%% LOAD GRID METHOD DATA (assuming the required coordinates are extracted ('v 2.4'))
if numCameras == 2
    % master camera
    fprintf('Loading processed data for master camera...\n');
    load(strcat(imagePathM,gridDataFile));
    dispM = disp; accelM = accel; strainM = strain; strainRateM = strainRate; gridM = grid; posM = pos; RectM = Rect; mPointsX = pointsX; mPointsY = pointsY;
    clearvars disp accel strain strainRate grid pos Rect pointsX pointsY
    % slave camera
    fprintf('Loading processed data for slave camera...\n');
    load(strcat(imagePathS,gridDataFile));
    dispS = disp; accelS = accel; strainS = strain; strainRateS = strainRate; gridS = grid; posS = pos; RectS = Rect; sPointsX = pointsX; sPointsY = pointsY;
    clearvars disp accel strain strainRate grid pos Rect pointsX pointsY
else
    % master camera
    fprintf('Loading processed data for master camera...\n');
    load(strcat(imagePathM,gridDataFile));
    dispM = disp; accelM = accel; strainM = strain; strainRateM = strainRate; gridM = grid; posM = pos; RectM = Rect; mPointsX = pointsX; mPointsY = pointsY;
    clearvars disp accel strain strainRate grid pos Rect pointsX pointsY
end

%% COMPUTE COMMON ROI BETWEEN CAMERA AND CROP FIELDS
if numCameras == 2
    fprintf('Computing common ROI between cameras.\n');
    % compute common ROI between cameras based on selected ROIs and position of
    % top corner of sample (must be selected in processing of individual
    % images)
    refImageM = imread([imagePathM,imageFileM]);
    refImageM = double(refImageM);
    refImageS = imread([imagePathS,imageFileS]);
    refImageS = fliplr(double(refImageS));
    
    [mROIcrop, sROIcrop] = f_computeCommonROI_MasterSlaveCameras(RectM, mPointsX, mPointsY, RectS, sPointsX, sPointsY, refImageM, refImageS, debugOpts);
else
    refImageM = imread([imagePathM,imageFileM]);
    refImageM = double(refImageM);
end
%%
% only required for debugging data sets, comment out if desired
gridM.mPerPx = gridM.pitch/gridM.pxPerPeriod;
gridS.mPerPx = gridM.mPerPx;

% adjust and assign position variables to both cameras with cropped fields
fprintf('Adjust position vectors to account for offset of ROI from free edge.\n');
xOffset = gridM.mPerPx*(RectM(1)-mPointsX(1));
yOffset = gridM.mPerPx*(RectM(2)-mPointsY(1));
posM.x = posM.x+xOffset;
posM.xGrid = posM.xGrid+xOffset;
posM.y = posM.y+yOffset;
posM.yGrid = posM.yGrid+yOffset;

specimenM = specimen; specimenMC = specimen;
specimenM.length = max(posM.x)-min(posM.x); specimenM.height = max(posM.y)-min(posM.y);

if numCameras == 2
    xOffsetS = gridS.mPerPx*(RectS(1)-sPointsX(1));
    yOffsetS = gridS.mPerPx*(RectS(2)-sPointsY(1));
    posS.x = posS.x+xOffsetS;
    posS.xGrid = posS.xGrid+xOffsetS;
    posS.y = posS.y+yOffsetS;
    posS.yGrid = posS.yGrid+yOffsetS;
    
    % ---------- INTERPOLATE SLAVE FIELDS TO MASTER COORDINATES ----------
    fprintf('Interpolating fields from slave camera to master camera coordinates.\n');
    for i = 1:size(dispM.x,3)
        dispS_I.x(:,:,i) = interp2(posS.xGrid,posS.yGrid,dispS.x(:,:,i), posM.xGrid, posM.yGrid, 'linear');
        dispS_I.y(:,:,i) = interp2(posS.xGrid,posS.yGrid,dispS.y(:,:,i), posM.xGrid, posM.yGrid, 'linear');
        accelS_I.x(:,:,i) = interp2(posS.xGrid,posS.yGrid,accelS.x(:,:,i), posM.xGrid, posM.yGrid, 'linear');
        accelS_I.y(:,:,i) = interp2(posS.xGrid,posS.yGrid,accelS.y(:,:,i), posM.xGrid, posM.yGrid, 'linear');
        strainS_I.x(:,:,i) = interp2(posS.xGrid,posS.yGrid,strainS.x(:,:,i), posM.xGrid, posM.yGrid, 'linear');
        strainS_I.y(:,:,i) = interp2(posS.xGrid,posS.yGrid,strainS.y(:,:,i), posM.xGrid, posM.yGrid, 'linear');
        strainS_I.s(:,:,i) = interp2(posS.xGrid,posS.yGrid,strainS.s(:,:,i), posM.xGrid, posM.yGrid, 'linear');
        strainRateS_I.x(:,:,i) = interp2(posS.xGrid,posS.yGrid,strainRateS.x(:,:,i), posM.xGrid, posM.yGrid, 'linear');
        strainRateS_I.y(:,:,i) = interp2(posS.xGrid,posS.yGrid,strainRateS.y(:,:,i), posM.xGrid, posM.yGrid, 'linear');
        strainRateS_I.s(:,:,i) = interp2(posS.xGrid,posS.yGrid,strainRateS.s(:,:,i), posM.xGrid, posM.yGrid, 'linear');
    end
    
    specimenS = specimen; specimenSC = specimen;
    specimenA = specimen;
    specimenS.length = max(posS.x)-min(posS.x); specimenS.height = max(posS.y)-min(posS.y);
    
    % incorporate fix nan code here to extrapolate to NaN regions
    
    fprintf('Replacing NaNs with moving window average.\n');
    dispS_I.x = func_fixNaNsInDispX(extrapOpts,dispS_I.x);
    dispS_I.x = func_fixNaNsInDispY(extrapOpts,dispS_I.x);
    dispS_I.y = func_fixNaNsInDispX(extrapOpts,dispS_I.y);
    dispS_I.y = func_fixNaNsInDispY(extrapOpts,dispS_I.y);
    
    accelS_I.x = func_fixNaNsInDispX(extrapOpts,accelS_I.x);
    accelS_I.x = func_fixNaNsInDispY(extrapOpts,accelS_I.x); 
    accelS_I.y = func_fixNaNsInDispX(extrapOpts,accelS_I.y);
    accelS_I.y = func_fixNaNsInDispY(extrapOpts,accelS_I.y); 
    
    strainS_I.x = func_fixNaNsInDispX(extrapOpts,strainS_I.x);
    strainS_I.x = func_fixNaNsInDispY(extrapOpts,strainS_I.x);
    strainS_I.y = func_fixNaNsInDispX(extrapOpts,strainS_I.y);
    strainS_I.y = func_fixNaNsInDispY(extrapOpts,strainS_I.y);
    strainS_I.s = func_fixNaNsInDispX(extrapOpts,strainS_I.s);
    strainS_I.s = func_fixNaNsInDispY(extrapOpts,strainS_I.s);
    
    strainRateS_I.x = func_fixNaNsInDispX(extrapOpts,strainRateS_I.x);
    strainRateS_I.x = func_fixNaNsInDispY(extrapOpts,strainRateS_I.x);
    strainRateS_I.y = func_fixNaNsInDispX(extrapOpts,strainRateS_I.y);
    strainRateS_I.y = func_fixNaNsInDispY(extrapOpts,strainRateS_I.y);
    strainRateS_I.s = func_fixNaNsInDispX(extrapOpts,strainRateS_I.s);
    strainRateS_I.s = func_fixNaNsInDispY(extrapOpts,strainRateS_I.s);
    
    fprintf('Cropping fields.\n');
    % crop fields - position, grid, displacement, acceleration, strain, strain rate
    [dispMC,accelMC,strainMC,strainRateMC,gridMC,posMC] = f_cropFieldsROI_BTB(mROIcrop, dispM, accelM, strainM, strainRateM, gridM, posM);
    [dispSC,accelSC,strainSC,strainRateSC,gridSC,posSC] = f_cropFieldsROI_BTB(mROIcrop, dispS_I, accelS_I, strainS_I, strainRateS_I, gridS, posS);
    
    
    specimenMC.length = max(posMC.x)-min(posMC.x); specimenMC.height = max(posMC.y)-min(posMC.y);
    specimenA.length = specimenMC.length; specimenA.height = specimenMC.height;
    % after interpolation cropped fields are defined in the master camera coordinates
    posSC.x = posMC.x; posSC.y = posMC.y; 
    posSC.xGrid = posMC.xGrid; posSC.yGrid = posMC.yGrid;

    specimenSC.length = max(posSC.x)-min(posSC.x); specimenSC.height = max(posSC.y)-min(posSC.y);
    
    % ---------- AVERGAE FIELDS FOR FURTHER PROCESSING ----------
    fprintf('Computing average fields for processing.\n');
    posA = posMC; % fields cropped in the master camera coordinate system 
    % displacement
    dispA.x = 0.5*(dispMC.x+dispSC.x); dispA.y = 0.5*(dispMC.y+dispSC.y); %dispA.rot = 0.5*(dispM.rot+dispS.rot);
    % acceleration
    accelA.x = 0.5*(accelMC.x+accelSC.x); accelA.y = 0.5*(accelMC.y+accelSC.y);
    accelA.xAvg = 0.5*(accelMC.xAvg+accelSC.xAvg); 
    % strain
    strainA.x = 0.5*(strainMC.x+strainSC.x); strainA.y = 0.5*(strainMC.y+strainSC.y); strainA.s = 0.5*(strainMC.s+strainSC.s);
    strainA.xAvg = 0.5*(strainMC.xAvg+strainSC.xAvg); strainA.yAvg = 0.5*(strainMC.yAvg+strainSC.yAvg);
    % strain rate
    strainRateA.x = 0.5*(strainRateMC.x+strainRateSC.x); strainRateA.y = 0.5*(strainRateMC.y+strainRateSC.y); strainRateA.s = 0.5*(strainRateMC.s+strainRateSC.s);
    strainRateA.xAvg = 0.5*(strainRateMC.xAvg+strainRateSC.xAvg);
    
    % RECONSTRUCT STRESS AVERAGES WITH STRESS-GAUGE EQUATION
    fprintf('Calculating average stress with the stress-gauge equation.\n')
    % Standard Stress Gauge equation calculation
    [stressA.xAvg,~] = func_stressGaugeProcess(material,time,posA,accelA);
    [stressMC.xAvg,~] = func_stressGaugeProcess(material,time,posMC,accelMC);
    [stressSC.xAvg,~] = func_stressGaugeProcess(material,time,posSC,accelSC);
    [stressM.xAvg,~] = func_stressGaugeProcess(material,time,posM,accelM);
    [stressS.xAvg,~] = func_stressGaugeProcess(material,time,posS,accelS);
    % First moment of the stress-gauge
    fprintf('Calculating the average stress with the linear stress-gauge equation.\n')
    stressA = func_linearStressGaugeProcess(specimen,material,posA,accelA,stressA);
    stressMC = func_linearStressGaugeProcess(specimen,material,posMC,accelMC,stressMC);
    stressSC = func_linearStressGaugeProcess(specimen,material,posSC,accelSC,stressSC);
    stressM = func_linearStressGaugeProcess(specimen,material,posM,accelM,stressM);
    stressS = func_linearStressGaugeProcess(specimen,material,posS,accelS,stressS);
    
    % RENAME VARIABLES and SAVE FOR PROCESSING
    fprintf('Condensing variables for saving.\n');
    % pos
    pos = posMC; pos.M = posM; pos.S = posS;
    % disp
    disp.M = dispM; disp.S = dispS; disp.MC = dispMC; disp.SC = dispSC;
    % accel
    accel.M = accelM; accel.S = accelS; accel.MC = accelMC; accel.SC = accelSC;
    accel.A = accelA;
    % strain
    strain.M = strainM; strain.MC = strainMC; strain.S = strainS; 
    strain.SC = strainSC; strain.A = strainA;
    % strain rate
    strainRate.M = strainRateM; strainRate.MC = strainRateMC; strainRate.S = strainRateS; 
    strainRate.SC = strainRateSC; strainRate.A = strainRateA;
    % stress
    stress.M = stressM; stress.MC = stressMC; stress.S = stressS; 
    stress.SC = stressSC; stress.A = stressA;
    % specimen
    specimen.M = specimenM; specimen.MC = specimenMC; specimen.S = specimenS; 
    specimen.SC = specimenSC; specimen.A = specimenA;

    pathname_fields = imagePathM;

%     fprintf('Saving fields to master camera folder...\n');
    save(strcat(pathname_fields,filename_fields),'pos','disp','accel','strain','strainRate','stress','specimen');
else
    fprintf('Calculating average stress with the stress-gauge equation.\n')
    [stressM.xAvg,~] = func_stressGaugeProcess(material,time,posM,accelM);
    fprintf('Calculating the average stress with the linear stress-gauge equation.\n')
    stressM = func_linearStressGaugeProcess(specimen,material,posM,accelM,stressM);
    
    % RENAME VARIABLES and SAVE FOR PROCESSING
    pos = posM; disp = dispM; accel = accelM; strain = strainM; strainRate = strainRateM;
    stress = stressM; specimen = specimenM; 
    
    pathname_fields = imagePathM;
    
    fprintf('Saving fields to master camera folder...\n');
    save(strcat(pathname_fields,filename_fields),'pos','disp','accel','strain','strainRate','stress','specimen');
end

%
%%
% fprintf('Clearing variables and reloading processing file...\n');
% clearvars -except pos disp accel strain strainRate stress specimen imagePathM imagePathS initFile initPath time numCameras
% load([initPath,initFile])
%% % VFM: PIECE-WISE SPECIAL OPTIMISED

% VFOpts.nElemX = 3;
% xOffsetM = 0; yOffsetM = 0;
% VFOpts.cutEdgePx = smoothingOpts.spatialKernal(1) + gridM.pxPerPeriod;
% 
% if numCameras == 2
%     % Use reduced optimised virtual fields to obtain Qxx (master fields)
%     fprintf('Identifying Qxx with reduced optimised virtual fields (2D VFM - master fields).\n')
%     [identStiffVFOptM.ExxVsT,VFs_M] = func_VFDynPWSpecOptReducedLinElas(VFOpts,...
%     pos, strain.MC, accel.MC, material);
%     % Use reduced optimised virtual fields to obtain Qxx (slave fields)
%     fprintf('Identifying Qxx with reduced optimised virtual fields (2D VFM - slave fields).\n')
%     [identStiffVFOptS.ExxVsT,VFs_S] = func_VFDynPWSpecOptReducedLinElas(VFOpts,...
%         pos, strain.SC, accel.SC, material);
%         % Use reduced optimised virtual fields to obtain Qxx (average fields)
%     fprintf('Identifying Qxx with reduced optimised virtual fields (2D VFM - average fields).\n')
%     [identStiffVFOptA.ExxVsT,VFs_A] = func_VFDynPWSpecOptReducedLinElas(VFOpts,...
%         pos, strain.A, accel.A, material);
% else
%     % Use reduced optimised virtual fields to obtain Qxx (master fields)
%     fprintf('Identifying Qxx with reduced optimised virtual fields (2D VFM - master fields).\n')
%     [identStiffVFOptM.ExxVsT,VFs_M] = func_VFDynPWSpecOptReducedLinElas(VFOpts,...
%     pos, strain, accel, material);
% end
% time.vec = time.step:time.step:time.step*time.numFrames;
% figure('color',[1 1 1]);
% set(gcf, 'Units', 'Normalized', 'OuterPosition', [0.15, 0.3, 0.55, 0.5]);
% plot(time.vec*1e06,identStiffVFOptM.ExxVsT/1e09,'color',[0 0 0.47],'linewidth',1)
% hold on
% plot(time.vec*1e06,identStiffVFOptS.ExxVsT/1e09,'color',[0 0.47 0],'linewidth',1)
% plot(time.vec*1e06,identStiffVFOptA.ExxVsT/1e09,'color',[0.47 0 0],'linewidth',1)
% set(gca,'FontSize',10,'Linewidth',1,'FontName','Calibri Light')
% xlabel('Time, x [mm]')
% ylabel('E_{33} (GPa)')
% grid on
% ylim([6 16])
% legend('HPV-X','location','northeast')
% fprintf('Finished.\n');
% 
% %%
% plotParams.xlimI = [0 80];
% plotParams.ylimI = [6 22];
% if numCameras == 2
%     f_plotFieldsVFIdentification(strain, VFs_M, VFs_S, identStiffVFOptM, identStiffVFOptS, pos, plotParams, VFOpts)
% end
% 
% % %% --------------------------------------------------------------------------
% %% FIND TIME WHEN PULSE HITS FREE EDGE
% % fit slope of line using 10 frames, when slope changes by certain amount,
% % pulse hits edge
% frameWindow = 10;
% slopeThreshold = -7e05;
% framesToFit = 90;
% 
% slope = zeros(1,framesToFit);
% 
% frameOffset = 5; % offset to avoid temporal smoothing effects
% fprintf('Compute when pulse hits free edge and has fully reflected).\n')
% if numCameras == 2
%     figure;
%     plot(time.vec*1e06,stress.MC.xAvg(end,:)/1e06,'linewidth',1,'color',[0 0 0.47])
%     set(gca,'FontSize',10,'Linewidth',1,'FontName','Calibri Light')
%     xlabel('Time [\mu s]')
%     ylabel('\sigma_{xx}^{x=L} (MPa)')
%     grid on
%     legend('HPV-X','location','southeast')
%     xlim([0 25])
%     
%     tempVec = stress.MC.xAvg(1,frameOffset:framesToFit)-slopeThreshold;
%     temp = tempVec(tempVec<0);
% 
%     [~, plotParams.pulseFreeEdge] = min(abs(tempVec-temp(1)));
%     plotParams.pulseFreeEdge = plotParams.pulseFreeEdge+frameOffset;
% 
%     % find when pulse leaves free edge
%     [~,temp] = min(abs(stress.MC.xAvg(1,plotParams.pulseFreeEdge+20:framesToFit)));
% 
%     % diagnostic figure
%     figure;
%     plot(abs(stress.MC.xAvg(1,plotParams.pulseFreeEdge:framesToFit)),'linewidth',1,'color',[0 0 0.47])
%     set(gca,'FontSize',10,'Linewidth',1,'FontName','Calibri Light')
%     xlabel('Frame')
%     ylabel('\sigma_{xx} (MPa)')
%     grid on
%     legend('HPV-X','location','southeast')
%     
%     plotParams.pulseFreeEdge2 = temp + plotParams.pulseFreeEdge;
% 
%     % find when peak pulse reaches free edge
%     [~,temp2] = max(abs(stress.MC.xAvg(1,plotParams.pulseFreeEdge:plotParams.pulseFreeEdge2)));
%     plotParams.pulsePeakFreeEdge = plotParams.pulseFreeEdge + temp2;
%     
%     % diagnostic figure
%     figure;
%     plot(abs(accel.MC.xAvg(1,plotParams.pulseFreeEdge:plotParams.pulseFreeEdge2)))
% else
%    
%     figure;
%     plot(time.vec*1e06,stress.xAvg(end,:)/1e06,'linewidth',1,'color',[0 0 0.47])
%     set(gca,'FontSize',10,'Linewidth',1,'FontName','Calibri Light')
%     xlabel('Time [\mu s]')
%     ylabel('\sigma_{xx}^{x=L} (MPa)')
%     grid on
%     legend('HPV-X','location','southeast')
%     xlim([0 25])
%     
%     tempVec = stress.xAvg(1,frameOffset:framesToFit)-slopeThreshold;
%     temp = tempVec(tempVec<0);
% 
%     [~, plotParams.pulseFreeEdge] = min(abs(tempVec-temp(1)));
% 
%     % find when pulse leaves free edge
%     [~,temp] = min(abs(stress.xAvg(1,plotParams.pulseFreeEdge+10:framesToFit)));
%     
%     % diagnostic figure
%     figure;
%     plot(abs(stress.xAvg(1,plotParams.pulseFreeEdge:framesToFit)),'linewidth',1,'color',[0 0 0.47])
%     set(gca,'FontSize',10,'Linewidth',1,'FontName','Calibri Light')
%     xlabel('Frame')
%     ylabel('\sigma_{xx} (MPa)')
%     grid on
%     legend('HPV-X','location','southeast')
%     
%     plotParams.pulseFreeEdge2 = temp + plotParams.pulseFreeEdge;
% 
%     % find when peak pulse reaches free edge
%     [~,temp2] = max(abs(stress.xAvg(1,plotParams.pulseFreeEdge:plotParams.pulseFreeEdge2-10)));
%     plotParams.pulsePeakFreeEdge = plotParams.pulseFreeEdge + temp2;
%     
%     % diagnostic figure
%     figure;
%     plot(abs(accel.xAvg(1,plotParams.pulseFreeEdge:plotParams.pulseFreeEdge2)))
% end
% 
% %% ------------------ PLOT OVERLAY STRESS-STRAIN CURVES --------------------
% % determine positions along sample from which to plot stress-strain curves
% plotParams.numFigs = 8;
% tempPos = pos.x(smoothingOpts.spatialKernal:end-smoothingOpts.spatialKernal);
% plotParams.posInc = (max(tempPos)-min(tempPos))/plotParams.numFigs;
% tempPos = plotParams.posInc+min(tempPos);
% 
% 
% for i = 1:plotParams.numFigs
%     [~, plotIndex(i)] = min(abs(pos.x-tempPos));
%     tempPos = tempPos + plotParams.posInc;
% end
% plotIndex = int16(plotIndex);
% plotParams.ylim = [-150,100];
% plotParams.xlim = [-16,16];
% 
% % compute time at which extracted locations see second pulse
% plotParams.tIndex1 = plotParams.pulseFreeEdge + 20 - round(plotIndex*0.337/7/(2.5*0.2));
% plotParams.tIndex2 = plotParams.tIndex1+plotParams.pulsePeakFreeEdge-plotParams.pulseFreeEdge;
% plotParams.tIndex3 = plotParams.tIndex2+plotParams.pulseFreeEdge2-plotParams.pulseFreeEdge;
% 
% plotParams.tIndexMPeak = plotParams.pulsePeakFreeEdge - round(plotIndex*0.337/7/(2.5*0.2));
% 
% fprintf('Ploting stress-strain curves along the sample.\n')
% if numCameras == 2
%     % plot stress-strain curves overlayed from two cameras
%     f_plotOverlayStressStrainCurves(strain, stress, pos, plotIndex, plotParams)
%     % f_plotAverageStressStrainCurves(strain, stress, pos, plotIndex, plotParams)
% else
%     f_plotOverlayStressStrainCurves_SingleCamera(strain, stress, pos, plotIndex, plotParams)
% end
% 
% %% Identify stiffness using stress-strain curves
% if numCameras == 2
%     [identStiffSG_M.QxxVsL,identStiffSG_M.QxxLinFitCoeffs] = ...
%             func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stress.MC.xAvg,strain.MC.xAvg);
%     [identStiffSG_S.QxxVsL,identStiffSG_S.QxxLinFitCoeffs] = ...
%         func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stress.SC.xAvg,strain.SC.xAvg);
% 
%     % set regression options
%     regOpts.fitFrames = 5;
%     regOpts.frameMax = plotParams.pulseFreeEdge2;
%     regOpts.threshold = 0.90;
%     regOpts.startFrameThreshold = plotParams.pulseFreeEdge;
% 
%     % compute regressions to obtain stiffness from stress-strain curves
%     [identStiffSG_M2.QxxVsL,Rsq_M2,indexThreshold_M,RsqTM_M,tangMod_M] = f_identStiffStressStrainCurve_TangentThreshold(stress.MC.xAvg,strain.MC.xAvg,regOpts);
%     [identStiffSG_S2.QxxVsL,Rsq_S2,indexThreshold_S,RsqTM_S,tangMod_S] = f_identStiffStressStrainCurve_TangentThreshold(stress.SC.xAvg,strain.SC.xAvg,regOpts);
% 
% %     figure;
% %     plot(RsqTM_M(330,:))
% 
%     figure('color',[1 1 1]);
%     set(gcf, 'Units', 'Normalized', 'OuterPosition', [0.15, 0.3, 0.55, 0.5]);
%     plot(identStiffSG_M.QxxVsL/1e09,'color',[0 0 0.47],'linewidth',1)
%     hold on
%     plot(identStiffSG_S.QxxVsL/1e09,'color',[0 0.47 0],'linewidth',1)
%     plot(identStiffSG_M2.QxxVsL/1e09,'color',[0 0 0.47],'linewidth',1,'linestyle','--')
%     plot(identStiffSG_S2.QxxVsL/1e09,'color',[0 0.47 0],'linewidth',1,'linestyle','--')
%     set(gca,'FontSize',10,'Linewidth',1,'FontName','Calibri Light')
%     xlabel('Position, x [mm]')
%     ylabel('E_{33} (GPa)')
%     grid on
%     legend('HPV-X','HPV-X2','HPV-X (Reg Threshold)','HPV-X2 (Reg Threshold)','location','southeast')
% else
%     [identStiffSG_M.QxxVsL,identStiffSG_M.QxxLinFitCoeffs] = ...
%         func_identStiffLinFitStressStrainCurve(stressGaugeOpts,stress.xAvg,strain.xAvg);
% 
%     % set regression options
%     regOpts.fitFrames = 7;
%     regOpts.frameMax = plotParams.pulseFreeEdge2;
%     regOpts.threshold = 0.92;
%     regOpts.startFrameThreshold = plotParams.pulseFreeEdge;
% 
%     % compute regressions to obtain stiffness from stress-strain curves
%     [identStiffSG_M2.QxxVsL,Rsq_M2,indexThreshold_M,RsqTM_M,tangMod_M] = f_identStiffStressStrainCurve_TangentThreshold(stress.xAvg,strain.xAvg,regOpts);
% 
% %     figure;
% %     plot(RsqTM_M(300,:))
% 
%     figure('color',[1 1 1]);
%     set(gcf, 'Units', 'Normalized', 'OuterPosition', [0.15, 0.3, 0.55, 0.5]);
%     plot(identStiffSG_M.QxxVsL/1e09,'color',[0 0 0.47],'linewidth',1)
%     hold on
%     plot(identStiffSG_M2.QxxVsL/1e09,'color',[0 0 0.47],'linewidth',1,'linestyle','--')
%     set(gca,'FontSize',10,'Linewidth',1,'FontName','Calibri Light')
%     xlabel('Position, x [mm]')
%     ylabel('E_{33} (GPa)')
%     grid on
%     legend('HPV-X','HPV-X (Reg Threshold)','location','southeast')
% end
% 
% %% ------------ PLOT FIELDS AND DIFFERENCE BETWEEN FIELDS -----------------
% % ------------------- FIELDS WITH STRESS-STRAIN CURVE ---------------------
% plotParams.cropPxFields = 0;%grid.pxPerPeriod;
% plotParams.posFields = 5; % location to plot in figures (5 approx. location of failure for SPC 1)
% plotParams.climAccel = [-0.75e7,0.75e7];
% plotParams.climStrain = [-16,10];
% 
% % 
% % % --------------------- PLOT FIELDS AND DIFFERENCE ------------------------
% 
% if numCameras == 2
% 
%     stress.MC.xAvgF = zeros(size(strain.MC.x));
%     stress.SC.xAvgF = stress.MC.xAvgF;
%     % convert SG matrix to fields
%     for j = 1:size(strain.MC.x,3)
%         for i = 1:size(strain.MC.x,2)
%             stress.MC.xAvgF (:,i,j) = stress.MC.xAvg(i,j);
%             stress.SC.xAvgF (:,i,j) = stress.SC.xAvg(i,j);
%         end
%     end
%     
%     % plotting functions
%     f_plotFieldsWithOverlayStressStrainCurve(strain, stress, accel, pos, plotIndex, plotParams)
%     f_plotFieldsDifference(strain, accel, pos, plotIndex, plotParams)
%     f_plotFieldsDifferenceStress(strain, stress, pos, plotIndex, plotParams)
%     f_plotDifferenceStress(strain, stress, pos, plotParams)
% 
%     figure('color',[1 1 1]);
%     subplot(2,1,1)
%     plot(stress.MC.xAvg(end-gridM.pxPerPeriod,1:plotParams.tempLimit)/1e06,'linewidth',1,'color',[0 0 0.47])
%     hold on
%     plot(stress.SC.xAvg(end-gridM.pxPerPeriod,1:plotParams.tempLimit)/1e06,'linewidth',1,'color',[0 0.47 0])
%     set(gca,'FontSize',10,'FontName','Calibri Light','LineWidth',1);
%     legend('HPV-X (master)','HPV-X2 (slave)','location','southeast')
%     xlim([0 105])
%     ylim([-150 50])
%     xlabel('$$Frame$$','interpreter','latex')
%     ylabel('$$\overline{\sigma_{xx}}^{y} (MPa)$$','interpreter','latex')
%     subplot(2,1,2)
%     plot(stress.MC.xAvg(end-gridM.pxPerPeriod,1:plotParams.tempLimit)/1e06 - stress.SC.xAvg(end-gridM.pxPerPeriod,1:plotParams.tempLimit)/1e06,'linewidth',1,'color',[0.47 0 0])
%     set(gca,'FontSize',10,'FontName','Calibri Light','LineWidth',1);
%     legend('HPV-X - HPV-X2','location','southeast')
%     xlim([0 105])
%     ylim([-30 30])
%     xlabel('$$Frame$$','interpreter','latex')
%     ylabel('$$\overline{\sigma_{xx}}^{y} (MPa)$$','interpreter','latex')
% else
%     f_plotFieldsWithOverlayStressStrainCurve_SingleCamera(strain, stress, accel, pos, plotIndex, plotParams)
% end
