function [disp,accel] = func_smoothCalcAccel_DispPad(pos,time,disp,smoothingOpts,diffOpt)
% Author: Lloyd Fletcher
% PhotoDyn Group, University of Southampton
% Date: 27/3/2017
% Modified by: Jared Van Blitterswyk
% Date: 15 Oct. 2017
%
% Processes output displacement fields from the grid method to obtain width
% averaged variables for processing with the stress gauge approach. Output 
% variables are obtained via numerical differentiation using a central 
% difference method. Variables output include velocity, acceleration, strain 
% and strain rate. Width averaging then smoothing is performed before taking 
% derivatives. Following differentiation variables are extrapolated spatially 
% to regain the missing pitch from the grid method.

%% FULL-FIELD: Temporal Smoothing of Full-Field Displacements for Accel 
% Crop one pitch from the edges to remove intial rubbish data
dispCrop.x = func_cropFields(disp.x,2,extrapOpts.FFDispPx);
dispCrop.y = func_cropFields(disp.y,1,extrapOpts.FFDispPx);

%--------------------------------------------------------------------------
% FULL-FIELD: Calculate full-field acceleration maps with no spatial
% smoothing to remove edge effects
fprintf('\tTemporally differentiating to obtain full-field acceleration maps.\n')

if smoothingOpts.FFTempSmooth
    fprintf('\tTemporally smoothing full-field displacements.\n')
    frameSize = smoothingOpts.FFTemporalKernal(1);
    polyOrder = smoothingOpts.FFTemporalKernal(2);
    fprintf('\tTemporal full-field filter is %s of order %i and frame width %i.\n',...
        smoothingOpts.FFTemporalFilt,polyOrder,frameSize)
    
    % Pad out the array to reduce the edge effects of the filter
    padRadius = 2*max(smoothingOpts.spatialKernal(1),smoothingOpts.spatialKernal(2));
    dispTemp = func_padDispArrayExt(pos,dispCrop,grid,padRadius,extrapOpts.padEdgeMethod,extrapOpts.dispPx,extrapOpts.padFitWindow,extrapOpts); 
    % Crop back to the original FOV
    dispTemp.x = dispTemp.x(:,padRadius-extrapOpts.FFDispPx+1:end-padRadius+extrapOpts.FFDispPx,:);
    dispTemp.y = dispTemp.y(padRadius-extrapOpts.FFDispPx+1:end-padRadius+extrapOpts.FFDispPx,:,:);
       
    if smoothingOpts.FFTemporalPad
        % Pad the displacement matrices in time
        padFrames = ceil(frameSize);
        dispTemporalPadX = padarray(dispTemp.x,[0,0,padFrames],smoothingOpts.tempPadMethod,'pre'); 
        dispTemporalPadY = padarray(dispTemp.y,[0,0,padFrames],smoothingOpts.tempPadMethod,'pre');
        
        disp.tSmooth.x = func_reshapeAndGolayFilt3D(dispTemporalPadX,smoothingOpts);
        disp.tSmooth.y = func_reshapeAndGolayFilt3D(dispTemporalPadY,smoothingOpts);
        
        % Crop the vector back to the original size to remove padding
        disp.tSmooth.x = disp.tSmooth.x(:,:,(padFrames+1):end);
        disp.tSmooth.y = disp.tSmooth.y(:,:,(padFrames+1):end);
    else
        disp.tSmooth.x = func_reshapeAndGolayFilt3D(dispTemp.x,smoothingOpts);
        disp.tSmooth.y = func_reshapeAndGolayFilt3D(dispTemp.y,smoothingOpts);
    end
else
    disp.tSmooth.x = dispTemp.x;
    disp.tSmooth.y = dispTemp.y;
end

% pad array prior to differentiating
if smoothingOpts.tempPadAccel
    if smoothingOpts.FFTempSmooth
        fprintf('\tPadding displacement prior to calculating acceleration to account for temporal smoothing and differentiation scheme.\n');
        diffOpt.tempKernel = smoothingOpts.FFTemporalKernal(1)+2; % add 2 for central difference
    else
        fprintf('\tPadding displacement prior to calculating acceleration to account for differentiation scheme only.\n');
        diffOpt.tempKernel = 2; % central difference smoothing kernel
    end
else
    diffOpt.tempKernel = 0;
end

padFrames = diffOpt.tempKernel;

% pad frames with values at the start and end of the array
disp.tSmooth.x = padarray(disp.tSmooth.x,[0,0,padFrames],smoothingOpts.tempPadMethod,'pre'); 
disp.tSmooth.y = padarray(disp.tSmooth.y,[0,0,padFrames],smoothingOpts.tempPadMethod,'pre');

% calculate velocity and acceleration
[~,accel] = func_calcFFAccelFromDisp(time,disp.tSmooth,diffOpt.method);

% crop back to original size
accel.x = accel.x(:,:,padFrames+1:end);
accel.y = accel.y(:,:,padFrames+1:end);
disp.tSmooth.x = disp.tSmooth.x(:,:,1+padFrames:end);
disp.tSmooth.y = disp.tSmooth.y(:,:,1+padFrames:end);

%--------------------------------------------------------------------------
%% WIDTH AVG: Average displacement and strain over the width
fprintf('\tAveraging smoothed strain and raw displacement over the width.\n')
disp.xAvg = func_avgFFVarOverWidth(disp.tSmooth.x);         

%--------------------------------------------------------------------------
% WIDTH AVG: Temporal Smoothing of Width Averaged Data
fprintf('\tTemporally smoothing width averaged displacement.\n')
if smoothingOpts.WATempSmooth == true
    frameSize = smoothingOpts.WATemporalKernal(1);
    polyOrder = smoothingOpts.WATemporalKernal(2);
    
    if smoothingOpts.WATemporalPad
        padFrames = ceil(frameSize/2);
        dispTemporalPadX = padarray(disp.xAvg,[0,padFrames],smoothingOpts.tempPadMethod,'pre');
        disp.tSmooth.xAvg = sgolayfilt(dispTemporalPadX,polyOrder,frameSize,[],2);
        clear dispTemporalPadX
        disp.tSmooth.xAvg = disp.tSmooth.xAvg(:,(padFrames+1):end);
    else
        disp.tSmooth.xAvg = sgolayfilt(disp.xAvg,polyOrder,frameSize,[],2);
    end
end

%--------------------------------------------------------------------------
%% WIDTH AVG: Calculate the Width Averaged Acceleration
fprintf('\tTemporally differentiating width averaged data to obtain acceleration.\n')
if smoothingOpts.WATempSmooth == true
    [~,accel.xAvg] = func_calcAccelFromDisp(time,disp.tSmooth,diffOpt);
else
    [~,accel.xAvg] = func_calcAccelFromDisp(time,disp,diffOpt);
end

end