%% Create Processing Parameters Data Structure
% Authors: Lloyd Fletcher, Jared Van-Blitterswyk
% PhotoDyn Group, University of Southampton
% Date: 15/2/2018

clc
clear all
close all

fprintf('--------------------------------------------------------------\n')
fprintf('INITIALISE PROCESSING PARAMETERS: IBII Test Processing\n')
fprintf('--------------------------------------------------------------\n')

%% Create the paths for saving the processing parameters
fprintf('Specifying paths to save processing parameter file.\n')
%{
masterPath = [pwd,'\'];
specimens = {'S1','S2','S3','S12','S32','S33','S34'};
for s = 1:length(specimens)
    savePaths{s} = [masterPath,specimens{s},'\'];
end
%}
savePaths = {pwd};
saveFile = 'processingParameters.mat';

%% INITALISE - Test Data Structures
%//////////////////////////////////////////////////////////////////////////
fprintf('Creating data structures for processing constants.\n')

% Information about the test for the readme file
test.description = 'CFTT_13';
test.specNum = 1;               % specimen number
test.impactVel = 55;            % impact speed [m/s]
test.wgLength = 50;             % length of waveguide [mm]
test.projLength = 15;           % length of projectile [mm]
test.projMaterial = 'Al6061T6';

%--------------------------------------------------------------------------
% Global Processing Options - Controls program behaviour
globalOpts.imageDef = false;     % Allows addition of noise to images
globalOpts.smoothingOn = false;  % Turns smoothing on/off, useful for processin image def data with no noise
% Possible material models: 'isotropic', 'orthotropicReduced', 'orthotropic', 'orthotropicAngle'
globalOpts.matModel = 'orthotropicReduced';
% Which kinematic field components are plotted and calculated
globalOpts.fieldComponents = 'all'; % 'all' plot/calc x,y,s components or 'xOnly' 
% Control use of legacy edge extrapolation method 
globalOpts.useOldEdgeMethod = false;
globalOpts.hardCodeFreeEdge = false;
% Flags controlling which parts of the post-processing are run
globalOpts.plotImageSeqs = 'prompt';    % Plot videos or not, Options: 'yes' 'no' 'prompt' 
globalOpts.processOptVF = true;     % Run piecewise optimised VF 
globalOpts.identifyStrength = false; % Run strength indentification code
globalOpts.calcEBal = false;        % Run full-field energy balance code
globalOpts.plotPresentationImages = false;  % Plot videos for presentations
globalOpts.autoSaveProcData = true; % Automatically save processed data without prompt

%--------------------------------------------------------------------------
% Time Data Structure
% NOTE: frame rate is used for acceleration calculation
time.frameRate = 5e6;
time.step = 1/time.frameRate;
time.cutFrames = 5; % Frames to cut from the end of the accel/stress to avoid numerical diff effects
time.numFrames = 128;

%--------------------------------------------------------------------------
% Material Data Structure - Nominal Target Properties
% NOTE: density is used for stress calculation
material.name = 'CFTT_13';
material.rho = 1605;
material.Exx = 10e9;
material.nuxy = 0.015;
material.nuyx = 0.225;
material.rotAngle = 0;
if strcmp(globalOpts.matModel,'orthotropic') || strcmp(globalOpts.matModel,'orthotropicR')
    material.Q = 1/(1-material.nuxy*material.nuyx).*...
            [material.Exx  material.nuyx*material.Exx 0;...
             material.nuxy*material.Eyy material.Eyy 0;...
             0 0 material.Gxy*(1-material.nuxy*material.nuyx)];
    material.Q11 = material.Q(1,1);
    material.Q12 = material.Q(1,2);
    material.Q22 = material.Q(2,2);
    material.Q66 = material.Q(3,3);
else
    material.Q = material.Exx/(1-material.nuxy^2).*...
                [1 material.nuxy 0;...
                 material.nuxy 1 0;...
                 0 0 (1-material.nuxy)/2];
    material.Qxx = material.Q(1,1);
    material.Qxy = material.Q(1,2);
end

% Specimen Data Struct - Nominal Geometry
specimen.length = 18e-3;
specimen.height = 11e-3;
specimen.thickness = 3e-3;
specimen.volume = specimen.length*specimen.height*specimen.thickness;
specimen.mass = specimen.volume*material.rho;
specimen.freeEdge = 'Left';

% Grid Data Struct
grid.name = '0337mm_7pxpp_Print';
grid.pitch = 0.337e-3;
grid.pxPerPeriod = 7;
grid.rotAngle = 0;  % Used to rotate displacements for misaligned grids
grid.length = specimen.length;
grid.height= specimen.height;
grid.mPerPx = grid.pitch/grid.pxPerPeriod;
grid.numXPeriods = grid.length/grid.pitch;
grid.asymmPitch = true;
grid.pitchX = 0.335998e-3;
grid.pitchY = 0.337916e-3;
%grid.pitchX = 0.337916e-3;
%grid.pitchY = 0.335998e-3;

% Struct to control how the grid images are processed
gridMethodOpts.windowFlag = 1;      % 0 = gauss, 1 = bi-triang
gridMethodOpts.windowWidth = 1;     % Multiplier for grid pitch to set window width
gridMethodOpts.dispCalcMethod = 2;  % 1) Calculate using phase subtraction or 2) iterative method
gridMethodOpts.temporalUnwrap = true;
gridMethodOpts.debug  = true; % Plots diagnostic figures to debug errors
gridMethodOpts.autoLoadProccessedDataFile = false;
gridMethodOpts.hardCodeRotAngle = true;
gridMethodOpts.correctDefects = false;
gridMethodOpts.cameraSlave = false;
gridMethodOpts.padPhaseMaps = true; % false to use old method
gridMethodOpts.gmPadMethod = 'linear'; % can be set to 'constant' as well

%--------------------------------------------------------------------------
% Image Noise Structure - Used for adding noise to image def simulations
if globalOpts.imageDef
    imageNoise.addNoise = false;
    imageNoise.pcNoise = 0.35;
    imageNoise.bits = 16;
    imageNoise.convToUInt16 = true;
else
    imageNoise.addNoise = false;
end

%--------------------------------------------------------------------------
% Slice Data Structure - only needs to be created for orthotropicAngle
if strcmp('orthotropicAngle',globalOpts.matModel)
    slice.angle = material.rotAngle;
    slice.length = abs(specimen.height/sind(slice.angle));
    slice.lengthX = abs(slice.length*cosd(slice.angle));
    slice.xMin = 0; 
    slice.xMax = specimen.length - slice.lengthX;
    slice.xMinInd = 1;
    slice.returnComps = [0,1,1]; % Return the 22 and 12 components only
    slice.returnSliceVals = false; % Returns interpolated slice values in 3D array
end

%% INITALISE - Post-Processing Options
% Create smoothing and edge extrapolation options data structures
fprintf('Creating data structures for smoothing and extrapolation options.\n')

% -------------------------------------------------------------------------
% Differentiation Options
% Selects method for numerical differentiation
diffOpts.method = 'gradient'; % Options 'gradient', 'cDiff'
diffOpts.temporalPad = false;
diffOpts.temporalPadFrames = 3;
diffOpts.temporalPadMethod = 'replicate';

%--------------------------------------------------------------------------
% Smoothing Options 
if globalOpts.smoothingOn
    smoothingOpts.spatialSmooth = true;
    smoothingOpts.WATempSmooth = true;
    smoothingOpts.FFTempSmooth = true;
else
    smoothingOpts.spatialSmooth = false;
    smoothingOpts.WATempSmooth = false;
    smoothingOpts.FFTempSmooth = false;
end
% Descriptors
smoothingOpts.strainMethod = 'Spatial smooth only (on disp) then diff in space.';
smoothingOpts.WAAccelAvgMethod = 'Average over width first, temporal smooth only (on disp), then double diff in time.';
smoothingOpts.FFAccelMethod = 'Temporal smooth full-field disp, then spatial smooth full-field disp before double diff in time';
% Spatial Smoothing Options
smoothingOpts.spatialFilt = 'gauss';
% For gauss filter kernal length is specified giving: KernLeng = 2*ceil(2*sigma)+1
% NOTE: an asymmetric kernal can be specified: [LKernX,LKernY]
smoothingOpts.spatialKernal = [51,51]; % Specify kernal size in px, must be odd
smoothingOpts.spatialEdgeMode = 'symmetric';
smoothingOpts.padEdgeMode = 'linear';
% Width Averaged Temporal Smoothing Options
smoothingOpts.WATemporalFilt = 'sgolay';
smoothingOpts.WATemporalKernal = [11,3];
smoothingOpts.WATemporalPad = false; 
smoothingOpts.WATemporalPadFrames = 3;
smoothingOpts.WATemporalPadMethod = 'replicate';
% Full-Field Temporal Smoothing Options
smoothingOpts.FFTemporalFilt = 'sgolay';
smoothingOpts.FFTemporalKernal = [11,3];
smoothingOpts.FFTemporalPad = false;
smoothingOpts.FFTemporalPadFrames = 3;
smoothingOpts.FFTemporalPadMethod = 'replicate';

%--------------------------------------------------------------------------
% Data Extrapolation Options
% Fix NaNs in the displacement field
extrapOpts.fixNaNs = true;
extrapOpts.fixNaNKernal = 5*grid.pxPerPeriod;    % Radius for averaging to replace NaNs in px 

% Extrapolation over the length to replace missing pitches
extrapOpts.extrapWidthAvgData = true;
extrapOpts.dispPx = grid.pxPerPeriod;
if smoothingOpts.spatialSmooth
    extrapOpts.strainPx = grid.pxPerPeriod+floor(smoothingOpts.spatialKernal(1)/2);
else
    extrapOpts.strainPx = grid.pxPerPeriod;
end
extrapOpts.dispMethod = 'linear';
extrapOpts.strainMethod = 'linear';

% Extrapolation options for the full-field data used with the VFM
extrapOpts.extrapFullFieldData = true;
extrapOpts.FFDispPx = grid.pxPerPeriod;
if smoothingOpts.spatialSmooth
    extrapOpts.FFStrainPx = grid.pxPerPeriod+floor(smoothingOpts.spatialKernal(1)/2);
else
    extrapOpts.FFStrainPx = grid.pxPerPeriod;
end
extrapOpts.FFDispMethod = 'linear';
extrapOpts.FFStrainMethod = 'linear';

% Options for new edge padding method
extrapOpts.edgeMethod = 'NReg';
extrapOpts.imageDef = false;

if ~globalOpts.useOldEdgeMethod
    if strcmp(extrapOpts.edgeMethod,'NQuad')
        extrapOpts.padEdgeMethod = 'quadratic'; % options: quadratic, regression (linear fit), linear (default - interp1)
        extrapOpts.padFitWindow = grid.pxPerPeriod; % number of pixels to use for extrapolation fitting (quadratic and regression options only)
    elseif strcmp(extrapOpts.edgeMethod,'NReg')
        extrapOpts.padEdgeMethod = 'regression'; % options: quadratic, regression (linear fit), linear (default - interp1)
        extrapOpts.padFitWindow = grid.pxPerPeriod; % number of pixels to use for extrapolation fitting (quadratic and regression options only)
    else
        extrapOpts.padEdgeMethod = 'linear'; % options: quadratic, regression (linear fit), linear (default - interp1)
        extrapOpts.padFitWindow = grid.pxPerPeriod; % number of pixels to use for extrapolation fitting - not used for default MATLAB extrap
    end
end 

%--------------------------------------------------------------------------
% Stiffness Identification Options

% Stress-Gauge Stiffness Identification Options
% Possible choices: 
% 'full' - fits whole ss curve 
% 'componly' - fits up to the max compressive stress
% 'tensonly' - fits up to the max tensile stress
% 'maxabs' - fits up to the absolute max stress tensile/compressive
stressGaugeOpts.fitRangeOpt = 'componly';
stressGaugeOpts.limParam = 'stress';
stressGaugeOpts.cutPoints = 0;
stressGaugeOpts.fitDefaultRange = 1:time.numFrames;
stressGaugeOpts.robustFit = false;
stressGaugeOpts.avgQVsLRangePc = [0.25,0.75];
stressGaugeOpts.strainCalcNuxy = 'VFOpt'; % VFOpt - use average from VFO, VFMan - average from VFMan , QS - take nu from material struct 

% Virtual Fields Method Options
VFOpts.startFrame = 5;
VFOpts.endFrame = time.numFrames;
VFOpts.cutImpactEdge = true;
VFOpts.cutEdgePx = grid.pxPerPeriod;
VFOpts.avgQVsTRange = 25:60;
VFOpts.nElemX = 5;
VFOpts.nElemY = 1;

%--------------------------------------------------------------------------
% Energy Balance Options
% [LHS(xx),LHS(xy),RHS(xx),RHS(xy),TOP(xx),TOP(xy),BOT(xx),BOT(xy)]
EBalOpts.sv = [0,0,1,0,0,0,0,0]; % Defines the edge stresses used for the impact energy calc
EBalOpts.calcExtWorkFromAccel = true;
EBalOpts.cutPxX = grid.pxPerPeriod;
EBalOpts.cutPxY = grid.pxPerPeriod;
EBalOpts.impactEnerCutOff = false;
EBalOpts.impactEnerCutOffFrame = round(12e-6/time.step);
EBalOpts.impEdgeXForceOnly = true;
EBalOpts.maxErrRange = 1:time.numFrames;

%--------------------------------------------------------------------------
% Strength Identification Options

% Specify the method for determining the strength
% 'Manual' - allows user to specify strength frame
% 'Automated' - takes the maximum value of the stress gauge in the virtual gauge area
strOpts.method = 'Automated';
% Size of the virtual gauge for strength identification in pixels
strOpts.virtualGauge.Xpx = 20;
strOpts.virtualGauge.Ypx = 30;
% Specify which identified stiffness to to use for stress calc from strain
% 'Avg' - average of VF and SG, 'VF' - use VF value, 'SG' use stress gauge
strOpts.stiffnessMethod = 'SG';
strOpts.autoLoadFractureFile = true;
    
%--------------------------------------------------------------------------
% Plot Parameters Structure
plotParams.formatType = 'article';
plotParams.imageAxis = true;
plotParams.cutEdgePx = true;
plotParams.cutPxX = grid.pxPerPeriod+round(smoothingOpts.spatialKernal(1)/2);
plotParams.cutPxY = grid.pxPerPeriod;
% Options: 'Specified','Auto','MaxQuantileOverCompOnly','MaxQuantile'
plotParams.cAxisType = 'MaxQuantile';

plotParams.cAxisDisp = [-0.06,0.06];
plotParams.cAxisAccel = [-7e6,7e6];
plotParams.cAxisStrain = [-4,4];
plotParams.cAxisRawStrain = [-10,10];
plotParams.cAxisStrainRate = [-1500,1500];
plotParams.cAxisStress = [-2000,1500];

%//////////////////////////////////////////////////////////////////////////

%% INITALISE - Save all initialisation paramters
fprintf('Saving processing parameters file(s).\n')
for i = 1:length(savePaths)
    save(fullfile(savePaths{i},saveFile))
end
fprintf('COMPLETE.\n')

