% Script to produce the Iterated Ray method traces in Figure 5, single
% anistropic layer sandwiched between two isotropic layers.
%
% Original code: Jordan Gill
% Modified by: Giampaolo D'Alessandro
% 06/11/20
%
% If you use this code you are requested to reference this paper:
%
% J.R. Gill, E. Perivolari, M. Kaczmarek and G. D'Alessandro
% Efficient scattering model of multi-layer systems with anisotropic films
% J. Opt. Soc. Am. A (2021)
% DOI: https://doi.org/10.1364/JOSAA.416265
%
% Licensing
% =========
%
% **License**: CC BY-NC-SA
%
% This program is free software: you can redistribute it and/or modify it
% under the terms of the CC BY-NC-SA license.  This license lets you remix,
% tweak, and build upon this work non-commercially, as long as you credit
% us  and license your new creations under the identical terms.
%
% More info on the license type is available at:
%
% https://creativecommons.org/licenses/by-nc-sa/4.0/
%
% Please contact dales@soton.ac.uk if you want to use this code commercially.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

% Refractive index of the input isotropic layer
n_in = 1;
% Ordinary and extraordinary refractive indices of the anisotropic layer
n_o = 1.5;
n_e = 5;
% Refractive index of the output isotropic layer.
n_out = 2;

% Wave number in vacuum (in meters)
lambda = 632.8e-9; 
k0 = 2*pi/lambda;

% Range of angle of incidence and corresponding tangential component of the
% wave vector.  The wave vector is k=k0*[kx,0,kz].
n_psi = 100;
psi_max = 60;
psi = (pi/180)*linspace(-psi_max,psi_max,n_psi)';
kx = -n_in*sin(psi);

% Thickness of the anisotropic layer (in meters)
d_ani = 2*lambda;
% Polar and azimuthal angle of the optical axis
% The plane of incidence is the xz plane and the layers are perpendicular to
% the z axis.
% theta - polar angle of the optical axis (angle made between the x axis
% and the projection of the optical axis onto the xz plane)
theta = pi/5;
% phi - azimuthal angle of the optical axis (angle made between the optical
% axis and the y axis).
phi = pi/3;

%% Compute all the transmission and reflection coefficients
[rpp,rps,rsp,rss,tpp,tps,tsp,tss] = ...
  ani_layer(n_in,kx,k0,n_e,n_o,n_out,theta,phi,d_ani);

%% Plot the outputs

% Plot parameters
FontName='Times';
FontSize=10;
FontAngle='normal'; %'italic';
LineWidth=2;
MarkerSize=12;  % In points (72 points per inch)

% Longitudinal wave number in the output medium
q_out=sqrt(n_out^2-kx.^2);

% Plot of the transmittance and reflectance.

% The transmittance is the ratio of the magnitude of the normal component
% of the Poynting vectors.  This is abs(tss)^2 (or equivalent) multiplied 
% by a geometrical correction factor.  For the reflected field the
% geometrical correction factor is 1.

figure(1); clf;
% set(gcf,'PaperOrientation','Landscape','PaperType','tabloid',...
%   'PaperUnits','normalized','PaperPosition',[0 0 1 1]);
set(gcf,'PaperType','<custom>', ...
  'PaperUnits','centimeters','PaperSize',[20,6],...
  'PaperPosition',[0 0 20 6]);
axx=axes('Parent',gcf,...
  'YMinorTick','on','XMinorTick','on',...
  'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);

ax1=subplot(1,5,1,...
  'YMinorTick','on','XMinorTick','on',...
  'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
hold(gca,'all');
box(gca,'on');
plot(psi/pi*180,q_out./(n_in*cos(psi)).*abs(tss).^2,'--','linewidth',LineWidth);
plot(psi/pi*180,abs(rss).^2,'--','linewidth',LineWidth);
title('s to s');
xlim([-psi_max,psi_max]);
ylim([0,1]);
xlabel('$\theta_{1}$ [deg]','interpreter','latex');
ylabel('Amplitude');

ax2=subplot(1,5,2,...
  'YMinorTick','on','XMinorTick','on',...
  'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
hold(gca,'all');
box(gca,'on');
plot(psi/pi*180,q_out./(n_in*cos(psi)).*abs(tsp).^2,'--','linewidth',LineWidth);
plot(psi/pi*180,abs(rsp).^2,'--','linewidth',LineWidth);
title('s to p');
xlim([-psi_max,psi_max]);
ylim([0,1]);
xlabel('$\theta_{1}$ [deg]','interpreter','latex');
ax2.YTickLabel='';

ax3=subplot(1,5,3,...
  'YMinorTick','on','XMinorTick','on',...
  'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
hold(gca,'all');
box(gca,'on');
plot(psi/pi*180,q_out./(n_in*cos(psi)).*abs(tpp).^2,'--','linewidth',LineWidth);
plot(psi/pi*180,abs(rpp).^2,'--','linewidth',LineWidth);
title('p to p');
xlim([-psi_max,psi_max]);
ylim([0,1]);
xlabel('$\theta_{1}$ [deg]','interpreter','latex');
ax3.YTickLabel='';

ax4=subplot(1,5,4,...
  'YMinorTick','on','XMinorTick','on',...
  'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
hold(gca,'all');
box(gca,'on');
plot(psi/pi*180,q_out./(n_in*cos(psi)).*abs(tps).^2,'--','linewidth',LineWidth);
plot(psi/pi*180,abs(rps).^2,'--','linewidth',LineWidth);
title('p to s');
xlim([-psi_max,psi_max]);
ylim([0,1]);
xlabel('$\theta_{1}$ [deg]','interpreter','latex');
ax4.YTickLabel='';

subplot(1,5,5)
plot(0,0,  0,0,  0,0,  0,0,'linewidth',LineWidth)
axis off
legend({'T: Ray','R: Ray'},'box','off',...
  'FontSize',round(1*FontSize),'FontName',FontName,'FontAngle',FontAngle)