function C = ani_fresnel_in(coeffs,n1,K,qo,Eo,qe,Ee)
% calculates the Fresnel coefficients for light incident on a uniaxial
% crystal from an isotropic medium.  
%
%input:
%       - coeffs - array of strings - the coefficients to be returned,
%         e.g., 'rsp' gives the reflection coefficient for the p-wave for
%         an incident s-wave.
%       - n1 - scalar - refractive index of incident (isotropic) medium
%       - K - column vector - tangential components of wavevectors
%       - qo - column vector - normal wavevector components of o-waves
%       - qe - column vector - normal wavevector components of e-waves
%       - Eo - length(K) by 3 matrix - normalised E fields of the o-waves
%       - Ee - length(K) by 3 matrix - normalised E fields of the e-waves

%output: - C - structure - contains the coefficients specified by the coeffs
%              input, e.g., C.rsp is a column vector of the reflection
%              coefficients  of the p-waves for an incident s-wave.
%
% Original code: Jordan Gill
% 06/11/20
%
% If you use this code you are requested to reference this paper:
%
% J.R. Gill, E. Perivolari, M. Kaczmarek and G. D'Alessandro
% Efficient scattering model of multi-layer systems with anisotropic films
% J. Opt. Soc. Am. A (2021)
% DOI: https://doi.org/10.1364/JOSAA.416265
%
% Licensing
% =========
%
% **License**: CC BY-NC-SA
%
% This program is free software: you can redistribute it and/or modify it
% under the terms of the CC BY-NC-SA license.  This license lets you remix,
% tweak, and build upon this work non-commercially, as long as you credit
% us  and license your new creations under the identical terms.
%
% More info on the license type is available at:
%
% https://creativecommons.org/licenses/by-nc-sa/4.0/
%
% Please contact dales@soton.ac.uk if you want to use this code commercially.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

q1=sqrt(n1^2-K.^2);  %normal component of incident

A=(qo+n1^2./q1).*Eo(:,1)-K.*Eo(:,3);
B=(qe+n1^2./q1).*Ee(:,1)-K.*Ee(:,3);
D=(q1+qe).*A.*Ee(:,2)-(q1+qo).*B.*Eo(:,2);

for i=1:length(coeffs)
  if coeffs{i}=='rss'
    C.rss=((q1-qe).*A.*Ee(:,2)-(q1-qo).*B.*Eo(:,2))./D;
  end
  if coeffs{i}=='rsp'
    C.rsp=-2*n1.*(A.*Ee(:,1)-B.*Eo(:,1))./D;
  end
  if coeffs{i}=='rpp'
    C.rpp=-2*n1^2./q1.*((q1+qe).*Eo(:,1).*Ee(:,2)-(q1+qo).*Ee(:,1).*Eo(:,2))./D+1;
  end
  if coeffs{i}=='rps'
    C.rps=2*n1.*(qe-qo).*Eo(:,2).*Ee(:,2)./D;
  end
  if coeffs{i}=='tso'
    C.tso=-2*q1.*B./D;
  end
  if coeffs{i}=='tse'
    C.tse=2*q1.*A./D;
  end
  if coeffs{i}=='tpo'
    C.tpo=2*n1.*(q1+qe).*Ee(:,2)./D;
  end
  if coeffs{i}=='tpe'
    C.tpe=-2*n1.*(q1+qo).*Eo(:,2)./D;
  end
end    