function C = ani_fresnel_out(coeffs,K,n_out,qo_f,Eo_f,qo_b,Eo_b,qe_f,Ee_f,qe_b,Ee_b)
% Calculates the Fresnel coefficients for waves incident on a
% uniaxial/isotropic interface given the angles and fields of the forwards
% (incident) and backwards (reflected) o- and -e waves
%
%input:
%       - coeffs - array of strings - the coefficients to be returned,
%         e.g., 'reo' gives the reflection coefficient for the o-wave for
%         an incident e-wave 
%       - n_out - scalar - refractive index of exit(isotropic) medium
%       - K - column vector - tangential components of wavevectors
%       - qo_f/qo_b - column vector - normal wavevector components of the forward/backward o-waves
%       - qe_f/qe_b - column vector - normal wavevector components of the forward/backward e-waves
%       - Eo_f/Eo_b - length(K) by 3 matrix - normalised E fields of the forward/backward o-waves
%       - Ee_f/Ee_b - length(K) by 3 matrix - normalised E fields of the forward/backward e-waves
%
%output: - %output: - C - structure - contains the coefficients specified by the coeffs
%                          input, e.g., C.roe is a column vector of the reflection coefficients 
%                          of the e-waves for an incident o-wave
%
% Original code: Jordan Gill
% 06/11/20
%
% If you use this code you are requested to reference this paper:
%
% J.R. Gill, E. Perivolari, M. Kaczmarek and G. D'Alessandro
% Efficient scattering model of multi-layer systems with anisotropic films
% J. Opt. Soc. Am. A (2021)
% DOI: https://doi.org/10.1364/JOSAA.416265
%
% Licensing
% =========
%
% **License**: CC BY-NC-SA
%
% This program is free software: you can redistribute it and/or modify it
% under the terms of the CC BY-NC-SA license.  This license lets you remix,
% tweak, and build upon this work non-commercially, as long as you credit
% us  and license your new creations under the identical terms.
%
% More info on the license type is available at:
%
% https://creativecommons.org/licenses/by-nc-sa/4.0/
%
% Please contact dales@soton.ac.uk if you want to use this code commercially.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

q_out=sqrt(n_out^2-K.^2);  %normal component of exit wave

A=n_out^2./q_out;
B=Ee_b(:,2).*(q_out-qe_b).*(Eo_b(:,1).*(A-qo_b)+K.*Eo_b(:,3))...
  -Eo_b(:,2).*(q_out-qo_b).*(Ee_b(:,1).*(A-qe_b)+K.*Ee_b(:,3));

for i=1:length(coeffs)
  if coeffs{i}=='roo'
    C.roo=(Eo_f(:,2).*(q_out-qo_f).*(Ee_b(:,1).*(A-qe_b)+K.*Ee_b(:,3))...
      -Ee_b(:,2).*(q_out-qe_b).*(Eo_f(:,1).*(A-qo_f)+K.*Eo_f(:,3)))./B;
  end
  if coeffs{i}=='roe'
    C.roe=(Eo_b(:,2).*(q_out-qo_b).*(Eo_f(:,1).*(A-qo_f)+K.*Eo_f(:,3))...
      -Eo_f(:,2).*(q_out-qo_f).*(Eo_b(:,1).*(A-qo_b)+K.*Eo_b(:,3)))./B;
  end
  if coeffs{i}=='ree'
    C.ree=(Eo_b(:,2).*(q_out-qo_b).*(Ee_f(:,1).*(A-qe_f)+K.*Ee_f(:,3))...
      -Ee_f(:,2).*(q_out-qe_f).*(Eo_b(:,1).*(A-qo_b)+K.*Eo_b(:,3)))./B;
  end
  if coeffs{i}=='reo'
    C.reo=(Ee_f(:,2).*(q_out-qe_f).*(Ee_b(:,1).*(A-qe_b)+K.*Ee_b(:,3))...
      -Ee_b(:,2).*(q_out-qe_b).*(Ee_f(:,1).*(A-qe_f)+K.*Ee_f(:,3)))./B;
  end
  if coeffs{i}=='tos'
    C.tos=(Eo_f(:,2).*(Ee_b(:,2).*(qo_f-qe_b).*(Eo_b(:,1).*(A-qo_b)+K.*Eo_b(:,3))...
      -Eo_b(:,2).*(qo_f-qo_b).*(Ee_b(:,1).*(A-qe_b)+K.*Ee_b(:,3)))...
      +Ee_b(:,2).*Eo_b(:,2).*(qe_b-qo_b).*(Eo_f(:,1).*(A-qo_f)+K.*Eo_f(:,3)))./B;
  end
  if coeffs{i}=='top'
    C.top=(Eo_b(:,2).*(q_out-qo_b).*(-Ee_b(:,1).*Eo_f(:,1).*(qo_f-qe_b)...
      +K.*(Ee_b(:,1).*Eo_f(:,3)-Eo_f(:,1).*Ee_b(:,3)))+Ee_b(:,2).*(q_out-qe_b)...
      .*(Eo_b(:,1).*Eo_f(:,1).*(qo_f-qo_b)+K.*(Eo_b(:,3).*Eo_f(:,1)-Eo_b(:,1).*Eo_f(:,3)))...
      +Eo_f(:,2).*(q_out-qo_f).*(Ee_b(:,1).*Eo_b(:,1).*(qo_b-qe_b)...
      +K.*(Eo_b(:,1).*Ee_b(:,3)-Ee_b(:,1).*Eo_b(:,3))))./(B.*q_out/n_out);
  end
  if coeffs{i}=='tes'
    C.tes=(Ee_f(:,2).*(Ee_b(:,2).*(qe_f-qe_b).*(Eo_b(:,1).*(A-qo_b)+K.*Eo_b(:,3))...
      -Eo_b(:,2).*(qe_f-qo_b).*(Ee_b(:,1).*(A-qe_b)+K.*Ee_b(:,3)))...
      +Ee_b(:,2).*Eo_b(:,2).*(qe_b-qo_b).*(Ee_f(:,1).*(A-qe_f)+K.*Ee_f(:,3)))./B;
  end
  if coeffs{i}=='tep'
    C.tep=(Ee_b(:,2).*(q_out-qe_b).*(Ee_f(:,1).*Eo_b(:,1).*(qe_f-qo_b)...
      +K.*(Ee_f(:,1).*Eo_b(:,3)-Eo_b(:,1).*Ee_f(:,3)))+Eo_b(:,2).*(q_out-qo_b)...
      .*(-Ee_b(:,1).*Ee_f(:,1).*(qe_f-qe_b)+K.*(Ee_f(:,3).*Ee_b(:,1)-Ee_f(:,1).*Ee_b(:,3)))...
      +Ee_f(:,2).*(q_out-qe_f).*(Ee_b(:,1).*Eo_b(:,1).*(qo_b-qe_b)...
      +K.*(Eo_b(:,1).*Ee_b(:,3)-Ee_b(:,1).*Eo_b(:,3))))./(B.*q_out/n_out);
  end
end