function [rpp,rps,rsp,rss,tpp,tps,tsp,tss] = ...
  ani_layer(n_in,kx,k0,ne,no,n_out,theta,phi,L)
%Calculates the transmission and reflection coefficients  for a plane wave
%incident on an anisotropic layer using the Iterated Ray Method.
%
%The plane of incidence is the xz plane and the layers are perpendicular to
%the z axis.
% 
%input: - n_in - scalar - refractive index of incident (isotropic) layer
%       - kx - column vector - tangential component of the wavevectors k=k0*[kx,0,kz]
%       - k0 - scalar - wavenumber in vacuum
%       - ne - scalar - extraordinary refractive index
%       - no - scalar - ordinary refractive index
%       - n_out - scalar - refractive index of exit (isotropic) layer
%       - theta - scalar - polar angle of optical axis (angle made between
%                          the x axis and the projection of the optical axis 
%                          onto the xz plane)
%       - phi - scalar - azimuthal angle of optical axis (angle made
%                        between the optical axis and the y axis)
%       - L - scalar - thickness of the anisotropic layer
%
%output: - rps - column vector - reflection coefficient of the s polarised
%                                wave for an incident p polarised wave
%        - tsp - column vector - tranmission coefficient of the p polarised 
%                                wave for an incident s polarised wave 
%        - In general the notation is:
%          First letter: 'r' or 't' for reflection and transmission
%          coefficient respectively.
%          Second letter: polarisation of the incident wave,'s' or 'p'.
%          Third letter: polarisation of the outgoing ray, 's' or 'p'.
%          All the outputs are column vectors of the same length as kx.
%
% Original code: Jordan Gill
% 06/11/20
%
% If you use this code you are requested to reference this paper:
%
% J.R. Gill, E. Perivolari, M. Kaczmarek and G. D'Alessandro
% Efficient scattering model of multi-layer systems with anisotropic films
% J. Opt. Soc. Am. A (2021)
% DOI: https://doi.org/10.1364/JOSAA.416265
%
% Licensing
% =========
%
% **License**: CC BY-NC-SA
%
% This program is free software: you can redistribute it and/or modify it
% under the terms of the CC BY-NC-SA license.  This license lets you remix,
% tweak, and build upon this work non-commercially, as long as you credit
% us  and license your new creations under the identical terms.
%
% More info on the license type is available at:
%
% https://creativecommons.org/licenses/by-nc-sa/4.0/
%
% Please contact dales@soton.ac.uk if you want to use this code commercially.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

% Calculate the waves in the layer
[qo_f,Eo_f,qo_b,Eo_b,qe_f,Ee_f,~,qe_b,Ee_b,~] = ani_propagation(kx,ne,no,theta,phi);

% Compute the air/crystal interface coefficients
coeffs={'rpp','rps','rsp','rss','tpo','tpe','tso','tse'};
in = ani_fresnel_in(coeffs,n_in,kx,qo_f,Eo_f,qe_f,Ee_f);
                            
% Compute the coefficients from crystal to exit medium (forward waves)
coeffs={'roo','reo','roe','ree','top','tep','tos','tes'};
f=ani_fresnel_out(coeffs,kx,n_out,qo_f,Eo_f,qo_b,Eo_b,qe_f,Ee_f,qe_b,Ee_b);

% coefficients from crystal to air (backward waves):
% - switch n_out for n_air=n_in
% - reverse the sign of KX 
% - switch the role of the forward and backward fields
%   - reverse sign of the q's (as positve and negative z have switched)
%   - need to rotate fields around y axis by pi

coeffs={'roo','reo','roe','ree','top','tos','tep','tes'}; 
b=ani_fresnel_out(coeffs,-kx,n_in,-qo_b,[-Eo_b(:,1),Eo_b(:,2),-Eo_b(:,3)],...
                                  -qo_f,[-Eo_f(:,1),Eo_f(:,2),-Eo_f(:,3)],...
                                  -qe_b,[-Ee_b(:,1),Ee_b(:,2),-Ee_b(:,3)],...
                                  -qe_f,[-Ee_f(:,1),Ee_f(:,2),-Ee_f(:,3)]);
                          
%phase shifts
exp_delta_t_oo=exp(1i*2*k0*L*qo_f);
exp_delta_t_eo=exp(1i*k0*L.*(qo_f-qe_b));
exp_delta_t_oe=exp(1i*k0*L.*(qe_f-qo_b));
exp_delta_t_ee=exp(1i*k0*L.*(qe_f-qe_b));

exp_delta_r_oo=exp(1i*2*k0*L*qo_f);
exp_delta_r_eo=exp(1i*k0*L.*(qe_f-qo_b));
exp_delta_r_oe=exp(1i*k0*L.*(qo_f-qe_b));
exp_delta_r_ee=exp(1i*k0*L.*(qe_f-qe_b));

%calculate transmitted field
tsp=zeros(length(kx),1);
tss=zeros(size(tsp));
tpp=zeros(size(tsp));
tps=zeros(size(tsp));
rpp=zeros(size(tsp));
rps=zeros(size(tsp));
rsp=zeros(size(tsp));
rss=zeros(size(tsp));

for i=1:length(kx)
  %reflection matrix for transmission
  r=[f.roo(i).*b.roo(i).*exp_delta_t_oo(i)+f.roe(i).*b.reo(i).*exp_delta_t_eo(i),...
    f.reo(i).*b.roo(i).*exp_delta_t_oo(i)+f.ree(i).*b.reo(i).*exp_delta_t_eo(i);...
    f.roe(i).*b.ree(i).*exp_delta_t_ee(i)+f.roo(i).*b.roe(i).*exp_delta_t_oe(i),...
    f.ree(i).*b.ree(i).*exp_delta_t_ee(i)+f.reo(i).*b.roe(i).*exp_delta_t_oe(i)];
    
  %transmission matrix into exit medium
  t_out=[f.top(i) , f.tep(i); f.tos(i) , f.tes(i)];
  
  %input vector
  E0=[in.tpo(i)*exp(1i*k0*L*qo_f(i)) , in.tso(i)*exp(1i*k0*L*qo_f(i));
    in.tpe(i)*exp(1i*k0*L*qe_f(i)) , in.tse(i)*exp(1i*k0*L*qe_f(i))];
  
  %transmitted field
  t=t_out*(-r+eye(2))^-1*E0;
  tpp(i)=t(1,1);
  tps(i)=t(2,1);
  tsp(i)=t(1,2);
  tss(i)=t(2,2);
  
  %reflection matrix for reflection
  r=[b.roo(i).*f.roo(i).*exp_delta_r_oo(i)+b.roe(i).*f.reo(i).*exp_delta_r_eo(i),...
    b.reo(i).*f.roo(i).*exp_delta_r_oo(i)+b.ree(i).*f.reo(i).*exp_delta_r_eo(i);...
    b.roe(i).*f.ree(i).*exp_delta_r_ee(i)+b.roo(i).*f.roe(i).*exp_delta_r_oe(i),...
    b.ree(i).*f.ree(i).*exp_delta_r_ee(i)+b.reo(i).*f.roe(i).*exp_delta_r_oe(i)];
  
  %transmission matrix back into air
  t_out=[b.top(i) , b.tep(i); b.tos(i) , b.tes(i)];
  
  %input vector
  E0=[in.tpo(i).*f.roo(i).*exp_delta_r_oo(i)+in.tpe(i).*f.reo(i).*exp_delta_r_eo(i),...
    in.tso(i).*f.roo(i).*exp_delta_r_oo(i)+in.tse(i).*f.reo(i).*exp_delta_r_eo(i);...
    in.tpe(i).*f.ree(i).*exp_delta_r_ee(i)+in.tpo(i).*f.roe(i).*exp_delta_r_oe(i),...
    in.tse(i).*f.ree(i).*exp_delta_r_ee(i)+in.tso(i).*f.roe(i).*exp_delta_r_oe(i)];
  
  R=[in.rpp(i) , in.rsp(i); in.rps(i) , in.rss(i)]+t_out*(-r+eye(2))^-1*E0;
  rpp(i)=R(1,1);
  rps(i)=R(2,1);
  rsp(i)=R(1,2);
  rss(i)=R(2,2);
end