function [qe_f,qe_b,neff_f,neff_b]=e_wave(theta,phi,ne,no,kx)
%calculate the normal wavevector component of the extraordinary waves and 
%the effective refractive index.
%
%input:
%       - ne - scalar - extraordinary refractive index
%       - no - scalar - ordinary refractive index
%       - theta - scalar - polar angle of optical axis (angle made between
%                          the x axis and the projection of the optical axis 
%                          onto the xz plane)
%       - phi - scalar - azimuthal angle of optical axis (angle made
%                        between the optical axis and the y axis)
%
%output: - qe_f/qe_b - column vector - normal wavevector components of the 
%                                      forwards and backwards e-waves 
%        - neff_f/neff_b - column vector - effective refractive indices of
%                                          the forwards and backwards e-waves
%
% Original code: Jordan Gill
% 06/11/20
%
% If you use this code you are requested to reference this paper:
%
% J.R. Gill, E. Perivolari, M. Kaczmarek and G. D'Alessandro
% Efficient scattering model of multi-layer systems with anisotropic films
% J. Opt. Soc. Am. A (2021)
% DOI: https://doi.org/10.1364/JOSAA.416265
%
% Licensing
% =========
%
% **License**: CC BY-NC-SA
%
% This program is free software: you can redistribute it and/or modify it
% under the terms of the CC BY-NC-SA license.  This license lets you remix,
% tweak, and build upon this work non-commercially, as long as you credit
% us  and license your new creations under the identical terms.
%
% More info on the license type is available at:
%
% https://creativecommons.org/licenses/by-nc-sa/4.0/
%
% Please contact dales@soton.ac.uk if you want to use this code commercially.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

R=rot_mat(theta,phi);  %rotation matrix from lab to principal frame of the uniaxial crystal.

for i=find(kx==0)  %avoid divide by 0 error
    kx(i)=1e-12;
end

A=(R(1,3)^2+R(2,3)^2)/ne^2+R(3,3)^2/no^2;                       %coefficients of quadratic
B=kx*(2*((R(1,1)*R(1,3)+R(2,1)*R(2,3))/ne^2+R(3,1)*R(3,3)/no^2));
C=kx.^2*((R(1,1)^2+R(2,1)^2)/ne^2+R(3,1)^2/no^2)-1;

qe_f=zeros(size(kx));
qe_b=zeros(size(kx));
neff_f=zeros(size(kx));
neff_b=zeros(size(kx));
for i=1:length(kx)
    %solve quadratic for normalised kz component
    qe_b(i)=(-B(i)-sqrt(B(i)^2-4*A*C(i)))/(2*A);  
    qe_f(i)=(-B(i)+sqrt(B(i)^2-4*A*C(i)))/(2*A);  

    %effective refractive index
    neff_f(i)=sqrt(kx(i)^2+qe_f(i)^2);   
    neff_b(i)=sqrt(kx(i)^2+qe_b(i)^2);
end