% Script to produce the figure that checks the accuracy of the Iterated Ray
% Method against the S-Matrix method for a set of isotropic
% layers.  This is code used to produce figure 4 of the paper.
%
% Original code: Jordan Gill
% Modified by: Giampaolo D'Alessandro
% 06/11/20
%
% If you use this code you are requested to reference this paper:
%
% J.R. Gill, E. Perivolari, M. Kaczmarek and G. D'Alessandro
% Efficient scattering model of multi-layer systems with anisotropic films
% J. Opt. Soc. Am. A (2021)
% DOI: https://doi.org/10.1364/JOSAA.416265
%
% Licensing
% =========
%
% **License**: CC BY-NC-SA
%
% This program is free software: you can redistribute it and/or modify it
% under the terms of the CC BY-NC-SA license.  This license lets you remix,
% tweak, and build upon this work non-commercially, as long as you credit
% us  and license your new creations under the identical terms.
%
% More info on the license type is available at:
%
% https://creativecommons.org/licenses/by-nc-sa/4.0/
%
% Please contact dales@soton.ac.uk if you want to use this code commercially.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

%% Problem setup

% Refractive indices of the layers.  The first and last layers are
% semi-infinite, light enters from the first one and exits from the last
% one.
n=[2,1,2,2.5];
% Thickness of the layers in units of the vacuum wavelength.  The two
% semi-infinite layers have thickness 0.
L=[0,1.25,0.5,0];
% Maximum angle of incidence in degrees: the script computes the reflection
% and transmission coefficients for a range of incident angles between 0
% and psi_max.
psi_max = 60;
% Number of angle of incidence for which to compute the reflection and
% transmission coefficients.
ngp = 200;
% Angles of incidence and corresponding transverse wave numbers scaled to
% the vacuum wavenumber.
psi=linspace(0,psi_max,ngp)*pi/180;
kx=-n(1)*sin(psi);

%% Computation of the reflection and transmission coefficent
% Transmission and reflection coefficients using the Iterated Ray Method
% Note that we set the wavelength to 0 because the thickness of the layers
% is expressed in units of the wavelength.
[t14,~]=effective_coeffs(1,4,n,L,kx,'s');
[~,r13]=effective_coeffs(1,3,n,L,kx,'s');
% Transmission and reflection coefficients using the S-Matrix
S14=S_matrix(1,4,psi,n.^2,[1,1,1,1],L,2*pi);
T14=squeeze(S14(1,1,:)).';
S13=S_matrix(1,3,psi,n.^2,[1,1,1,1],L,2*pi);
R13=squeeze(S13(2,1,:)).';

%% Plot the output

% Plot parameters
FontName='Times';
FontSize=32;
FontAngle='normal'; %'italic';
LineWidth=2;
MarkerSize=12;  % In points (72 points per inch)

figure(1); clf;
set(gcf,'PaperOrientation','Landscape','PaperType','A4',...
  'PaperUnits','normalized','PaperPosition',[0 0 1 1]);
axes('Parent',gcf,...
  'YMinorTick','on','XMinorTick','on',...
  'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
hold(gca,'all');
box(gca,'on');
plot(psi/pi*180,abs(t14),'linewidth',LineWidth)
plot(psi/pi*180,abs(r13),'linewidth',LineWidth);
plot(psi/pi*180,abs(T14),'--','linewidth',LineWidth)
plot(psi/pi*180,abs(R13),'--','linewidth',LineWidth);
legend({'|t_{14}| - IRM','|r_{13}| - IRM',...
  '|t_{14}| - S-Matrix','|r_{13}| - S-Matrix'},...
  'box','off','location','east');
xlim([0,psi_max]);
ylim([0,1]);
xlabel('$\theta_{1}$ [deg]','interpreter','latex');
ylabel('Amplitude');
saveas(gcf,'SvsRayMultiLayer','pdf');

figure(2); clf;
set(gcf,'PaperOrientation','Landscape','PaperType','A4',...
  'PaperUnits','normalized','PaperPosition',[0 0 1 1]);
axes('Parent',gcf,...
  'YMinorTick','on','XMinorTick','on',...
  'FontSize',FontSize,'FontName',FontName,'FontAngle',FontAngle);
hold(gca,'all');
box(gca,'on');
plot(psi/pi*180,angle(t14),'linewidth',LineWidth)
plot(psi/pi*180,angle(r13),'linewidth',LineWidth);
plot(psi/pi*180,angle(T14),'--','linewidth',LineWidth)
plot(psi/pi*180,angle(R13),'--','linewidth',LineWidth);
legend({'arg(t_{14}) - IRM','arg(r_{13}) - IRM',...
  'arg(t_{14}) - S-Matrix','arg(r_{13}) - S-Matrix'},...
  'box','off','location','northeast');
set(gca,'YTick',...
  [-pi -pi/2 0 pi/2 pi],'TickLabelInterpreter','latex',...
  'YTickLabel',{'$-\pi$','$-\pi/2$','0','$\pi/2$','$\pi$'});
xlim([0,psi_max]);
ylim([-pi,pi]);
xlabel('$\theta_{1}$ [deg]','interpreter','latex')
ylabel('Phase [rad]');
saveas(gcf,'SvsRayMultiLayerPhase','pdf');