function S=S_matrix(i,j,psi,EPS,MU,LL,k)
% Calculates the effective Fresnel coefficients between layer i and j for a
% range of incidence angles using the S-matrix method.  Works only for s
% polarisation.
%
% Input parameters
% ----------------
%
% i,j (integers) - Indices of the two layers between which to compute the
%                  reflection and transmission coefficients.
% psi (vector)   - The angle of incidence in layer i.  If it has more
%                  than one element then the program computes the
%                  reflection and transmission coefficients for all the
%                  values of psi.
% EPS (vector)   - Vector of layer relative dielectric permittivities
% MU (vecotr)    - Vector of layer relative magnetic permeabilities.
% LL (vector)    - Vector of layer thicknesses.
% k (real)       - vacuum wave number in the same unit as LL.
% coeff (string) - Determines the type of output.  
%
% Output parameters
% -----------------
% S (array)      - The S-matrix for each angle of incidence.  Array of size
%                  (2,2,length(psi)).
%
% Original code: Jordan Gill
% Modified by: Giampaolo D'Alessandro
% 06/11/20
%
% If you use this code you are requested to reference this paper:
%
% J.R. Gill, E. Perivolari, M. Kaczmarek and G. D'Alessandro
% Efficient scattering model of multi-layer systems with anisotropic films
% J. Opt. Soc. Am. A (2021)
% DOI: https://doi.org/10.1364/JOSAA.416265
%
% Licensing
% =========
%
% **License**: CC BY-NC-SA
%
% This program is free software: you can redistribute it and/or modify it
% under the terms of the CC BY-NC-SA license.  This license lets you remix,
% tweak, and build upon this work non-commercially, as long as you credit
% us  and license your new creations under the identical terms.
%
% More info on the license type is available at:
%
% https://creativecommons.org/licenses/by-nc-sa/4.0/
%
% Please contact dales@soton.ac.uk if you want to use this code commercially.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

% Extract the parameters of the layers of interest, i.e. those between i
% and j, with both ends included.
eps=EPS(i:j);
mu=MU(i:j);
L=LL(:,i:j);

% Create S matrix for each angle of incidence
S=zeros(2,2,length(psi));

for p=1:length(psi)
  %set S0 as the identity
  S(:,:,p)=eye(2,2);
  
  %calculate kx components in the layers
  kx=k*sqrt(EPS(1)*MU(1))*sin(psi(p));
  
  %calculate kz in each layer
  kz=sqrt(k^2*eps.*mu-kx^2);
  
  %recursively build S matrix at each layer
  for n=2:length(mu)
    
    %build M matrix
    r=mu(n-1)*kz(n)./(mu(n)*kz(n-1));
    M=0.5*[1+r,1-r;1-r,1+r];
    
    %build D matrix
    phase=exp(1i*kz(n)*L(n));
    D=[phase^-1,0;0,phase];
    
    %build interface matrix
    I=M*D;
    
    %build temporary copy of S at previous step
    s=S(:,:,p);
    
    %update components of S
    S(1,1,p)=s(1,1)/(I(1,1)-s(1,2)*I(2,1));
    S(1,2,p)=(s(1,2)*I(2,2)-I(1,2))/(I(1,1)-s(1,2)*I(2,1));
    S(2,1,p)=s(2,2)*I(2,1)*S(1,1,p)+s(2,1);
    S(2,2,p)=s(2,2)*I(2,1)*S(1,2,p)+s(2,2)*I(2,2);
  end
end