function [t,r]=effective_coeffs(i,j,n,L,KX,pol)
% Calculates the effective Fresnel coefficients between layer i and j for a
% range of incidence angles using the Iterated Ray Method formulae.
%
% Input parameters
% ----------------
% i,j (integers) - Indices of the two layers between which to compute the
%                  reflection and transmission coefficients.
% n (vector)     - Vector of layer refractive indices
% L (vector)     - Vector of layer thicknesses in units of the vacuum wave
%                  length.
% KX (vector)    - Tangential component of the wave vector of the input
%                  beam scaled to the vacuum wave number.  If it has more
%                  than one element then the program computes the
%                  reflection and transmission coefficients for all the
%                  values of KX, i.e. for all the equivalent incidence
%                  angles.
% pol (string)   - The polarisation of the input beam, can be either 's' or
%                  'p', for perpendicular or parallel to the incidence
%                  plane. 
%
% Output parameters
% -----------------
% t, r (vectors) - The transmission and reflection coefficients for each
%                  value of KX, i.e. of the angle of incidence.  They are
%                  column vectors of the same length as KX.
%
% Original code: Jordan Gill
% Modified by: Giampaolo D'Alessandro
% 06/11/20
%
% If you use this code you are requested to reference this paper:
%
% J.R. Gill, E. Perivolari, M. Kaczmarek and G. D'Alessandro
% Efficient scattering model of multi-layer systems with anisotropic films
% J. Opt. Soc. Am. A (2021)
% DOI: https://doi.org/10.1364/JOSAA.416265
%
% Licensing
% =========
%
% **License**: CC BY-NC-SA
%
% This program is free software: you can redistribute it and/or modify it
% under the terms of the CC BY-NC-SA license.  This license lets you remix,
% tweak, and build upon this work non-commercially, as long as you credit
% us  and license your new creations under the identical terms.
%
% More info on the license type is available at:
%
% https://creativecommons.org/licenses/by-nc-sa/4.0/
%
% Please contact dales@soton.ac.uk if you want to use this code commercially.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

% Set direction of travel
if i<j
    J=1;
else
    J=-1;
end

% Make sure vector of angles is a column vector
KX=reshape(KX,length(KX),1);

% Calculate the phase shift across the layers.
q=sqrt(n.^2-KX.^2);
exp_delta=exp(1i*2*pi*L.*q);

% Set first iteration of effective coeffs as  regular coeffs from j-J to j
[r,t]=fresnel_coeffs(KX,n(j-J),n(j),pol);

% Iterate the effective coefficient by repeatedly adding another layer to
% the previous effective coefficient
for k=j-2*J:-J:i 
  [R,T]=fresnel_coeffs(KX,n(k),n(k+J),pol);
  t=T.*t.*exp_delta(:,k+J)./(1+r.*R.*exp_delta(:,k+J).^2);
  r=(R+r.*exp_delta(:,k+J).^2)./(1+R.*r.*exp_delta(:,k+J).^2);
end