function [r,t]=fresnel_coeffs(KX,ni,nj,pol)
% Calculates the Fresnel coefficients between two layers of refractive
% index ni and nj for a beam of polarisation pol and a range of incidence
% angles.
%
% Input parameters
% ----------------
%
% KX (vector)    - Tangential component of the wave vector of the input
%                  beam in units of the vacuum wave number.  If it has more
%                  than one element then the program computes the
%                  reflection and transmission coefficients for all the
%                  values of KX, i.e. for all the equivalent incidence
%                  angles.
% ni (real)      - The refractive index of the input layer
% nj (real)      - The refractive index of the output layer
% pol (string)   - The polarisation of the input beam, can be either 's' or
%                  'p', for perpendicular or parallel to the incidence
%                  plane. 
%
% Output parameters
% -----------------
% t, r (vectors) - The transmission and reflection coefficients for each
%                  value of KX, i.e. of the angle of incidence.  They are
%                  column vectors of the same length as KX.
%
% Original code: Jordan Gill
% Modified by: Giampaolo D'Alessandro
% 06/11/20
%
% If you use this code you are requested to reference this paper:
%
% J.R. Gill, E. Perivolari, M. Kaczmarek and G. D'Alessandro
% Efficient scattering model of multi-layer systems with anisotropic films
% J. Opt. Soc. Am. A (2021)
% DOI: https://doi.org/10.1364/JOSAA.416265
%
% Licensing
% =========
%
% **License**: CC BY-NC-SA
%
% This program is free software: you can redistribute it and/or modify it
% under the terms of the CC BY-NC-SA license.  This license lets you remix,
% tweak, and build upon this work non-commercially, as long as you credit
% us  and license your new creations under the identical terms.
%
% More info on the license type is available at:
%
% https://creativecommons.org/licenses/by-nc-sa/4.0/
%
% Please contact dales@soton.ac.uk if you want to use this code commercially.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

qi=sqrt(ni^2-KX.^2);
qj=sqrt(nj^2-KX.^2);

if pol=='s'
    r=((qi-qj)./(qi+qj))';
    t=(2*qi./(qi+qj))';
else
    r=((nj/ni*qi-ni/nj*qj)./(nj/ni*qi+ni/nj*qj))';
    t=(2*qi./(nj/ni*qi+ni/nj*qj))';
end
r=r';
t=t';