"""
Created on Aug 2019

@author: Dhirendra Vaidya (dhirendra22121987@gmail.com)
"""
import numpy as np
import matplotlib.pyplot as plt
import sys
from scipy.optimize import least_squares
import pickle

class SchottkyLikeIVFit:
    def __init__(self, xdata, ydata, c=0.5):
        self.name = 'Schottky Like IV Fit'
        self.equation = 'ln(I) = A_prime + alpha*V**c/(kT)'
        self.xdata = xdata
        self.ydata = ydata
        self.c = c

    def analytical(self, V, APrime, alpha, T):
        k = 8.617e-5
        return APrime + alpha/(k*T)*V**self.c

    def error(self, p, T):
        return self.ydata - self.analytical(self.xdata, p[0], p[1], T)

    def fit(self, T, p_init= np.array([-17.5, 0.25])):
        results = least_squares(self.error, p_init, args=(T,))
        return results.x

class APhiBFit:
    def __init__(self, xdata, ydata, cT=2.0):
        self.name = 'Schottky IV \'A\' Fit'
        self.equation = 'ln(A*T**c)-phi_B/(kT)'
        self.xdata = xdata
        self.ydata = ydata
        self.cT = cT

    def analytical(self, Temps, A, phi_B):
        k = 8.617e-5
        return np.log(A*Temps**self.cT) - phi_B/(k*Temps)

    def error(self, p):
        return self.ydata - self.analytical(self.xdata, p[0], p[1])

    def fit(self, p_init = np.array([0.001, 0.6])):
        results = least_squares(self.error, p_init, bounds = [(0,0), (np.inf, np.inf)])
        return results.x

class StaticIVModel:
    def __init__(self, A, phi_B, alpha, c=0.5, cT=2.0):
        self.A = A
        self.phi_B = phi_B
        self.alpha = alpha
        self.c = c
        self.cT = cT
        self.kB = 8.617e-5

    def getIV(self, V, T):
        alpha = self.getalpha(T)
        return self.A*T**self.cT*np.exp(-(self.phi_B-alpha*V**self.c)/(self.kB*T))

    def getalpha(self, T):
        alpha_ = 0.0
        for pi, p_ in enumerate(self.alpha):
            N_ = len(self.alpha)
            alpha_ = alpha_ + p_*T**(N_-1-pi)
        return alpha_

    def getR2byR1(self, Temperatures, V=0.2):
        print(V)
        T1 = Temperatures[0]
        k = 8.617e-5
        R2byR1 = np.zeros(len(Temperatures))
        for Ti, T2 in enumerate(Temperatures):
            alpha_T1 = self.getalpha(T1)
            alpha_T2 = self.getalpha(T2)
            R2byR1[Ti] = (T1/T2)**self.cT*np.exp(((self.phi_B-alpha_T2*V**self.c)/(k*T2))-((self.phi_B-alpha_T1*V**self.c)/(k*T1)))
        return R2byR1
