import {Page} from "storyplacesauthoringlib/lib/models/Page";
import {StoryFunctionSet} from "storyplacesauthoringlib/lib/models/StoryFunction";
import {VariableReference} from "storyplacesauthoringlib/lib/models/VariableReference";
import {VariableScope} from "storyplacesauthoringlib/lib/schemas/multiplayer/VariableScopes";
import {
    StoryCondition,
    StoryConditionComparison,
    StoryConditionLogical
} from "storyplacesauthoringlib/lib/models/StoryCondition";
import {ComparisonOperand, ComparisonType} from "storyplacesauthoringlib/lib/schemas/multiplayer/ConditionSchema";
import {LogicalOperand} from "storyplacesauthoringlib/lib/schemas/storyplaces/ConditionSchema";
import {unlocks} from "./locking";

class CalligraphicBuilder<T> {

    protected nodes: Set<T> = new Set<T>();
    protected outboundEdges: Map<T,T[]> = new Map<T, T[]>();
    protected inboundNodes:Map<T, T[]> = new Map<T, T[]>();
    protected id: string;
    protected startNodes: T[];

    constructor(id: string) {
        this.id = id;
    }

    public startAt(...startNodes: T[]) {
        this.startNodes = startNodes;
        this.startNodes.forEach(node => this.nodes.add(node));
    }

    public connect(node1: T, node2: T) {
        this.nodes.add(node1);
        this.nodes.add(node2);

        let outbound = this.outboundEdges.get(node1) || [];
        this.outboundEdges.set(node1, outbound.concat(node2));

        let inbound = this.inboundNodes.get(node2) || [];
        this.inboundNodes.set(node2, inbound.concat(node1));
    }

    protected varRef(): VariableReference {
        return new VariableReference(VariableScope.shared, `calligraphic${this.id}`, "lastVisited");
    }

    private nodeWasLastVisitedConditions: Map<T, StoryCondition> = new Map<T, StoryCondition>();
    public nodeWasLastVisitedCondition(node: T): StoryCondition {
        let condition = this.nodeWasLastVisitedConditions.get(node);
        if(condition) { return condition;}

        condition = new StoryConditionComparison(
                   `hasVisited${node.toString()}`,
                   ComparisonOperand.EQUAL,
                   this.varRef(),
                   node.toString(),
                   ComparisonType.Variable,
                   ComparisonType.String
        );
        this.nodeWasLastVisitedConditions.set(node, condition);
        return condition;
    }

    public build(mapper: (node: T) => Page): Page[] {
        let calligraphicVarRef = this.varRef();

        return Array.from(this.nodes).map(node => {
           let page = mapper(node);

           let setVisited = new StoryFunctionSet(`set${page.id}Visited`, calligraphicVarRef, node.toString());
           page.functions.push(setVisited);

           let inboundNodes = this.inboundNodes.get(node) || [];
           let previousNodeVisitedConditions: StoryCondition[] = [];
           for(let inboundNode of inboundNodes) {
               previousNodeVisitedConditions.push(this.nodeWasLastVisitedCondition(inboundNode));
           }

           //Only return a new page if it's reachable - i.e, it has conditions OR it's the start node.
           //Create a condition, where if any of the dependent pages have been visited, we can visit this one.
           if (previousNodeVisitedConditions.length > 0) {
               let canVisitCurrentNodeCondition = new StoryConditionLogical(`canVisit${page.id}`, LogicalOperand.OR, previousNodeVisitedConditions);
               page.conditions.push(canVisitCurrentNodeCondition);

               return page;
           }

           if (this.startNodes.indexOf(node) > -1) {
               return page;
           }

           return null;
        }).filter(item => !!item);
    }
}

export class MultiplayerCallographicBuilder<T> extends CalligraphicBuilder<T> {
    public buildMultiplayer(mapper: (node: T) => Page[]): Page[] {
        let calligraphicVarRef = this.varRef();

        let result: Page[] = [];
        let setNowhereVisited = new StoryFunctionSet(`setNowhereVisited`, calligraphicVarRef, "");

        Array.from(this.nodes).forEach(node => {
           let pages = mapper(node);

           let decisionPage = pages[0];
           let readOnlyPage = pages[1];

           let setVisited = new StoryFunctionSet(`set${decisionPage.id}Visited`, calligraphicVarRef, node.toString());

           let hasOutboundEdges = !!this.outboundEdges.get(node);

           if (readOnlyPage) {
               unlocks([decisionPage], [readOnlyPage]);
               //Hacky fix for start node that leads nowhere.
               //Needs to be visitable without affecting the rest of the hypertext.
               if(hasOutboundEdges) {
                   //Need to visit both decision and read only to move on.
                   decisionPage.functions.push(setNowhereVisited);
                   readOnlyPage.functions.push(setVisited);
               }
           } else {
               if(hasOutboundEdges) {
                   decisionPage.functions.push(setVisited);
               }
           }


           let inboundNodes = this.inboundNodes.get(node) || [];
           let previousNodeVisitedConditions: StoryCondition[] = [];
           for(let inboundNode of inboundNodes) {
               previousNodeVisitedConditions.push(this.nodeWasLastVisitedCondition(inboundNode));
           }

           //Only return a new page if it's reachable - i.e, it has conditions OR it's the start node.
           //Create a condition, where if any of the dependent pages have been visited, we can visit this one.
           if (previousNodeVisitedConditions.length > 0) {
               let canVisitCurrentNodeCondition = new StoryConditionLogical(`canVisit${decisionPage.id}`, LogicalOperand.OR, previousNodeVisitedConditions);
               decisionPage.conditions.push(canVisitCurrentNodeCondition);

               return result.push(decisionPage, readOnlyPage);
           }

           if (this.startNodes.indexOf(node) > -1) {
               return result.push(decisionPage, readOnlyPage);
           }

           return null;
        });

        return result.filter(item => !!item);
    }
}

export default CalligraphicBuilder;