import {ExtendedStory} from "./extendedstory";
import {PageHint} from "storyplacesauthoringlib/lib/models/PageHint";
import {Role} from "storyplacesauthoringlib/lib/models/Role";
import {Chunk} from "./chunk";
import {MultiplayerCallographicBuilder} from "./Calligraphic";
import {Page} from "storyplacesauthoringlib/lib/models/Page";
import importFromTwison from "./import_twine";
import {StoryConditionComparison} from "storyplacesauthoringlib/lib/models/StoryCondition";
import {VariableReference} from "storyplacesauthoringlib/lib/models/VariableReference";
import {VariableScope} from "storyplacesauthoringlib/lib/schemas/multiplayer/VariableScopes";
import {StoryFunctionIncrement, StoryFunctionSet} from "storyplacesauthoringlib/lib/models/StoryFunction";
import {ComparisonOperand, ComparisonType} from "storyplacesauthoringlib/lib/schemas/multiplayer/ConditionSchema";
import {unlocks} from "./locking";
import * as fs from "fs";
import {nodeToPageInStory, parseNodeDescriptionStringIntoGroups} from "./import_content";
import {
    finaleIntroContentSarah,
    finaleIntroContentTodd,
    leaveWithSarahContent,
    sarahShootsToddContent,
    sarahShootsToddToddContent,
    stayWithToddContent,
    toddGivesSarahKeysContent,
    toddShootsSarahContent,
    toddShootsSarahSarahContent
} from "./finale_nodes";

export function addFinaleToStory(story: ExtendedStory, finaleChunk: Chunk) {
    let rookieRole: Role = story.roles.find(role => role.id == "Rookie");
    let mentorRole: Role = story.roles.find(role => role.id == "Mentor");

    function getPageByName(name: string): Page {
        return story.pages.find(page => page.name == name);
    }

    let finaleTwison = importFromTwison("./finale.twison");
    let finaleNodes = finaleTwison.nodes;
    let finaleStartNodes = {
      "1": finaleIntroContentTodd,
      "286": finaleIntroContentSarah
    };

    let heist_content = fs.readFileSync("./heist_content.txt", {encoding: 'utf8'});
    let nodeGroups = parseNodeDescriptionStringIntoGroups(heist_content);


    /*story.NewPage({
       name: "Skip to Finale",
       content: "",
       hint: new PageHint(""),
       functions: [finaleChunk.unlockFunction]
    });*/

    let pageIndex = new Map<string, Page[]>();

    let nodeCreationFunc = (nodeId: string) => {
        let existingPage = pageIndex.get(nodeId);
        if(existingPage) { return existingPage; }

        let nodeInfo = finaleNodes.get(nodeId);

        let namePerson = nodeInfo.tags.indexOf("Todd") > -1? "Todd" : "Sarah";
        let nameOtherPerson = namePerson == "Todd" ? "Sarah" : "Todd";

        let conditions =
            namePerson == "Todd"?
                [rookieRole.getIsRoleCondition()] :
                namePerson == "Sarah"?
                    [mentorRole.getIsRoleCondition()] :
                    [];

        let readOnlyConditions =
            namePerson == "Sarah"?
                [rookieRole.getIsRoleCondition()] :
                namePerson == "Todd"?
                    [mentorRole.getIsRoleCondition()] :
                    [];

        let isStartNode = nodeId in finaleStartNodes;

        let mainPage = new Page({
            name: isStartNode? nodeInfo.name : `Say to ${nameOtherPerson}`,
            content: isStartNode? finaleStartNodes[nodeId] : nodeInfo.name,
            hint: new PageHint(isStartNode? "" : nodeInfo.name),
            conditions: conditions,
            singleVisit: true
        });

        let readOnlyPage = null;

        if(!isStartNode) {
            readOnlyPage = new Page({
                name: `${namePerson} speaks`,
                content: `${namePerson} says ${nodeInfo.name}`,
                hint: new PageHint(nodeInfo.name),
                conditions: readOnlyConditions,
                singleVisit: true
            });
        }

        pageIndex.set(nodeId, [mainPage, readOnlyPage]);
        return [mainPage, readOnlyPage];
    };

    let conversationHypertext = new MultiplayerCallographicBuilder<string>("FinaleConversation");

    conversationHypertext.startAt(...Object.keys(finaleStartNodes));

    finaleNodes.forEach(twisonNode => {
        if(twisonNode.tags.indexOf("Content-Reviewed") < 0) { return; }

        twisonNode.links.forEach(linkId => {
            if(finaleNodes.get(linkId)) {
                conversationHypertext.connect(twisonNode.id, linkId);
            }
        })
    });

    let pages = conversationHypertext.buildMultiplayer(nodeCreationFunc);

    pages.forEach(page => page.conditions.push(finaleChunk.isUnlockedCondition));

    story.pages.push(...pages);

    /* Add counters for number of finale nodes visited */

    let finalePagesVisitedCounterVariable = new VariableReference(VariableScope.shared, "finale", "pageCounter");

    let finalePagesVisitedIncrement = new StoryFunctionIncrement("IncrementFinalePagesRead", finalePagesVisitedCounterVariable, "1");

    pages.forEach(page => page.functions.push(finalePagesVisitedIncrement));

    /* Manually handle unique node behaviour */

    //The below is a hack to have a "You are waiting for the other player" node.
    //It is not pretty, and only works under the assumption that the conversation tree alternates roles.
    //That's because we have no way to check what nodes are currently available as part of a node condition.

    let varRefPlayerWaiting = new VariableReference(VariableScope.shared, "finale", "conversationToggle");
    let setMentorWaiting = new StoryFunctionSet("setMentorWaiting", varRefPlayerWaiting, "Mentor");
    let setRookieWaiting = new StoryFunctionSet("setRookieWaiting", varRefPlayerWaiting, "Rookie");
    let isMentorWaiting = new StoryConditionComparison("isMentorWaiting", ComparisonOperand.EQUAL, varRefPlayerWaiting, "Mentor", ComparisonType.Variable, ComparisonType.String);
    let isRookieWaiting = new StoryConditionComparison("isRookieWaiting", ComparisonOperand.EQUAL, varRefPlayerWaiting, "Rookie", ComparisonType.Variable, ComparisonType.String);

    //Set Mentor waiting at the start of the finale.
    pageIndex.get("1")[0].functions.push(setMentorWaiting);

    finaleNodes.forEach(node => {
        let pages = pageIndex.get(node.id);

        if (node.links && node.links.length > 0 && !(node.id in finaleStartNodes)) {
            if(node.tags.indexOf("Todd") > -1) {
                pages[0].functions.push(setRookieWaiting);
            } else if (node.tags.indexOf("Sarah") > -1) {
                pages[0].functions.push(setMentorWaiting);
            }

        }

    });

    let mentorWaitingPage = story.NewPage({
        name: "Todd is thinking about what to say.",
        content: "Todd is thinking about what to say. Todd will be able to speak once Sarah has spoken.",
        hint: new PageHint("Todd is thinking about what to say. Todd will be able to speak once Sarah has spoken."),
        conditions: [mentorRole.getIsRoleCondition(), finaleChunk.isUnlockedCondition, isMentorWaiting],
        singleVisit: false
    });

    let rookieWaitingPage = story.NewPage({
        name: "Sarah is thinking about what to say.",
        content: "Sarah is thinking about what to say. Todd will be able to speak once Sarah has spoken.",
        hint: new PageHint("Sarah is thinking about what to say. Todd will be able to speak once Sarah has spoken."),
        conditions: [rookieRole.getIsRoleCondition(), finaleChunk.isUnlockedCondition, isRookieWaiting],
        singleVisit: false
    });

    // None

    /* Ending logic */
    //These are doubled because of the read-only nodes.
    let visited5PagesCondition = new StoryConditionComparison("5PagesVisited", ComparisonOperand.GREATER_OR_EQUAL, finalePagesVisitedCounterVariable, "10", ComparisonType.Variable, ComparisonType.Integer);
    let visited10PagesCondition = new StoryConditionComparison("10PagesVisited", ComparisonOperand.GREATER_OR_EQUAL, finalePagesVisitedCounterVariable, "20", ComparisonType.Variable, ComparisonType.Integer);

    let toddShootsSarahTodd = story.NewPage({
        name: "> Make a decision: Shoot Sarah",
        content: toddShootsSarahContent,
        hint: new PageHint("Todd tries to draw his gun and shoot Sarah."),
        conditions: [rookieRole.getIsRoleCondition(), finaleChunk.isUnlockedCondition, visited5PagesCondition],
        functions: [finaleChunk.lockFunction]
    });

    let toddShootsSarahSarah = story.NewPage({
        name: "A gunshot rings out.",
        content: toddShootsSarahSarahContent,
        hint: new PageHint("Todd has shot Sarah."),
        conditions: [mentorRole.getIsRoleCondition()],
        singleVisit: true
    });

    unlocks([toddShootsSarahTodd], [toddShootsSarahSarah]);

    let sarahShootsTodd = story.NewPage({
        name: "> Make a decision: Shoot Todd",
        content: sarahShootsToddContent,
        hint: new PageHint("Sarah tries to draw her gun and shoot Todd."),
        conditions: [mentorRole.getIsRoleCondition(), finaleChunk.isUnlockedCondition, visited5PagesCondition],
        functions: [finaleChunk.lockFunction]
    });

    let sarahShootsToddTodd = story.NewPage({
        name: "A gunshot rings out.",
        content: sarahShootsToddToddContent,
        hint: new PageHint("Sarah has shot Todd."),
        conditions: [rookieRole.getIsRoleCondition()],
        singleVisit: true
    });

    unlocks([sarahShootsTodd], [sarahShootsToddTodd]);

    unlocks([pageIndex.get("1")[0]], [sarahShootsTodd, toddShootsSarahTodd])
    unlocks([pageIndex.get("286")[0]], [sarahShootsTodd, toddShootsSarahTodd])


    let sarahLeavesTodd = story.NewPage({
        name: "> Make a decision: Give Sarah the Keys",
        content: toddGivesSarahKeysContent,
        hint: new PageHint("Todd decides to let Sarah leave, giving her the keys to the car."),
        conditions: [rookieRole.getIsRoleCondition(), finaleChunk.isUnlockedCondition, visited5PagesCondition],
        functions: [finaleChunk.lockFunction]
    });

    let sarahLeavesSarah = story.NewPage({
        name: "Todd handed Sarah the keys.",
        content: toddGivesSarahKeysContent,
        hint: new PageHint("Todd gives in and allows Sarah to leave."),
        conditions: [mentorRole.getIsRoleCondition()],
        singleVisit: true
    });

    unlocks([sarahLeavesTodd], [sarahLeavesSarah]);

    let decideLeaveWithSarah = story.NewPage({
        name: "> Make a decision: Leave with Sarah",
        content: leaveWithSarahContent,
        hint: new PageHint("Todd decides that Sarah's right - their best chance is to leave, and offers to go with her."),
        conditions: [rookieRole.getIsRoleCondition(), finaleChunk.isUnlockedCondition, visited10PagesCondition],
        functions: [finaleChunk.lockFunction]
    });

    let decideLeaveWithSarahSarah = story.NewPage({
        name: "Todd agreed to go with Sarah.",
        content: leaveWithSarahContent,
        hint: new PageHint("Disarming his gun, Todd agreed to join Sarah, wherever she was headed."),
        conditions: [mentorRole.getIsRoleCondition()],
        singleVisit: true
    });

    unlocks([decideLeaveWithSarah], [decideLeaveWithSarahSarah]);

    let sarahStays = story.NewPage({
        name: "> Make a decision: Stay with Todd",
        content: stayWithToddContent,
        hint: new PageHint("Sarah decides to stay with Todd and face the dangers posed by the organisation together."),
        conditions: [mentorRole.getIsRoleCondition(), finaleChunk.isUnlockedCondition, visited5PagesCondition],
        functions: [finaleChunk.lockFunction]
    });

    let sarahStaysTodd = story.NewPage({
        name: "Sarah agreed to not leave",
        content: stayWithToddContent,
        hint: new PageHint("Sarah gave in, and decided to stay with Todd."),
        conditions: [rookieRole.getIsRoleCondition()],
        singleVisit: true
    });

    unlocks([sarahStays], [sarahStaysTodd]);

    let epilogueRawNodes = nodeGroups.get("Epilogue");
    let getEpiloguePage = (pageName: string) => {
        let rawNode = epilogueRawNodes.find(rawNode => rawNode.name == pageName);
        if(rawNode) {
            return nodeToPageInStory(rawNode, story);
        } else {
            console.warn("Unable to find epilogue node " + pageName);
        }
        return null;
    };

    let stayTodd = getEpiloguePage("Business as usual.");
    let staySarah = getEpiloguePage("And so it begins again.");
    let bothLeaveTodd = getEpiloguePage("Hiding in wait.");
    let bothLeaveSarah = getEpiloguePage("Finally, home.");
    let leaveTodd = getEpiloguePage("The meeting.");
    let leaveSarah = getEpiloguePage("The end of the road.");
    let toddShotSarahTodd = getEpiloguePage("The aftermath.");
    let toddShotSarahSarah = getEpiloguePage("On the run.");
    let sarahShotToddTodd = getEpiloguePage("Loyalty.");
    let sarahShotToddSarah = getEpiloguePage("Freedom?");


    unlocks([sarahStays], [staySarah]);
    unlocks([sarahStaysTodd], [stayTodd]);

    unlocks([decideLeaveWithSarah], [bothLeaveTodd]);
    unlocks([decideLeaveWithSarahSarah], [bothLeaveSarah]);

    unlocks([toddShootsSarahTodd], [toddShotSarahTodd]);
    unlocks([toddShootsSarahSarah], [toddShotSarahSarah]);

    unlocks([sarahShootsTodd], [sarahShotToddSarah]);
    unlocks([sarahShootsToddTodd], [sarahShotToddTodd]);

    unlocks([sarahLeavesTodd], [leaveTodd]);
    unlocks([sarahLeavesSarah], [leaveSarah]);


    return {
        chunk: finaleChunk,
        pages: pages
    }
}