import {Page} from "storyplacesauthoringlib/lib/models/Page";
import {ConditionReferenceOrDefinition, StoryCondition} from "storyplacesauthoringlib/lib/models/StoryCondition";
import {StoryFunctionReferenceOrDefinition} from "storyplacesauthoringlib/lib/models/StoryFunction";
import {PageTransition} from "storyplacesauthoringlib/lib/schemas/multiplayer/PageTransition";
import {PageHint} from "storyplacesauthoringlib/lib/models/PageHint";
import {alternativePages, unlocks} from "./locking";
import {Role} from "storyplacesauthoringlib/lib/models/Role";

export interface BuiltFlashback {
    name: string;
    choosingCharacter: string;
    prepPage1: Page;
    choicePages1: Page[];
    prepPage2: Page;
    choicePages2: Page[];
}

export function listBuiltFlashbackPages(flashback: BuiltFlashback): Page[] {
    return [flashback.prepPage1, flashback.prepPage2, ...flashback.choicePages1, ...flashback.choicePages2];
}

export class Flashback {
    constructor(
        public name: string,
        public choosingCharacter: string,
        public prep: FlashbackNode,
        public choices: FlashbackNode[]
    ) {}

    //Build a set of nodes for each character, then manually add the unlocks between them.
    buildPages(characters: Role[]): BuiltFlashback {
        let choosingCharacter = characters.find((char) => char.id == this.choosingCharacter);
        let otherCharacter = characters.find((char) => char.id != this.choosingCharacter);

        let prepPageChar1 = this.prep.buildPage({conditions: [choosingCharacter.getIsRoleCondition()]});
        let choicePagesChar1 = this.choices.map((node) =>
            node.buildPage({conditions: [choosingCharacter.getIsRoleCondition()]}
        ));

        let altPagesChar1 = alternativePages(choicePagesChar1);
        unlocks([prepPageChar1], altPagesChar1.pages)

        let prepPageChar2 = this.prep.buildPage({conditions: [otherCharacter.getIsRoleCondition()]});
        let choicePagesChar2 = this.choices.map((node) =>
            node.buildPage({conditions: [otherCharacter.getIsRoleCondition()]}
        ));

        unlocks([prepPageChar1], [prepPageChar2]);
        unlocks([prepPageChar2], choicePagesChar2);

        for(let pageChar1 of choicePagesChar1) {
           let matchingPageChar2 = choicePagesChar2.find((pageChar2) => pageChar2.name == pageChar1.name);
           unlocks([pageChar1], [matchingPageChar2]);
        }

        return {
            name: this.name,
            choosingCharacter: this.choosingCharacter,
            prepPage1: prepPageChar1,
            choicePages1: choicePagesChar1,
            prepPage2: prepPageChar2,
            choicePages2: choicePagesChar2
        }
    }
}

export interface FlashbackPageOptions {
  functions?: StoryFunctionReferenceOrDefinition[],
  conditions?: ConditionReferenceOrDefinition[],
  pageTransition?: PageTransition,
  singleVisit?: boolean
}

export class FlashbackNode {
    constructor(
        public name: string,
        public hint: string,
        public content: string
    ) {}

    buildPage(flashbackPageOptions: FlashbackPageOptions = {}): Page {
        return new Page({
            name: "Memory - " + this.name,
            hint: new PageHint(this.hint),
            content: this.content,
            functions: flashbackPageOptions.functions,
            conditions: flashbackPageOptions.conditions,
            pageTransition: flashbackPageOptions.pageTransition,
            singleVisit: flashbackPageOptions.singleVisit || true
        })
    }
}