import {promises} from 'fs';
import {Flashback, FlashbackNode} from "./flashback_import";
import {Role} from "storyplacesauthoringlib/lib/models/Role";
import {ExtendedStory} from "./extendedstory";
import {Page} from "storyplacesauthoringlib/lib/models/Page";
import {PageHint} from "storyplacesauthoringlib/lib/models/PageHint";

export class RawNodeData {
    flashback: string = "";
    nodeType: string = "";
    character: string = "";
    name: string = "";
    hint: string = "";
    content: string = "";
}

function parseNodeContent(content: string): RawNodeData[] {
    let lines = content.split(/\r?\n/);
    let nodeData = [];

    let currentNodeData: RawNodeData = null;

    lineLoop:
    for(let line of lines) {
        let lineSplitOnDelimiter = line.split(":");
        let key = lineSplitOnDelimiter[0].trim();

        switch(key) {
            case "COMMENT":
                continue;
            case "//END//":
                nodeData.push(currentNodeData);
                currentNodeData = new RawNodeData();
                break lineLoop;
            case "flashback":
                if(currentNodeData) {
                    nodeData.push(currentNodeData);
                }
                currentNodeData = new RawNodeData();
            case "nodeType":
            case "character":
            case "name":
            case "hint":
                currentNodeData[key] = lineSplitOnDelimiter.slice(1).join(":").trim();
                break;
            case "content":
                currentNodeData[key] = lineSplitOnDelimiter.slice(1).join(":").trimLeft();
                break;
            default:
                currentNodeData.content = currentNodeData.content + "\n" + line;
        }
    }

    return nodeData;
}

export function groupNodesByFlashback(nodes: RawNodeData[]): Map<string, RawNodeData[]> {
    let groupedNodes = new Map();
    for (let node of nodes) {
        let group = groupedNodes.get(node.flashback) || [];
        group.push(node);
        groupedNodes.set(node.flashback, group);
    }
    return groupedNodes;
}

class RawFlashbackData {
    character: string = "";
    prepNode: RawNodeData = null;
    choiceNodes: RawNodeData[] = [];
}

function rawNodeToFlashbackNode(rawNode: RawNodeData) {
    if(!rawNode) {
        return null;
    }
    return new FlashbackNode(
        rawNode.name,
        rawNode.hint,
        rawNode.content
    )
}

export function nodesToFlashback(nodes: RawNodeData[]) {
    let flashbackName = nodes[0].flashback;
    let flashbackCharacter = nodes[0].character;

    if(!nodes.every(node => node.flashback == flashbackName)) {
        throw `${flashbackName}: Trying to create flashback out of nodes with different flashback names`;
    }

    if(!nodes.every(node => node.character == flashbackCharacter)) {
        throw `${flashbackName}: Trying to create flashback out of nodes with different characters`;
    }

    let prepNodes = nodes.filter(node => node.nodeType.toLowerCase() == "prep");
    if(prepNodes.length != 1) {
        throw `${flashbackName}: Too few or too many prep nodes in flashback`;
    }

    let prepNode = prepNodes[0];
    let choiceNodes = nodes.filter(node => node.nodeType.toLowerCase() == "choice");

    return new Flashback(
        flashbackName,
        flashbackCharacter,
        rawNodeToFlashbackNode(prepNode),
        choiceNodes.map(rawNodeToFlashbackNode));
}

export function nodeToPageInStory(node: RawNodeData, story: ExtendedStory): Page {
    return story.NewPage({
        name: node.name,
        hint: new PageHint(node.hint),
        content: node.content,
        conditions: [story.roles.find(role => role.id == node.character).getIsRoleCondition()],
        singleVisit: true
    });
}

function nodeContentToFlashbacks(nodes: RawNodeData[]) {
    let rawFlashbacks: any = {};

    for(let node of nodes) {
        let flashback = rawFlashbacks[node.flashback] || new RawFlashbackData();

        if(!flashback.character) {
            flashback.character = node.character;
        }

        switch(node.nodeType) {
            case "prep":
                flashback.prepNode = node;
                break;
            default:
                flashback.choiceNodes.push(node);
        }

        rawFlashbacks[node.flashback] = flashback;
    }

    let flashbacks = [];
    for(let flashbackName in rawFlashbacks) {
        let rawFlashback = rawFlashbacks[flashbackName];
        flashbacks.push(new Flashback(flashbackName,
                                      rawFlashback.character,
                                      rawNodeToFlashbackNode(rawFlashback.prepNode),
                                      rawFlashback.choiceNodes.map(rawNodeToFlashbackNode)));
    }

    return flashbacks;

}

export function parseNodeDescriptionStringIntoFlashbacks(content: string): Flashback[] {
    return nodeContentToFlashbacks(parseNodeContent(content));
}

export function parseNodeDescriptionStringIntoGroups(content: string): Map<string, RawNodeData[]> {
    return groupNodesByFlashback(parseNodeContent(content));
}

async function main() {
    let content = await promises.readFile("./heist_content.txt", {encoding: 'utf8'});
    let rawNodes = parseNodeContent(content);
    let flashbacks = nodeContentToFlashbacks(rawNodes);
    console.log(flashbacks);
}
