import * as fs from "fs";

export interface Twison {
    startNode: string;
    nodes: Map<string, TwisonNode>;
}

export interface TwisonNode {
    id: string;
    name: string;
    content: string;
    tags: string[];
    links: string[];
}

export default function importFromTwison(pathToTwison: string): Twison {
    let fileContents = fs.readFileSync(pathToTwison, {encoding: 'utf8'});

    let twison = JSON.parse(fileContents);

    let nodes = new Map<string, TwisonNode>();

    twison.passages.forEach(twisonNode => {
       nodes.set(twisonNode["pid"], {
           id: twisonNode.pid,
           name: twisonNode.name,
           content: twisonNode.text,
           tags: twisonNode.tags || [],
           links: twisonNode.links ? twisonNode.links.map(link => link.pid) : []
       });
    });

    let removeIndirectLinks = twisonNode => {

        let oldLinks = twisonNode.links;

        //Zero out the link list, to prevent infinite loops when recursing.
        twisonNode.links = [];

        let directLinks = oldLinks.reduce((links, currentLink) => {
            let linkedNode = nodes.get(currentLink);
            if(!linkedNode) { return links; }
            if(linkedNode.tags.indexOf("Forwarding") < 0) { return links.concat(currentLink) }

            return links.concat(removeIndirectLinks(linkedNode));

        }, []);

        twisonNode.links = directLinks;
        return directLinks;
    };

    nodes.forEach(twisonNode => {
        removeIndirectLinks(twisonNode);
    });

    return {
        startNode: twison.startnode,
        nodes: nodes
    };
}