import {Page} from "storyplacesauthoringlib/lib/models/Page";
import {VariableReference} from "storyplacesauthoringlib/lib/models/VariableReference";
import {StoryFunction, StoryFunctionSet} from "storyplacesauthoringlib/lib/models/StoryFunction";
import {
    StoryCondition,
    StoryConditionCheck,
    StoryConditionComparison,
    StoryConditionLogical
} from "storyplacesauthoringlib/lib/models/StoryCondition";
import {VariableScope} from "storyplacesauthoringlib/lib/schemas/multiplayer/VariableScopes";
import {
    ComparisonOperand,
    ComparisonType,
    LogicalOperand
} from "storyplacesauthoringlib/lib/schemas/multiplayer/ConditionSchema";
import {Chunk, createChunk} from "./chunk";


let lockCount: number = 0;

function pageArrayDescription(pages: Page[]): string {
    return "[" + pages.map(page => page.id).join(", ") + "]";
}

export function locks(lockingPages: Page[], pagesToLock: Page[]): Chunk {
    lockCount += 1;

    let pagesToLockChunk = createChunk({
        pages: pagesToLock,
        scope: VariableScope.shared
    });

    lockingPages.forEach(page => {
        page.functions.push(pagesToLockChunk.lockFunction);
    });

    return pagesToLockChunk;
}

let unlockCount: number = 0;

export function unlocks(unlockingPages: Page[], pagesToUnlock: Page[]): Chunk {
    unlockCount += 1;

    let pagesToUnlockChunk = createChunk({
        pages: pagesToUnlock,
        scope: VariableScope.shared,
        startsLocked: true
    });

    unlockingPages.forEach(page => {
        page.functions.push(pagesToUnlockChunk.unlockFunction);
    });

    return pagesToUnlockChunk;
}

export let visitingAnyUnlocks = unlocks;

export interface AlternativePages {
    pages: Page[],
    functions: StoryFunction[],
    conditions: StoryCondition[]
}

export function alternativePages(pages: Page[], variable?: VariableReference, name?: string): AlternativePages {
    //Should be unique
    name = name || pageArrayDescription(pages);
    variable = variable || new VariableReference(VariableScope.shared, "alternativePages", name);
    let alternativePages = {pages: [], functions: [], conditions: []};
    let isSetCondition = new StoryConditionCheck("Alternate Node Chosen for " + name, variable);

    pages.forEach(page => {
        alternativePages.pages.push(page);
        let lockOthersFunction = new StoryFunctionSet("Disable All But " + page.name + " for " + name, variable, page.id);
        let pageWasNotChosenCondition = new StoryConditionComparison(
            "Is " + page.name + " selected for " + name,
            ComparisonOperand.NOT_EQUAL,
            variable,
            page.id,
            ComparisonType.Variable,
            ComparisonType.String);

        //Node is unavailable if Alternate Node Choice has occurred && this page wasn't chosen - so invert using NAND.
        let isAvailableCondition = new StoryConditionLogical(
            "Is " + page.id + " available for alternate node " + name,
            LogicalOperand.NAND,
            [isSetCondition, pageWasNotChosenCondition]
        );


        alternativePages.functions.push(lockOthersFunction);
        alternativePages.conditions.push(isAvailableCondition);

        page.functions.push(lockOthersFunction);
        page.conditions.push(isAvailableCondition);
    });

    return alternativePages;
}

export function visitingAllUnlocks(unlockingPages: Page[], pagesToUnlock: Page[]) {
    let name = `Visiting ${pageArrayDescription(unlockingPages)} unlocks ${pageArrayDescription(pagesToUnlock)}`;

    let chunks = unlockingPages.map(unlockingPage =>
        unlocks([unlockingPage], pagesToUnlock));

    return chunks;
}