

// Program : 1D_CNFD.cpp
//
// A program to simulate the dimensionless concentration profile
// and calculate dimensionless tip current at a macroelectrode.
//
// The simulation uses the Crank Nicholson Finite Difference method 
// and a one dimensional simulation domain and stops at T=1

// Last Updates 16 January 2000

#include <iostream.h>
#include <math.h>
#include <fstream.h>

#define XLIM 10.0
#define MAX_ARRAY 2000

ofstream outdat;

int main(void)
{
	int klim, ilim;				// number of steps in time and distance
	double delta_t, delta_x;	// size of time, space step
	double lamda;				// simulation parameter defined by (dT/dX*dX)

	double *pnew, *pold;		// pointers to new and old concentration arrays
	double cr1[MAX_ARRAY]; 		// concentration array 1
	double cr2[MAX_ARRAY]; 		// concentration array 2

	double var_a;				// CNFD grid parameter
	double var_b[MAX_ARRAY];	// CNFD explicit parameter
	double a_1[MAX_ARRAY], b_1[MAX_ARRAY];	// CNFD implicit parameters

	char banana[50];

	int i, j;					// integer counters for space, time

	// ask the user for input
	cout << "Please Input.. " << endl << " numt: ";
	cin >> klim;
	cout << " lamda: ";
	cin >> lamda;

	cout << endl <<  "Processing .." << endl;

	// delete this!!!
	outdat.open("current.dat");

	// calculate simulation parameters
	delta_t = 1.0 / (double)klim;
	delta_x = sqrt(delta_t/lamda);
	ilim = (int)(XLIM / delta_x); 
	var_a = (-2.0/lamda)*(lamda+1.0);
	
	a_1[ilim-1] = var_a;
	for (i=(ilim-2); i>=1; i--)
	{
		a_1[i] = var_a - (1.0/a_1[i+1]);
	}

	// assign the concentration array pointers
	pnew = cr2; pold = cr1;

	// initial boundary condition (T=0)
	for (i=0; i<=ilim; i++) pold[i] = 1.0;
	
	// simulation loop
	for (j=1; j<=klim ; j++)
	{

		// loop outwards to calculate the explicit parameter var_b
		for (i=1; i<=(ilim-1); i++)	
		{
			var_b[i] = ((2.0/lamda)*(lamda-1.0)*pold[i])-pold[i+1]-pold[i-1]; 
		}
	
		// at the edge of the domain, Cr = Cr(bulk) so..
		// apply outer edge boundary condition (X=Xlim)
		pnew[ilim] = 1.0;
		i = ilim-1;
		b_1[i] = var_b[i] - pnew[ilim];

		// loop inwards to calculate the implicit parameter b_1
		for (i=(ilim-2); i>=1; i--)	
		{
			b_1[i] = var_b[i] - (b_1[i+1]/a_1[i+1]);
		}
		
		// apply electrode boundary condition (X=0)
		pnew[0] = 0.0;
		
		// loop outwards to calculate the rest of concentrations within domain
		for (i=1; i<=(ilim-1); i++)
		{
			pnew[i] = (b_1[i] - pnew[i-1])/a_1[i] ;
		}

		// swap the old and new concentration arrays
		if ((j%2)!=0) { pnew = cr1; pold = cr2; }
		else { pnew = cr2; pold = cr1; }

		outdat << ((double)j*delta_t) << "\t" << (pold[1] - pold[0])/delta_x << endl;
		
	}

	// print dimensionless tip current and exit
	cout << " Dimensionless Tip Current at T=0 : " << (pold[1] - pold[0])/delta_x
		<< endl << " Bye! " << endl;

	outdat.close();

	return 1;

}

