
/*

PROGRAM: adi4.cpp
   
        ********************************************************
        *                                                      *
        *  PROGRAM: The simulation of the diffusion controlled *
        *  Faradaic current to a microdisc [during reduction]  *
        *  using Alternating Direction Implicit (ADI) methods  *
        *    and FULLY IMPLICIT 2 point boundary conditions    *
        *                                                      *
        ********************************************************

LAST UPDATE		: Thursday 08-05-1997
C++ version		: Tuesday 08-06-1999

*/
 
/*

Global Variables And Constants ....

version : version number
dro     : dr/do <user input>
fileDAT : output stream for tip current output
cob     : dimensionless bulk concentration at t=0
pold    : pointer to the old dimensionless concentration array
pnew    : pointer to the old dimensionless concentration array
lamda   : lamda value
dx      : size of dimensionless step in x and z directions
dt      : size of dimensionless step in time
numd   : steps considered along the disc <user input>
numg  : steps considered alond disc and glass <user input>
jgsd    : steps considered away from glass/electrode surface <user input>
numt    : number of kt's for 1 dimensionless time unit 
numtmax : kt from 1 to numtmax <numtmax = numt default>
map_points  : array of times that a concentration map is required <end in -ve>

*/

#include <iostream.h>
#include <fstream.h>
#include <functional>
#include <math.h>
#include <string.h>
#include <iomanip.h>

// constants

#define MAXNUMX 1000
#define MAXNUMY 1000
#define M_PI 3.14159265

/*--------Prototypes--------------------------- */

void main(void);
void welcome(void);
void opendatafile(void);
void createheader(void);
void adi_calc(void);
void adi_precalc(void);
void closedatafile(void);
void clearbulk(int a,int b,int c,int d);
void electrode(int a,int b,int c);
void perp(int a,int b,int c);
void parral(int a,int b,int c);

// --------Global Variables And Constants-------

// concentration matrices and pointers
double cr1[(MAXNUMX+2)][MAXNUMY+1];
double cr2[(MAXNUMX+2)][MAXNUMY+1];
double (*pnew)[MAXNUMX+1];
double (*pold)[MAXNUMY+1];

double a2[MAXNUMX+1], a3[MAXNUMX+1];
double b4[MAXNUMX+1];
double a_1[MAXNUMX+1],b_1[MAXNUMX+1];
double c2;
double c_1[MAXNUMY];
double d4[MAXNUMY];
double d_1[MAXNUMY];

// simulation variables
ofstream fileDAT;
double lamda,dx,dt,dro;
char fileout[20];
int numd,numg,jgsd,numt,numtmax;

// settings
char version[] = {"ADI v1.0i"};
double cob = 1.0;
double map_points[] = {0.01,0.02,0.03,0.04,0.1,1,-1};
char concfile[]="concX.smp";

/* ------------------------------------------- */
/* -----------STARTING MAIN PROGRAM----------- */

void main(void)
{

	cout << "\n\n\n\n\n\n";
	welcome();
	opendatafile();
	createheader();
	adi_precalc();
	adi_calc();
	closedatafile();

}

// -----------END OF MAIN PROGRAM-------------- 
// -------------------------------------------- 

// ---WELCOME---------------------------------- 
// Procedure greets user and asks for the inputs 
// required for the calculation. It also performs 
// basic checks on the validity of the input values 
// and calculates dx,dt,lam,lamd and kdta2 

void welcome(void)
{      
	start:

	cout << "\n\n\n\n"
		<< " A.D.I. MICRODISC SIMULATION \n"
		<< " version : " << version
		<< " ****************************** \n\n"
		<< " Please Input..\n\n"
		<< " D(R)/D(O): ";
	cin >> dro;
	cout << endl << " lamda: ";
	cin >> lamda;
	cout << endl << " numd: "; 
	cin >> numd;
	cout << endl << " numg: ";
	cin >> numg;
	cout << endl << " jgsd: ";
	cin >> jgsd;

	// Error checking
	if ((numg > MAXNUMX) || (jgsd > MAXNUMY)) 
	{
		cout << "\n****WARNING*****\n" << "Large matrix - reduce numg or jgsd \n";
		goto start;
	}
	if (numg <= numd) 
	{ 
		cout << "\n****WARNING*****\n" << "numg must be bigger than numd\n";
		goto start;
	}
	
	if ((numg*numd*jgsd==0)||(lamda*dro==0.0)) 
	{
		cout << "\n****WARNING*****\n" << "no zero values allowed\n";
		goto start;
	}

	dx = (1.0 / (double)numd);
	dt = (double)lamda * dx * dx;
	numt = (int)(1.0 / dt);

	cout << "\n\n number of time steps for dt/a2 = " << numt;
	cout << "\n input the total number of time steps : ";
	cin >> numtmax;
 
	// If numt value entered <= 0 then numt = numtmax
	if (numt <= 0)
	{
		cout << "\n*****WARNING******\n" << "Using default value for ktmax\n";
		numtmax = numt;
    }

	cout << "\n enter the data file name to save to : ";
	cin >> fileout;

	cout << "\n*****RUNNING PROGRAM****\n";

	// prepare for 'half time' steps
	dt = dt / 2.0; lamda = lamda / 2.0 ;

}      //welcome
 

// ---OPENDATAFILE----------------------------------
// opens data file for current storage

void opendatafile(void)
{      

	fileDAT.open(fileout);

}	// opendatafile      

// ----CLOSEDATAFILE----------------------------- 

void closedatafile(void)
{      

	fileDAT.close();
	cout << "\n*****FILE : " << fileout << "CLOSED****\n\n";

}	//closedatafile


// -----CREATEHEADER-----------------------------
void createheader(void)
{   
	
	fileDAT << "\nA.D.I. Microdisc Simulation\n"
			<< "    version: \n" << version
			<< "\nParameters Used: " << endl
			<< " dro: " << dro << " lamda: " << lamda << endl
			<< " numd: " << numd << " numg: " << numg << " jgsd: " << jgsd << endl
			<< " numt: " << numt << " numtmax: " << numtmax
			<< "\nResults To Follow..\n"
			<< "\n dt/a2\t  dlctip" << endl;
   
}      // createheader


// ---ADI_PRECALC-------------------------------
// calculates adi parameters that do not depend
// on time

void adi_precalc(void)
{

	// Variables ..
	// i, j     : integer steps in the R and Z directions resp.
	
	int i, j;

	double coeff_factor;

	// RADIAL PRELIM. CALCULATIONS
	// NOTE: a2, a3 and a_1 coefficients do not depend on Z
	// therefore it is simpler to calculate them beforehand
	// and pick them out of an array during a concentration calculation

	// work outwards to fill up the a2 and a3 arrays of coeffs
	for ( i = 1 ; i<= numg ; i++)
    {
		coeff_factor = lamda * ((1.0/(2.0 * (double)i)) - 1.0);
		a2[i] = (1.0 + (2.0 * lamda)) / coeff_factor;
		a3[i] = -1.0*(lamda*(1.0 + (1.0/(2.0*(double)i))))/coeff_factor;
    }

	// calculate last a_1 point      
	a_1[numg] = a2[numg];

	// fill up a_1 array working inwards, towards symmetry axis 
	for ( i = numg - 1 ; i >= 1 ; i--)
    {
		a_1[i] = a2[i] - ( a3[i] / a_1[i+1] );
    }

	// AXIAL PRELIM. CALCULATIONS

	// c2 is constant throughout r and z
	c2 = (-1.0)*(1.0 + (2.0 * lamda))/lamda;

	// first c_1 point
	c_1[jgsd-1] = c2;

	// fill c_1 array
	for ( j = jgsd-2 ; j>=1 ; j--)
    {
		c_1[j] = c2 - (1.0 / c_1[j+1]);
    }

} // adi_precalc

void adi_calc(void)
{
	// ifctr	- parameter to limit the number of concentrations calculated and printed
	// kt		- integer time counter
	// dim_time	- dimensionless time
	// sum		- used to sum tip current
	// fileMAP	- output stream for concentration map files
	// conclimitx, conclimity - used in the routine to limit map output size
	// imap		- number of cocntration maps printed so far
	// i, j		- integer counters
	// dlctip	- calculated dimensionless tip current

	double ifctr, dim_time, sum, dlctip;
	int kt, imap, i ,j;
	ofstream fileMAP;
	int conclimitx, conclimity;

	// no concentration maps printed as yet
	imap = 0;

	// set the approapriate matrix to old and new values
	pold = cr1;
	pnew = cr2;

	// clear the bulk solution
	clearbulk(0,numg+1,0,jgsd);

	// limit tip calculation points to 500 or below
	ifctr = numtmax/500;
	if (ifctr == 0) { ifctr = 1; }

	// ---------------
	// start time loop
	// ---------------

	for (kt = 1; kt<=numtmax ; kt++)
    {
		// FIRST HALF OF TIME STEP

		// set the approapriate matrix to old and new values
		pold = cr1;
		pnew = cr2;

		// implement easy boundary conditions */
		// radial border
		perp(numg+1,0,jgsd);
		// axial border
		parral(jgsd,0,numg);
		    
		// loop away from the electrode in j
        for ( j = 1 ; j<=(jgsd-1) ; j++ )
        {
			// fill up b4 array
            for (i = 1; i <= numg; i++)
            {
				b4[i] = ((lamda*pold[i][j-1]) + ((1.0 - (2.0 * lamda))*pold[i][j])
                        + (lamda*pold[i][j+1]))/((lamda/(double)(2.0*i)) - lamda);                         
            }                  

            // find outer b_1 value for each row
            b_1[numg] = b4[numg] - (a3[numg] * pnew[numg+1][j]); 
             
            // start backwards loop to fill b_1 array
            for ( i = numg-1 ; i>=1 ; i-- )
            {
				b_1[i] = b4[i] - ((a3[i]*b_1[i+1])/a_1[i+1]);
            } 

            // calculate new 'R' concentration using two point approx. at symm. axis 
            pnew[0][j] = b_1[1] / (1.0 + a_1[1]);

            // start outwards loop to calculate new concentrations      
            for ( i = 1 ; i <=numg ; i++ )
            {
				pnew[i][j] = (b_1[i] - pnew[i-1][j])/a_1[i];                         
            } 
                  
		} // end of rows (j) loop
     
        // apply electrode boundary condition
        electrode(0,0,numd);

        // calculate new concs on glass by using 2 point approx
        j = 0;
        for  (i = numd + 1; i <= numg ; i++)
        {
			pnew[i][j] = pnew[i][j+1];
        }
		
		// SECOND HALF OF TIME STEP

		// set the approapriate matrix to old and new values
		pold = cr2;
		pnew = cr1;
               
		// implement easy boundary conditions */
		// radial border
		perp(numg+1,0,jgsd);
		// axial border
		parral(jgsd,0,numg);

		// start outwards radially in i
        for ( i = 1 ; i<=numg ; i++ )
        {
			// fill up c2 array by looping away from the electrode
            for (j = 1; j <= jgsd-1; j++)
            {
				d4[j] = (((lamda - (lamda/(double)(2.0*i)))*pold[i-1][j])
						+ ((1-(2*lamda))*pold[i][j])
						+ ((lamda + (lamda/(double)(2.0*i)))*pold[i+1][j]))/((-1.0)*lamda);
            }

            // find outer b_1 value for each column
            d_1[jgsd-1] = d4[jgsd-1] - pnew[i][jgsd];
                                
            // start inwards loop to fill d_1 array
			for ( j = jgsd-2 ; j>=1 ; j--)
            {
				d_1[j] = d4[j] - (d_1[j+1]/c_1[j+1]);
			} 

			// test if column above glass or electrode and apply relevent b.c
            if (i <= numd) { pnew[i][0] = 0.0; }
            else { pnew[i][0] = d_1[1] / (1.0 + c_1[1]); }
                  
            // start outwards column loop for new concentrations     
            for ( j = 1 ; j <=jgsd-1 ; j++ )
            {
				pnew[i][j] = (d_1[j] - pnew[i][j-1])/c_1[j];
            }
                  
		} // end of column (i) loop

        // update remaining column - the symmetry axis
        pnew[0][0] = 0.0;
       	for (j = 1; j <= jgsd-1 ; j++)
        {    
			pnew[0][j] = pnew[1][j];
        }  
  
		// Result Calculation And Printing If Required
		if ((kt%(int)ifctr) == 0)
        {
			sum = 0.0;
			j = 0;
			for (i = 1; i<=numd; i++)
            {
                sum = sum + (pnew[i][j+1] - pnew[i][j]) * (double)i * dx;
            }

            dlctip = M_PI * sum / 2.0;
			dim_time = dt * 2.0 * (double)kt;

            // Print Answer To File
            fileDAT << dim_time << "\t" << dlctip << endl; 
        }
      
		// print conc profile if required
		if ((kt != 0) && ((kt%(int)(numtmax*map_points[imap])) == 0))
        {
			concfile[4]=(char)(imap+65);
            fileMAP.open(concfile);
            
			// routine to limit size of maps
            conclimitx = numg / 100;
            conclimity = jgsd / 100;
            if (conclimitx == 0) { conclimitx = 1; }
            if (conclimity == 0) { conclimity = 1; }
            
			// print out maps to output stream

            for (j = 0; j <= jgsd;)
            {
                for (i = 0; i<=(numg+1);)
                {
					fileMAP << pnew[i][j] << endl;
                    i+=conclimitx;
                }
				j+=conclimity;
            }
                
			fileMAP.close();
            imap=imap+1;
		} 
		
	} // end time loop

	cout << "\n**EXECUTION OF PROGRAM COMPLETE**\n";

} // adi_calc

/* ---CALCULATION FUNCTIONS---------------------- */

void clearbulk(int a,int b,int c, int d) // CLEARS BULK SOLUTION WITH BULK VALUE
{
	int i,j;

    for (i = a; i <= b; i++)
    {
        for (j = c; j <= d; j++)
        {
			pold[i][j] = cob;
            pnew[i][j] = 999.0;		// check that all these are written over
									// with realistic calculated values
        }
    }
} 
     
void electrode(int a,int b,int c)
{
	int i,j;
	
	j = a;
	for (i = b; i<=c; i++)
    {
		pnew[i][j] = 0.0;
    }
}

void perp(int a,int b,int c)
{
	int i,j;

    i = a;
    for (j = b; j<=c; j++)
    {
		pnew[i][j] = cob;
    }
}

void parral(int a,int b,int c)
{
	int i,j;

	j = a;
    for (i = b; i<=c; i++)
    {
		pnew[i][j] = cob;
    }
}

