

//    /adi13/winadi13.c
//
//    --------------------------
//    ADI - Microdisc Simulation
//		CONDUCTING SUBSTRATE
//    --------------------------
//
//        XY Expanding Grid
//  Implicit Boundary Conditions
//         Half Time Steps
//
//      Adapted From ADI7a/10
//
//

// Last Update		: Wednesday 3 February 1999
// Windows32 Version: Tuesday 2 February 1999
// Last AIX Update  : Friday 13 February 1998

// Globals And Constants

// maxnumx        -   Max Number Of Points in X(R) direction
// maxnumy        -   Max Number Of Points in Y(Z) direction

// Includes and Definitions

#include <iostream.h>
#include <fstream.h>
#include <functional>
#include <math.h>
#include <string.h>
#include <iomanip.h>

#undef MAXNUMX
#undef MAXNUMY
#undef M_PI
#undef idcode
#define MAXNUMX 2000
#define MAXNUMY 2000
#define M_PI 3.141592654
#define idcode -1

// Prototypes

void main(void);
double welcome(void);
void adi_init(double dt);
void adi_sum(void);
void clearbulk(int a,int b,int c,int d);
void electrode(int a,int b,int c);
void perp(int a,int b,int c);
void parral(int a,int b,int c);
double calc_the_current(void);
void map_the_conc(int bl, int br, int lb, int lt, double timepos, int timecount, int test);
void id_block(int a,int b,int c,int d);

// Global Variables and Constants 

char *version[] = {"CONDUC v win3.0"};
double cob = 1.0;
double mapping[] = {1E10};

char *filesmap[20]; 
double var_a,var_b,dx,dy,dr_t1,dz_m1,dz_m2,dr_m1,dz_b1,end_time,lamx,lamy,maxlam,topz,midz,depth;
int numd,numx,numxtot,numytot,numdepth,numbulk,numy,numy1,numg,eidisc,ejgsd;

// Global Calculation Arrays 
double lamda_y[MAXNUMY+1],x_exp1[MAXNUMX+2],x_exp2[MAXNUMX+2],coef_x[MAXNUMX+2];
double a2[MAXNUMX+1],a3[MAXNUMX+1],b4[MAXNUMX+1],a_1[MAXNUMX+1],b_1[MAXNUMX+1];
double c2[MAXNUMY],c3[MAXNUMY],d4[MAXNUMY],c_1[MAXNUMY],d_1[MAXNUMY];
double posr[MAXNUMX+2],posz[MAXNUMY+1];
double cu1[(MAXNUMX+2)][MAXNUMY+1],cofx[MAXNUMX+2],cofy[MAXNUMY+1];
double cu2[(MAXNUMX+2)][MAXNUMY+1]; 

double (*pnew)[MAXNUMX+1];
double (*pold)[MAXNUMY+1];

// MAIN 

void main(void)
{
	// Variables
	// dt            - dimensionless size of the time step
	// timepos       - dimensionless time position

	ofstream fileDAT;
	double dt,dt_old,dt_ref,timepos,ifctr,ifctrc,dlctip,dlctip_old;
	int current_count,map_count;

	dt = welcome();                // greets users.. returning initial dt value

	// opens current datafile
	fileDAT.open("current.dat");
	fileDAT.setf(ios::fixed, ios::floatfield);	// decimal notation
	fileDAT.setf(ios::showpoint);				// decimal point always shown
	fileDAT << setprecision(6);					// set num decimal places
	fileDAT << "time \t Itip" << endl;
     
	cout << "\n\n*****RUNNING PROGRAM****\n\n";

	// inititialising matrix for simulation
	pold = cu1;
	pnew = cu2;

	clearbulk(0,numxtot+1,0,numytot);
	id_block(0,numdepth-1,0,numx);
	timepos = 0.0; dt_old= 0.0; dlctip_old = 0.0;
	current_count=1; map_count = 0; dt_ref = dt;
	ifctrc = end_time/50000.0;
	ifctr = ifctrc;

	while (timepos <= end_time)
	{
		timepos = timepos + (2.0*dt);
		if (dt!=dt_old) { adi_init(dt); if (dt_old==0.0) { cout << endl << " RG: " << posr[numx+1] << endl; } }   /* initialises ADI coefficients at start or if dt changes */
		adi_sum();                          /* performs ADI calculation for variables from adi_init over 2dt*/

		if (timepos>=ifctr)                 /* calculates and prints current if necessary */
		{
			dlctip = calc_the_current();
			fileDAT << timepos << "\t" << dlctip << endl; 
			current_count=current_count+1;
			ifctr=ifctr+ifctrc;
        }
     
		if (timepos>=mapping[map_count])     /* print concentration map if approx time is in the array */
		{ 
			map_the_conc(0,numxtot+1,0,numytot,timepos,map_count,0); 
			map_count = map_count+1;
        }
          
		dt_old = dt;
      
		/*do we require a new value of dt?*/
		dlctip = calc_the_current(); 
		dt = dt_ref * (((int)timepos)+1);
		dlctip_old = dlctip;    
	}  // time loop

	map_the_conc(0,numxtot+1,0,numytot,timepos,map_count,1);
	cout << "\n\n*****SHUTTING DOWN PROGRAM****\n\n";
	fileDAT.close();
}


double welcome(void)
{      
	// User Inputs Simulation Parameters - returns dt
	// Variables
	// dt          - dimensionless time step size
	//

	double dt;
	int iteration,eiglass,eibulk,ejdepth;

thestart:

	cout << "\n\n\n\n\n" << "SECM MICRODISC SIMULATION\n"
		<< "to an conducting substrate\n" 
		<< "version: " << *version
		<< "\n USING ADI with IMPLICIT bc's\n";

	cout << "\nInput The Following Calculation Parameters\n"
		<< "\nRelative Dimensions Of Matrix\n\n";

	cout << "eidisc            : "; cin >> eidisc;
	cout << "eiglass           : "; cin >> eiglass;
	cout << "ejgsd             : "; cin >> ejgsd;
	cout << "eibulk            : "; cin >> eibulk;
	cout << "ejdepth           : "; cin >> ejdepth;
	if ((eidisc>eiglass)||(eibulk<eiglass)||(ejdepth<ejgsd)) { goto thestart; }

	cout << "\nShape Of Expansion Grid\n\n";

	cout << "numd              : "; cin >> numd;
	cout << "expansion A value : "; cin >> var_a;
	if ((numd<2)||(var_a<0)) { goto thestart; }

	cout << "\nTime Steps and Length\n\n";

	cout << "time to finish, tfin            : "; cin >> end_time;
	cout << "INITIAL iterations in 1ts, tts  : "; cin >> iteration;
	if ((iteration<1)||(end_time<0)) { goto thestart; }

	dt = 0.5 / (double)iteration;
	
	dx = log(1.0 + var_a)/(double)numd; 
	dy = dx;

	dr_m1 = (pow((1.0+var_a),1.0/(double)numd)-1.0)/var_a;

	dz_b1 = dr_m1;
	topz=(double)ejgsd/(double)eidisc;

	numg = (int)(log(1.0+(var_a*(((double)(eiglass)/(double)(eidisc))-1.0)))/dx);
	numbulk = (int)(log(1.0+(var_a*((eibulk/eidisc)-1.0)))/dx);
	numdepth = (int)(log(1.0+(var_a*(((ejdepth-ejgsd)/eidisc))))/dx);
	numx = numg + numd;
	numxtot = numd + numbulk;

	numy1 = (int)(log(1.0+(var_a*(ejgsd/(2.0*eidisc))))/dy);
	depth = ((exp(dy*numdepth))-1.0)/var_a;
	midz = ((exp(numy1*dy))-1.0)/var_a;

	var_b = ((exp(dy*numy1))-1.0)/(topz-midz);  
	numy=numy1*2;
	numytot = numdepth + numy;

	dz_m1 = midz - ((exp(dy*(numy1-1))-1.0)/var_a);
	dz_m2 = topz-((exp(dy*(numy1-1))-1.0)/var_b) - midz;

	cout << "\nChosen Parameters\n" << "numd: " << numd
		<< " numg: " << numg << " numx: \n" << numx << endl;
	cout << "numy: " << numy << " midz: " << midz << " B: "
		<< var_b << " A: " << var_a << endl;
	cout << "numdepth: " << numdepth << " numbulk: " << numbulk << " numxtot: "
		<< numxtot << " numytot: " << numytot << " depth: " << depth << endl;
	cout << "dr_m1: " << dr_m1 << " dz_m1: " << dz_m1 << " dz_m2: " << dz_m2 << endl;
	cout << "dx: " << dx << " dy: " << dy << endl;

	return dt;
} // welcome


void adi_init(double dt)
{
	//
	// Variables
	// i,j              - interger counters

	int i,j;
	double x2k,xk,y2k,yk,zsumdiv2;
	double r1,r2,z1;
	double var_a1,var_a2,var_a3,expc1,expa1,expa2;

	lamx = dt / (dx * dx);
	lamy = lamx;
	maxlam = dt / (dr_m1 * dr_m1);


	var_a1 = (var_a * var_a) / ((1.0 + var_a) * (1.0 + var_a));
	var_a2 = var_a * var_a * (1.0 + var_a) * (1.0 + var_a);
	var_a3 = (var_a + 1.0) * (var_a - 1.0);

	// Implicit Radial Loop Numbers
	posr[0] = 0.0;
	expa1 = 1.0;
	expc1 = exp(dx);

	for (i = 1 ; i <= numd-1 ; i++) 
    {
		expa1 = expa1 * expc1;
		r1 = (var_a1 * expa1) / (1.0 - (1.0 / expa1));
		r2 = var_a1 * (expa1 * expa1);
		x2k = r2;
		xk = r1 + r2;
		cofx[i] = ((-1.0*lamx*x2k) + (xk*lamx*dx/2.0));
		a2[i] = (1.0 + (x2k*2.0*lamx))/cofx[i];
		a3[i] = -1.0*((x2k*lamx) + (xk*lamx*dx/2.0))/cofx[i]; 
		posr[i] = ((1.0 + var_a)*(1.0 - (1.0 / expa1))) / var_a;
    }          

	// One Extra Increment for R=1 Missed Coefficient
	i = numd;
	expa1 = expa1 * expc1;
	cofx[i] = ((-1.0*maxlam) + (maxlam*dr_m1/2.0));
	a2[i] = (1.0 + (2.0*maxlam))/cofx[i];
	a3[i] = -1.0*((maxlam) + (maxlam*dr_m1/2.0))/cofx[i]; 
	posr[numd] = 1.0;

	for (i = numd+1 ; i <= numxtot+1 ; i++) 
    {
		expa1 = expa1 * expc1;
		r1 = (var_a2 / expa1) / (expa1 + var_a3);
		r2 = var_a2 / (expa1 * expa1);
		x2k = r2;
		xk = r1 - r2;
		cofx[i] = ((-1.0*lamx*x2k) + (xk*lamx*dx/2.0));
		a2[i] = (1.0 + (x2k*2.0*lamx))/cofx[i];
		a3[i] = -1.0*((x2k*lamx) + (xk*lamx*dx/2.0))/cofx[i]; 
		posr[i] = (expa1 + var_a3) / (var_a * (var_a + 1.0));
    }    

	// calculate last a_1 point     
	a_1[numxtot] = a2[numxtot];  //  {+a3[numxtot]; 

	// fill up a_1 array working backwards
	for ( i = numxtot - 1 ; i >= 1 ; i--)
    {
		a_1[i] = a2[i] - ( a3[i] / a_1[i+1] );
    }

	// Implicit Axial Loop Numbers 
	posz[0] = 0.0;
	expa1 = exp(numdepth*dy);
	expc1 = exp(-1.0*dy);

	expa2 = exp(numy * dy);

	for (j = 1 ; j <= numdepth-1 ; j++)
	{
		expa1 = expa1 * expc1;
		z1 = var_a * var_a / (expa1 * expa1);
		y2k = z1;
		yk = z1;
		cofy[j] = ((-1.0*y2k*lamy)+(yk*lamy*dy/2.0));
		c2[j] = (1.0 + (2.0*y2k*lamy)) / cofy[j];
		c3[j] = -1.0*((y2k*lamy)+(yk*lamy*dy/2.0)) / cofy[j];
		posz[j] = depth + ((1.0 - expa1) / var_a);     
    }

	// One Extra Increment for Baseline Join
	j = numdepth;
	expa1 = expa1 * expc1;
	cofy[j] = -1.0 * dt / (dz_b1 * dz_b1);
	c2[j] = (1.0 + (2.0*dt/(dz_b1*dz_b1)))/cofy[j];
	c3[j] = 1.0;
	posz[j] = depth;

	for (j = (numdepth+1) ; j <= (numdepth+numy1-1) ; j++)
    {
		expa1 = expa1 * expc1;
		z1 = var_a * var_a * (expa1 * expa1);
		y2k = z1;
		yk = -1.0 * z1;
		cofy[j] = ((-1.0*y2k*lamy)+(yk*lamy*dy/2.0));
		c2[j] = (1.0 + (2.0*y2k*lamy)) / cofy[j];
		c3[j] = -1.0*((y2k*lamy)+(yk*lamy*dy/2.0)) / cofy[j];
		posz[j] = depth + (((1.0/expa1) - 1.0) / var_a);     
    }

	// One Extra Increment for Expansion Grid Join about 1/2 way between tip/sub 
	j = numy1 + numdepth;
	expa1 = expa1 * expc1;
	zsumdiv2 = (dz_m1 + dz_m2)/2.0;
	cofy[j] = -1.0 * dt / (dz_m1 * zsumdiv2);
	c2[j] = (1.0 + (dt/(dz_m2*zsumdiv2)) + (dt/(dz_m1*zsumdiv2)))/cofy[j];
	c3[j] = (-1.0 * dt / (dz_m2 * zsumdiv2))/cofy[j];
	posz[j] = midz + depth;

	expa1 = expa1 * expa2;
	for (j = (numy1+1+numdepth) ; j <= (numy+numdepth) ; j++)
	{
		expa1 = expa1 * expc1;
		z1 = var_b * var_b / (expa1 * expa1);
		y2k = z1;
		yk = z1;
		cofy[j] = ((-1.0*y2k*lamy)+(yk*lamy*dy/2.0));
		c2[j] = (1.0 + (2.0*y2k*lamy)) / cofy[j];
		c3[j] = -1.0*((y2k*lamy)+(yk*lamy*dy/2.0))/cofy[j];
		posz[j] = (depth + ((double)ejgsd/(double)eidisc)) + ((1.0 - expa1) / var_b);
    }

	// calculate last c_1 point     
	c_1[numy+numdepth-1] = c2[numy+numdepth-1]; // +c3[numy+numdepth-1];

	// fill up c_1 array working backwards
	for ( j = numy+numdepth-2 ; j >= 1 ; j--)
    {
		c_1[j] = c2[j] - ( c3[j] / c_1[j+1] );
    }
      
	return;      
} 


void adi_sum(void)
{

	// Variables
	// i,j           - integer counters

	int i,j,int_limit;

	// Firstly Implicit in the RADIAL direction 
	pold = cu1;
	pnew = cu2;

	perp(numxtot+1,0,numytot);
	parral(0,numx+1,numxtot);

	for (j = 1; j<=(numy+numdepth-1); j++)
    {    
		if ( j <= numdepth ) { int_limit = numx+2; } else { int_limit = 1; }         
		for (i = int_limit; i<=numxtot; i++)
        {
			b4[i] = ((-1.0*pold[i][j-1]*cofy[j]) + 
			      (((-1.0*c2[j]*cofy[j])+2.0)*pold[i][j]) +
                  (-1.0*c3[j]*cofy[j]*pold[i][j+1]))/cofx[i];
        }
		b_1[numxtot] = b4[numxtot] - (a3[numxtot]*pnew[numxtot+1][j]);;
		for ( i = numxtot-1 ; i>=int_limit; i-- )
			{ b_1[i] = b4[i] - (a3[i]*b_1[i+1]/a_1[i+1]); }
		pnew[int_limit-1][j] = b_1[int_limit]/(1.0+a_1[int_limit]);
		for (i=int_limit ; i<=numxtot ; i++)
			{ pnew[i][j] = (b_1[i] - pnew[i-1][j])/a_1[i]; }
    }
    // boundary conditions
	electrode(numdepth,0,numd);
	j = numdepth;
	for (i=numd+1; i<=numx ; i++) { pnew[i][j] = pnew[i][j+1]; }
	j = numy+numdepth;
	for (i=0; i<=numxtot ; i++) { pnew[i][j] = 1.0; }//pnew[i][j-1]; }

	// Now Implicit Axially
	// swap over matrices - removes 'update' subroutine
	pold = cu2;
	pnew = cu1;

	perp(numxtot+1,0,numytot);
	parral(0,numx+1,numxtot);  
	for (i=1; i<=numxtot; i++)
    {
		if ( i <= numx+1 ) { int_limit = numdepth+1; } else { int_limit = 1; } 
		for (j=int_limit; j<=numytot-1; j++)
        {
			d4[j] = ((-1.0*cofx[i]*pold[i-1][j]) +
                  (((-1.0*a2[i]*cofx[i])+2.0)*pold[i][j]) +
                  (-1.0*a3[i]*cofx[i]*pold[i+1][j]))/cofy[j];
        }
		d_1[numytot-1] = d4[numytot-1] - c3[numy+numdepth-1];
		for (j=numytot-2 ; j>=int_limit ; j--)
		{
			d_1[j] = d4[j] - (c3[j]*d_1[j+1]/c_1[j+1]);
		}
		if (i <= numd) { pnew[i][int_limit-1] = 0.0; }
		if ((i > numd )&&(i<=numx+1)) 
		{
			pnew[i][int_limit-1] = d_1[int_limit] / (1.0 + c_1[int_limit]);
		}
		// otherwise parral bulk boundary already used   
		for ( j = int_limit ; j <=numytot-1 ; j++ )
		{
			pnew[i][j] = (d_1[j] - pnew[i][j-1])/c_1[j];
		}                
    } // i loop
 
	// filling in boundary 
	pnew[0][numdepth] = 0.0;
	for (j = numdepth+1; j <= numytot-1 ; j++) { pnew[0][j] = pnew[1][j]; }
	for (j = 1; j <= numdepth-1 ; j++) { pnew[numx+1][j] = pnew[numx+2][j]; }
	for (i = 0; i <= numxtot ; i++) { pnew[i][numytot] = 1.0; } // pnew[i][numytot-1]; }
                                              
	return;
}	// adi_sum 

void clearbulk(int a,int b,int c,int d)           
{
	// initialises arrays
	int i,j;

	for (i = a; i <= b; i++)
    {
		for (j = c; j <= d; j++)
		{
			pold[i][j] = cob;
			pnew[i][j] = 999.0;		// warning concentration, should all be overwritten
        }
    }
	return;     
}	// clearbulk
     
void electrode(int a,int b,int c)
{
	// sets electrode concentration (to zero)
	int i,j;

	j = a;
	for (i = b; i<=c; i++)
	{
		pnew[i][j] = 0.0;
    }
	return;
}	// electrode

void perp(int a,int b,int c)
{
	// sets perpendicular boundary to bulk conc
	int i,j;

	i = a;
	for (j = b; j<=c; j++)
    {
		pnew[i][j] = cob;
    }
	return;
}

void parral(int a,int b,int c)
{
	// sets parallel boundary to bulk conc
	int i,j;

	j = a;
	for (i = b; i<=c; i++)
    {
		pnew[i][j] = cob;
    }
	return;
}

double calc_the_current()
{
	// calculates 1 species tip current
	double nsum,dlctip;
	int i;

	nsum = 0.0;
	for (i = 1; i<=numd ; i++) 
	{
		nsum += posr[i]*(posr[i]-posr[i-1])*(pnew[i][numdepth+1]-pnew[i][numdepth]);
	}
	dlctip = M_PI * nsum / (2.0 * dz_b1);

	return dlctip;
}

void map_the_conc(int bl, int br, int lb, int lt, double timepos, int timecount, int test)
{

	// prints out a map of concentrations (range requested)
	// there is a check to see if files are too large

	ofstream mapOUT;
	int i,j,conclimitx,conclimity;
	double expo,expof,j_val;
	
	char concfile[] ={"xmap.smap"};

	concfile[0]=(char)(timecount+65);
	if (test==1) { strcpy(concfile,"FILE.smap"); }
	
	mapOUT.open(concfile);
	mapOUT.setf(ios::fixed, ios::floatfield);
	mapOUT.setf(ios::showpoint);
	mapOUT << setprecision(8) << timepos << endl;
      
	if (test==1) 
	{
		mapOUT << numxtot << "\n" << numytot << "\n" << numd << "\n" << numdepth
			 << "\n" <<  numg << "\n";
	}

	conclimitx=numxtot/100;
	conclimity=numytot/100;
	if (conclimitx==0) {conclimitx=1;}
	if (conclimity==0) {conclimity=1;}

	j=lb;

	// set up exponential factor calc.
	expo=1.0;
	expof=exp(dy);
	for (; j<=lt;)
    {
		i = bl;
		j_val = (expo - 1.0) / var_b;
		for (; i<=(br);)
        {
			mapOUT << posr[i] << "\t" << posz[j] << "\t" << pnew[i][j] << endl;
			i+=conclimitx;
        }
    j+=conclimity;
    }
	mapOUT.close();
	return;
}

void id_block(int a,int b,int c,int d)
{
	// id areas of matrix with idcode (normally negative)
	int i,j;

	for (j = a; j<=b; j++)
    {
		for (i = c; i<=d; i++)
        {
            pnew[i][j] = idcode; 
        }
    }
}                

